#include "create_shard.h"
#include "assignments.h"
#include "default_limits.h"
#include "shard_manager.h"

#include <solomon/libs/cpp/error_or/error_or.h>
#include <solomon/libs/cpp/actors/events/events.h>
#include <solomon/libs/cpp/yasm/constants/interval.h>
#include <solomon/libs/cpp/yasm/constants/project.h>

#include <library/cpp/threading/future/future.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/event_local.h>
#include <library/cpp/actors/core/hfunc.h>

#include <util/system/hostname.h>

using namespace NActors;
using namespace NThreading;
using namespace NSolomon::NDb;
using namespace NSolomon::NDb::NModel;

namespace NSolomon::NIngestor {
namespace {
    NModel::TShardConfig DefaultShardConfig() {
        NModel::TShardConfig conf;
        conf.MaxMemMetrics = MAX_MEM_METRICS;
        conf.MaxResponseSizeBytes = MAX_RESPONSE_SIZE_BYTES;
        conf.MaxFileMetrics = MAX_FILE_METRICS;
        conf.MaxMetricsPerUrl = MAX_METRICS_PER_URL;
        conf.State = EShardState::Active;
        return conf;
    }

    template <typename TModel, size_t Id>
    struct TEvCallback: TEventLocal<TEvCallback<TModel, Id>, Id> {
        TEvCallback(TModel m)
            : Result(Result.FromValue(std::move(m)))
        {
        }

        TEvCallback(TGenericError err)
            : Result(std::move(err))
        {
        }

        // TODO(msherbakov): use more descriptive error
        TErrorOr<TModel, TGenericError> Result;
    };


    class TShardCreator: public TActorBootstrapped<TShardCreator>, private TPrivateEvents {
        enum {
            EvCallbackProject = TEvents::ES_PRIVATE,
            EvCallbackService,
            EvCallbackCluster,
            EvCallbackShard,
        };

        using TEvCallbackProject = TEvCallback<TProjectConfig, EvCallbackProject>;
        using TEvCallbackService = TEvCallback<TServiceConfig, EvCallbackService>;
        using TEvCallbackCluster = TEvCallback<TClusterConfig, EvCallbackCluster>;
        using TEvCallbackShard = TEvCallback<NModel::TShardConfig, EvCallbackShard>;

    public:
        TShardCreator(TActorId clientId, ui64 cookie, TShardKey key, TShardCreationContext ctx)
            : ClientId_(clientId)
            , Cookie_(cookie)
            , ShardKey_(std::move(key))
            , ShardId_(MakeShardId(ShardKey_))
            , Ctx_(std::move(ctx))
        {
            Y_ENSURE(Ctx_.IdValidator);
            Y_ENSURE(Ctx_.NumIdGenerator);
        }

        bool ValidateRequest() {
            auto e = Ctx_.IdValidator->ValidateShardId(ShardId_);
            if (e.Fail()) {
                CompleteError(e);
                return false;
            }

            e = Ctx_.IdValidator->ValidateId(ShardKey_.ClusterName);
            if (e.Fail()) {
                CompleteError(e);
                return false;
            }

            e = Ctx_.IdValidator->ValidateId(ShardKey_.ServiceName);
            if (e.Fail()) {
                CompleteError(e);
                return false;
            }

            return true;
        }

        void Bootstrap() {
            if (!ValidateRequest()) {
                return;
            }

            Become(&TThis::StateEnsureProject);
            SelfId_ = SelfId();
            ActorSystem_ = TActorContext::ActorSystem();
            FindOrCreateProject();
        }

        STFUNC(StateEnsureProject) {
            Y_UNUSED(ctx);
            switch (ev->GetTypeRewrite()) {
                hFunc(TEvCallbackProject, HandleProject);
            }
        }

        STFUNC(StateEnsureServiceCluster) {
            Y_UNUSED(ctx);
            switch (ev->GetTypeRewrite()) {
                hFunc(TEvCallbackCluster, HandleCluster);
                hFunc(TEvCallbackService, HandleService);
            }
        }

        STFUNC(StateEnsureShard) {
            Y_UNUSED(ctx);
            switch (ev->GetTypeRewrite()) {
                hFunc(TEvCallbackShard, HandleShard);
            }
        }

        void HandleShard(const TEvCallbackShard::TPtr& ev) {
            auto& r = ev->Get()->Result;
            if (r.Fail()) {
                CompleteError(r);
                return;
            }

            CompleteSuccess(ev->Get()->Result.Value());
        }

        template <typename TEv, typename TModel>
        void HandleClusterService(typename TEv::TPtr ev, TModel& model) {
            --InFlight_;
            Y_VERIFY_DEBUG(InFlight_ >= 0);
            if (ev->Get()->Result.Fail()) {
                CompleteError(ev->Get()->Result);
                return;
            }

            model = ev->Get()->Result.Extract();

            if (InFlight_ == 0) {
                CreateShard();
                Become(&TThis::StateEnsureShard);
            }
        }

        void HandleService(const TEvCallbackService::TPtr& ev) {
            HandleClusterService<TEvCallbackService>(std::move(ev), ServiceConfig_);
        }

        void HandleCluster(const TEvCallbackCluster::TPtr& ev) {
            HandleClusterService<TEvCallbackCluster>(std::move(ev), ClusterConfig_);
        }

        void HandleProject(const TEvCallbackProject::TPtr& ev) {
            if (ev->Get()->Result.Fail()) {
                CompleteError(ev->Get()->Result);
                return;
            }

            ProjectConfig_ = ev->Get()->Result.Extract();

            // TODO(msherbakov): add ACL check

            FindOrCreateService();
            FindOrCreateCluster();
            InFlight_ = 2;

            Become(&TThis::StateEnsureServiceCluster);
        }

        template <typename T>
        void CompleteError(const TErrorOr<T, TGenericError>& r) {
            CompleteError(r.Error().MessageString());
        }

        void CompleteError(TString message) {
            Send(ClientId_, new TShardManagerEvents::TEvCreateShardResult{std::move(message)}, 0, Cookie_);
            PassAway();
        }

        void CompleteSuccess(const NModel::TShardConfig& conf) {
            Send(ClientId_, new TShardManagerEvents::TEvCreateShardResult{TCreateShardResult{
                .LeaderHost = FQDNHostName(),
                .AssignedToHost = Ctx_.Assigner->Assign(conf.Id).Fqdn,
                .ShardId = conf.Id,
                .NumId = conf.NumId,
                // XXX
            }}, 0, Cookie_);
            PassAway();
        }

    public:
        template <typename TEv, typename TModel>
        auto MakeInsertHandler(TModel model) {
            return [this, model] (auto f) {
                try {
                    f.TryRethrow();
                    ActorSystem_->Send(SelfId_, new TEv{std::move(model)});
                } catch (...) {
                    ActorSystem_->Send(SelfId_, new TEv{CurrentExceptionMessage()});
                }
            };
        }

        template <typename TEv, typename TConfFactory, typename TDao>
        auto MakeGetHandler(TDao dao, TConfFactory&& factory) {
            return [this, dao{std::move(dao)}, factory{std::move(factory)}] (auto f) {
                auto val = f.ExtractValue();
                if (val) {
                    ActorSystem_->Send(SelfId_, new TEv{*val});
                } else {
                    auto conf = factory();
                    dao->Insert(conf).Subscribe(MakeInsertHandler<TEv>(conf));
                }
            };
        }

        void FindOrCreateCluster() {
            Ctx_.ClusterDao->GetByName(ShardKey_.ClusterName, ShardKey_.ProjectId).Subscribe(
                MakeGetHandler<TEvCallbackCluster>(Ctx_.ClusterDao, [this] { return ClusterConfig(); })
            );
        }

        void FindOrCreateService() {
            Ctx_.ServiceDao->GetByName(ShardKey_.ServiceName, ShardKey_.ProjectId).Subscribe(
                MakeGetHandler<TEvCallbackService>(Ctx_.ServiceDao, [this] { return ServiceConfig(); })
            );
        }

        void CreateShard() {
            auto conf = ShardConfig();
            // TODO(msherbakov): check whether num id is unique
            // and retry with different num id if required
            Ctx_.ShardDao->Insert(conf).Subscribe(MakeInsertHandler<TEvCallbackShard>(conf));
        }

        void FindOrCreateProject() {
            Ctx_.ProjectDao->GetById(ShardKey_.ProjectId).Subscribe([this] (auto f) {
                auto proj = f.ExtractValue();
                if (proj) {
                    ActorSystem_->Send(SelfId_, new TEvCallbackProject{*proj});
                } else if (Ctx_.Opts.CanCreateProject()) {
                    auto conf = ProjectConfig();
                    Ctx_.ProjectDao->Insert(conf).Subscribe(MakeInsertHandler<TEvCallbackProject>(conf));
                } else {
                    ActorSystem_->Send(SelfId_, new TEvCallbackProject{TGenericError(TStringBuilder()
                        << "Project " << ShardKey_.ProjectId
                        << " does not exist and cannot be created"
                    )});
                }
            });
        }

    private:
        TProjectConfig ProjectConfig() const {
            auto conf = ProjectConfig_;
            conf.Id = ShardKey_.ProjectId;
            conf.Name = conf.Id;
            conf.Owner = Ctx_.User;
            conf.AbcService = "solomon";

            conf.CreatedAt = TInstant::Now();
            conf.CreatedBy = Ctx_.User;
            conf.UpdatedAt = TInstant::Now();
            conf.UpdatedBy = Ctx_.User;

            return conf;
        }

        TServiceConfig ServiceConfig() const {
            TServiceConfig conf;
            conf.Id = ShardKey_.ServiceName;
            conf.Name = ShardKey_.ServiceName;
            conf.ProjectId = ShardKey_.ProjectId;

            if (NYasm::IsYasmProject(conf.ProjectId)) {
                conf.Interval = NYasm::YASM_INTERVAL;
                conf.GridSec = static_cast<i32>(conf.Interval.Seconds());
            }

            conf.CreatedAt = TInstant::Now();
            conf.CreatedBy = Ctx_.User;
            conf.UpdatedAt = TInstant::Now();
            conf.UpdatedBy = Ctx_.User;

            return conf;
        }

        TClusterConfig ClusterConfig() const {
            TClusterConfig conf;
            conf.Id = ShardKey_.ClusterName;
            conf.Name = ShardKey_.ClusterName;
            conf.ProjectId = ShardKey_.ProjectId;

            conf.CreatedAt = TInstant::Now();
            conf.CreatedBy = Ctx_.User;
            conf.UpdatedAt = TInstant::Now();
            conf.UpdatedBy = Ctx_.User;

            return conf;
        }

        NModel::TShardConfig ShardConfig() const {
            auto conf = DefaultShardConfig();
            conf.Id = ShardId_;
            conf.NumId = Ctx_.NumIdGenerator->Generate();
            conf.ProjectId = ShardKey_.ProjectId;

            conf.ServiceId = ServiceConfig_.Id;
            conf.ServiceName = ServiceConfig_.Name;
            conf.ClusterId = ClusterConfig_.Id;
            conf.ClusterName = ClusterConfig_.Name;

            conf.CreatedAt = TInstant::Now();
            conf.CreatedBy = Ctx_.User;
            conf.UpdatedAt = TInstant::Now();
            conf.UpdatedBy = Ctx_.User;

            return conf;
        }

    private:
        const TActorId ClientId_;
        const ui64 Cookie_;
        const TShardKey ShardKey_;
        const TString ShardId_;
        int InFlight_{0};
        TShardCreationContext Ctx_;
        TActorId SelfId_;
        TActorSystem* ActorSystem_;

        TServiceConfig ServiceConfig_;
        TClusterConfig ClusterConfig_;
        TProjectConfig ProjectConfig_;
    };
} // namespace

IActor* CreateShard(TActorId clientId, ui64 cookie, TShardKey key, TShardCreationContext ctx) {
    return new TShardCreator{
        clientId,
        cookie,
        std::move(key),
        std::move(ctx)
    };
}

} // namespace NSolomon::NIngestor
