#include "make_shard_configs.h"

#include <solomon/services/ingestor/lib/holders/local_shards_holder.h>
#include <solomon/services/ingestor/lib/holders/global_shards_holder.h>
#include <solomon/services/ingestor/lib/shard_config/shard_config.h>
#include <solomon/services/ingestor/lib/shard_manager/shard_manager.h>

#include <solomon/libs/cpp/logging/logging.h>
#include <solomon/libs/cpp/conf_db/puller/puller.h>
#include <solomon/libs/cpp/slices/operations.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/actor.h>
#include <library/cpp/actors/core/actorid.h>
#include <library/cpp/actors/core/hfunc.h>
#include <library/cpp/actors/core/log.h>
#include <library/cpp/monlib/metrics/metric_registry.h>

#include <util/generic/hash.h>
#include <util/generic/list.h>
#include <util/generic/ptr.h>
#include <util/generic/yexception.h>

using namespace NActors;
using namespace NSolomon::NSlicer::NApi;
using namespace NSolomon;

namespace NSolomon::NIngestor {
namespace {

struct TProjectIdAndServiceId {
    TString ProjectId;
    TString ServiceId;

    bool operator==(const TProjectIdAndServiceId&) const = default;
};

struct TProjectIdAndClusterId {
    TString ProjectId;
    TString ClusterId;

    bool operator==(const TProjectIdAndClusterId&) const = default;
};

struct TProjectIdAndServiceIdHasher {
    ui64 operator()(const TProjectIdAndServiceId& key) const {
        return MultiHash(key.ProjectId, key.ServiceId);
    }
};

struct TProjectIdAndClusterIdHasher {
    ui64 operator()(const TProjectIdAndClusterId& key) const {
        return MultiHash(key.ProjectId, key.ClusterId);
    }
};

TShardConfig ShardConfigFromRawModels(
        const NDb::NModel::TShardConfig& rawShardConfig,
        const NDb::NModel::TServiceConfig& rawServiceConfig,
        const NDb::NModel::TClusterConfig& rawClusterConfig)
{
    TShardConfig shardConfig;
    shardConfig.Key = TShardKey{
            rawShardConfig.ProjectId,
            rawShardConfig.ClusterName,
            rawShardConfig.ServiceName
    };
    shardConfig.ClusterId = rawClusterConfig.Id;
    shardConfig.ServiceId = rawShardConfig.ServiceId;
    shardConfig.ShardStrId = rawShardConfig.Id;
    shardConfig.ShardNumId = rawShardConfig.NumId;
    shardConfig.IsShardMemOnly = rawServiceConfig.MetricConf.RawDataMemOnly;
    shardConfig.MetricNameLabel = rawServiceConfig.MetricNameLabel;
    shardConfig.AggrRules = rawServiceConfig.MetricConf.AggrRules;
    shardConfig.Interval = rawServiceConfig.Interval;

    shardConfig.CalculateSizeBytes();

    return shardConfig;
}

} // namespace

absl::flat_hash_map<TShardId, TShardConfig> MakeShardConfigs(TConfigs&& configs) {
    THashMap<TProjectIdAndServiceId, NDb::NModel::TServiceConfig, TProjectIdAndServiceIdHasher> serviceByIds;
    for (const auto& serviceConfig: configs.Services) {
        TProjectIdAndServiceId key = {
            .ProjectId = serviceConfig.ProjectId,
            .ServiceId = serviceConfig.Id,
        };
        serviceByIds[key] = serviceConfig;
    }

    THashMap<TProjectIdAndClusterId, NDb::NModel::TClusterConfig, TProjectIdAndClusterIdHasher> clusterByIds;
    for (const auto& clusterConfig: configs.Clusters) {
        TProjectIdAndClusterId key = {
            .ProjectId = clusterConfig.ProjectId,
            .ClusterId = clusterConfig.Id,
        };
        clusterByIds[key] = clusterConfig;
    }

    THashSet<TString> projectIds;
    for (const auto& projectConfig: configs.Projects) {
        projectIds.emplace(projectConfig.Id);
    }

    absl::flat_hash_map<TShardId, TShardConfig> shards;
    for (const auto& shardConf: configs.Shards) {
        auto serviceConfIter = serviceByIds.find(TProjectIdAndServiceId{
            .ProjectId = shardConf.ProjectId,
            .ServiceId = shardConf.ServiceId,
        });
        if (serviceConfIter == serviceByIds.end()) {
            MON_WARN(ShardUpdater, "Unable to find service information for shard: \"" << shardConf.Id << "\""
                    << ". service id: \"" << shardConf.ServiceId << "\""
                    << ", project id: \"" << shardConf.ProjectId << "\"");
            continue;
        }
        const auto& serviceConf = serviceConfIter->second;

        auto clusterConfIter = clusterByIds.find(TProjectIdAndClusterId{
            .ProjectId = shardConf.ProjectId,
            .ClusterId = shardConf.ClusterId,
        });
        if (clusterConfIter == clusterByIds.end()) {
            MON_WARN(ShardUpdater, "Unable to find cluster information for shard: \"" << shardConf.Id << "\""
                    << ". cluster id: \"" << shardConf.ClusterId << "\""
                    << ", project id: \"" << shardConf.ProjectId << "\"");
            continue;
        }
        const auto& clusterConf = clusterConfIter->second;

        if (!projectIds.contains(shardConf.ProjectId)) {
            MON_WARN(ShardUpdater, "Unable to find project information for shard: \"" << shardConf.Id << "\""
                    << ". project id: \"" << shardConf.ProjectId << "\"");
            continue;
        }

        shards.emplace(shardConf.NumId, ShardConfigFromRawModels(shardConf, serviceConf, clusterConf));
    }

    return shards;
}

} // namespace NSolomon::NIngestor
