#include "shard_manager.h"
#include "create_shard_opts.h"
#include "create_shard.h"
#include "assignments.h"
#include "shard_config_page.h"

#include <solomon/services/ingestor/config/ingestor_config.pb.h>
#include <solomon/services/ingestor/lib/coremon_sync/coremon_sync.h>
#include <solomon/services/ingestor/lib/holders/global_shards_holder.h>
#include <solomon/services/ingestor/lib/holders/local_shards_holder.h>
#include <solomon/services/ingestor/lib/shard_actor/shard_actor.h>

#include <solomon/libs/cpp/cluster_map/cluster.h>
#include <solomon/libs/cpp/config/units.h>
#include <solomon/libs/cpp/conf_db/puller/puller.h>
#include <solomon/libs/cpp/load_info_service/collector.h>
#include <solomon/libs/cpp/load_info_service/host_info.h>
#include <solomon/libs/cpp/load_info_service/load_info_service.h>
#include <solomon/libs/cpp/logging/logging.h>
#include <solomon/libs/cpp/clients/slicer/api_types.h>
#include <solomon/libs/cpp/clients/slicer/slicelet_listener.h>
#include <solomon/libs/cpp/slices/operations.h>
#include <solomon/libs/cpp/selfmon/selfmon.h>

#include <library/cpp/actors/core/actor.h>
#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/actorsystem.h>
#include <library/cpp/actors/core/defs.h>
#include <library/cpp/actors/core/hfunc.h>
#include <library/cpp/actors/core/log.h>

#include <util/datetime/base.h>
#include <util/generic/hash.h>
#include <util/generic/ptr.h>
#include <util/generic/scope.h>
#include <util/random/random.h>
#include <util/string/split.h>
#include <util/system/info.h>

#include <thread>

namespace NSolomon::NIngestor {
namespace {

using namespace NActors;

using namespace NSolomon::NLoadInfo;
using namespace NSolomon::NSlicer::NApi;
using namespace yandex::monitoring::slicer;
using yandex::monitoring::ingestor::IngestorConfig;
using yandex::monitoring::load_info::LoadInfoResponse;
using yandex::monitoring::load_info::ShardState;

constexpr static const TDuration RELOAD_CONFIGS_BASE_DURATION = TDuration::Minutes(3);
constexpr static const TDuration MEM_METERING_DEFAULT_INTERVAL = TDuration::Seconds(5);

    class TNumIdGenerator: public INumIdGenerator {
    public:
        TNumIdGenerator(const absl::flat_hash_map<TShardId, TShardConfig>& shards)
            : Shards(shards)
        {
        }

        TShardId Generate() const override {
            TShardId numId{0};
            do {
                numId = RandomNumber<TShardId>();
            } while (Shards.contains(numId));

            return numId;
        }

        const absl::flat_hash_map<TShardId, TShardConfig>& Shards;
    };

    class TShardManager: public TActorBootstrapped<TShardManager>, private TPrivateEvents {
    private:
        enum {
            EvCollectMemoryMetering = SpaceBegin,
            End
        };

        static_assert(End < SpaceEnd, "too many event types");

        struct TEvCollectMemoryMetering: TEventLocal<TEvCollectMemoryMetering, EvCollectMemoryMetering> {
        };

    public:
        TShardManager(
                const IngestorConfig& config,
                ui32 executorPool,
                NSolomon::NDb::IShardConfigDaoPtr shardDao,
                NSolomon::NDb::IClusterConfigDaoPtr clusterDao,
                NSolomon::NDb::IServiceConfigDaoPtr serviceDao,
                NSolomon::NDb::IProjectConfigDaoPtr projectDao,
                NMonitoring::TMetricRegistry& registry,
                std::shared_ptr<TShardMeteringMetricsRepository> meteringMetricsRepo,
                IThreadPool* processorsExecutor,
                const IClusterMap& clusterMap,
                bool isStandalone,
                std::shared_ptr<IClusterRpc<NMemStore::IMemStoreRpc>> memStoreRpc,
                TActorId memStoreClusterWatcher,
                IRequestTracerPtr tracer,
                TActorId sliceletId,
                TString yasmPrefix)
            : ExecutorPoolForShardActors_(executorPool)
            , LastShardAssignment_(new TLocalShardsHolder)
            , CurrentVersion_(0)
            , ShardDao_(std::move(shardDao))
            , ClusterDao_(std::move(clusterDao))
            , ServiceDao_(std::move(serviceDao))
            , ProjectDao_(std::move(projectDao))
            , Registry_(registry)
            , MeteringMetricsRepo_{std::move(meteringMetricsRepo)}
            , ProcessorsExecutor_(processorsExecutor)
            , IdValidator_(CreateIdValidator())
            , NumIdGenerator_(ShardConfigs_)
            , ClusterMap_(clusterMap)
            , IsStandalone_(isStandalone)
            , Assigner_(ClusterMap_)
            , MemStoreRpc_(std::move(memStoreRpc))
            , MemStoreClusterWatcher_(memStoreClusterWatcher)
            , Tracer_(std::move(tracer))
            , SliceletId_(sliceletId)
            , YasmPrefix_(std::move(yasmPrefix))
        {
            // XXX(msherbakov): this must be read from the app config
            CreateShardOpts_
                .SetCanCreateProject(true);

            MemoryMeteringCollectionInterval_ = FromProtoTime(
                    config.memory_metering_collection_interval(),
                    MEM_METERING_DEFAULT_INTERVAL);
        }

        void Bootstrap() {
            NSelfMon::RegisterPage(
                *TActorContext::ActorSystem(),
                "/shards", "Shards",
                SelfId());

            Become(&TThis::StateWork);

            auto configsPullerActor = Register(CreateConfigsPuller({
                    .ProjectDao = ProjectDao_,
                    .ClusterDao = ClusterDao_,
                    .ServiceDao = ServiceDao_,
                    .ShardDao = ShardDao_,
                    .UpdateInterval = RELOAD_CONFIGS_BASE_DURATION,
                    .MaxUpdateInterval = 4 * RELOAD_CONFIGS_BASE_DURATION,
                    .Registry = Registry_,
            }).release());

            Send(configsPullerActor, new TConfigsPullerEvents::TSubscribe);

            if (SliceletId_) {
                Send(SliceletId_, new NSolomon::NSlicerClient::TSliceletListenerEvents::TEvSubscribe);
            }

            if (MeteringMetricsRepo_) {
                Schedule(MemoryMeteringCollectionInterval_, new TEvCollectMemoryMetering);
            }

            MON_DEBUG(ShardManager, "Start shard updater");
        }

        STATEFN(StateWork) {
            switch (ev->GetTypeRewrite()) {
                hFunc(TConfigsPullerEvents::TConfigsResponse, OnUpdateShards);
                hFunc(TShardManagerEvents::TEvUpdateShardAssignments, OnUpdateLocalShardsByCoremon);
                hFunc(NSlicerClient::TSliceletListenerEvents::TSlicesUpdate, OnUpdateLocalShardsBySlicer);
                hFunc(TShardManagerEvents::TEvGetAssignedShards, OnGetAssignedShards);
                hFunc(TShardManagerEvents::TEvGetShardAssignments, OnGetShardAssignments);
                hFunc(TShardManagerEvents::TEvCreateShard, OnCreateShard);
                hFunc(TShardManagerEvents::TFindShard, OnFindShard);
                hFunc(TLoadInfoEvents::TGetLoadInfo, OnGetLoadInfo);
                hFunc(NSelfMon::TEvPageDataReq, OnSelfMon)
                cFunc(TEvents::TSystem::Poison, OnPoison);
                sFunc(TEvCollectMemoryMetering, OnCollectMemoryMetering);
            }
        }

    private:
        void OnPoison() {
            if (SliceletId_) {
                Send(SliceletId_, new NSolomon::NSlicerClient::TSliceletListenerEvents::TEvUnsubscribe);
            }
            PassAway();
        }

        TShardCreationContext MakeShardCreationContext() const {
            TShardCreationContext ctx;
            ctx.ClusterDao = ClusterDao_;
            ctx.ProjectDao = ProjectDao_;
            ctx.ServiceDao = ServiceDao_;
            ctx.ShardDao = ShardDao_;

            ctx.IdValidator = IdValidator_.Get();
            ctx.NumIdGenerator = &NumIdGenerator_;
            ctx.Assigner = &Assigner_;
            ctx.Opts = CreateShardOpts_;

            // XXX
            ctx.User = "robot-solomon";

            return ctx;
        }

        void OnCreateShard(const TShardManagerEvents::TEvCreateShard::TPtr& evPtr) {
            auto&& ev = *evPtr->Get();
            MON_DEBUG(ShardManager, "Create shard: " << ev.ShardKey);

            // TODO(msherbakov): add cache for in-flight requests
            RegisterWithSameMailbox(CreateShard(
                evPtr->Sender,
                evPtr->Cookie,
                std::move(ev.ShardKey),
                MakeShardCreationContext())
            );
        }

        void OnFindShard(const TShardManagerEvents::TFindShard::TPtr& evPtr) {
            TActorId shardActor;
            if (auto it = ShardMap_.find(evPtr->Get()->NumId); it != ShardMap_.end()) {
                shardActor = it->second;
            }
            Send(evPtr->Sender, new TShardManagerEvents::TFindShardResult{shardActor}, 0, evPtr->Cookie);
        }

        void OnGetAssignedShards(const TShardManagerEvents::TEvGetAssignedShards::TPtr& ev) {
            TVector<TNumId> numIds(::Reserve(ShardMap_.size()));
            for (auto [numId, _]: ShardMap_) {
                numIds.emplace_back(numId);
            }

            Send(ev->Sender, new TShardManagerEvents::TEvGetAssignedShardsResult{
                std::move(numIds),
            }, 0, ev->Cookie);
        }

        // deprecated api
        void OnGetShardAssignments(const TShardManagerEvents::TEvGetShardAssignments::TPtr& evPtr) {
            auto resp = MakeHolder<TShardManagerEvents::TEvGetShardAssignmentsResult>(Assignments_, FQDNHostName());
            Send(evPtr->Sender, resp.Release());
        }

        void RemoveShard(TShardId numId, const TString& shardStrId) {
            if (auto it = ShardMap_.find(numId); it != ShardMap_.end()) {
                Send(it->second, new TEvents::TEvPoison);

                ShardMap_.erase(it);
                StrToNumId_.erase(shardStrId);
            }
        }

        void ProcessShard(TShardId numId) {
            if (ShardMap_.contains(numId)) {
                return;
            }

            const auto& conf = ShardConfigs_[numId];
            NActors::IActor* shardActorPtr = nullptr;

            try {
                shardActorPtr = CreateShardActor(
                        conf,
                        ProcessorsExecutor_,
                        MemStoreRpc_,
                        MemStoreClusterWatcher_,
                        MeteringMetricsRepo_,
                        YasmPrefix_,
                        &Registry_);
            } catch (...) {
                MON_ERROR(ShardManager,
                          "Exception during shard actor initialization (id = "
                                  << conf.ShardStrId << "):" << CurrentExceptionMessage());
                return;
            }

            TActorId shardActorId = Register(
                    shardActorPtr,
                    TMailboxType::HTSwap,
                    ExecutorPoolForShardActors_
            );
            ShardMap_[numId] = shardActorId;
            StrToNumId_[conf.ShardStrId] = conf.ShardNumId;
        }

        /**
         * Stores a new assignment. Actual shards processing will be started later in OnUpdateShards
         */
        void OnUpdateLocalShardsByCoremon(const TShardManagerEvents::TEvUpdateShardAssignments::TPtr& request) {
            MON_DEBUG(ShardManager, "Got fresh shard assignment");
            LastShardAssignment_ = request->Get()->ShardAssignments;

            MatchShardConfigsWithAssignments();
        }

        void MatchShardConfigsWithAssignments() {
            if (!LastShardAssignment_ || LastShardAssignment_->LocalShards.empty() || ShardConfigs_.empty()) {
                return;
            }

            for (const auto& [numId, _]: ShardConfigs_) {
                if (!LastShardAssignment_->LocalShards.contains(numId)) {
                    auto strId = ShardConfigs_[numId].ShardStrId;
                    RemoveShard(numId, strId);
                } else {
                    ProcessShard(numId);
                }
            }
        }

        /**
         * Stores a new assignment. Actual shards processing will be started later in OnUpdateShards
         */
        void OnUpdateLocalShardsBySlicer(const NSlicerClient::TSliceletListenerEvents::TSlicesUpdate::TPtr& ev) {
            auto& update = ev->Get()->Update;
            LogSlices(update);

            // TODO(ivanzhukov): later, Slicer will send genuine assigned and unassigned silces. For now, it just
            //  rewrites assignments

            if (!LastLocalSlices_) {
                LastLocalSlices_ = std::make_shared<TSlices>();
            }

            LastLocalSlices_->clear();

            for (int i = 0; i != update.assigned_starts_size(); ++i) {
                LastLocalSlices_->emplace(update.assigned_starts(i), update.assigned_ends(i));
            }

            MatchShardConfigsWithSlices();
            MON_DEBUG(ShardManager, "Shards online: " << ShardMap_.size());
        }

        void LogSlices(const GetSlicesByHostResponse& update) {
            auto* logSettings = TActorContext::ActorSystem()->LoggerSettings();

            if (logSettings->Satisfies(NActors::NLog::PRI_TRACE, ShardManager)) {
                TString slicesStr = "assigned:";

                for (int i = 0; i != update.assigned_starts_size(); ++i) {
                    slicesStr += TStringBuilder()
                            << " [" << update.assigned_starts(i) << "; " << update.assigned_ends(i) << "]";
                }

                slicesStr += "\n";
                slicesStr += "unassigned:";

                for (int i = 0; i != update.unassigned_starts_size(); ++i) {
                    slicesStr += TStringBuilder()
                            << " [" << update.unassigned_starts(i) << "; " << update.unassigned_ends(i) << "]";
                }

                MON_TRACE(ShardManager, "got update from the slicelet listener. Slices: \n" << slicesStr);
            }
        }

        /**
         * Performs actions on shards based on:
         *   - previous shard assignment
         *   - last received shard assignment
         *   - configs stored in a global DB
         */
        void OnUpdateShards(TConfigsPullerEvents::TConfigsResponse::TPtr& evPtr) {
            MON_TRACE(ShardManager, "Got data from shard puller");

            decltype(ShardConfigs_) oldShardConfigs = std::move(ShardConfigs_);
            ShardConfigs_ = MakeShardConfigs(std::move(evPtr->Get()->Configs));

            struct TRemoveInfo {
                TShardId ShardNumId;
                TString ShardStrId;

                TRemoveInfo(TShardId numId, TString strId)
                    : ShardNumId{numId}
                    , ShardStrId{std::move(strId)}
                {
                }
            };

            TVector<TRemoveInfo> needToRemove;
            TVector<TShardId> needToUpdate;

            for (const auto& [numId, oldConf]: oldShardConfigs) {
                if (auto it = ShardConfigs_.find(numId); it == ShardConfigs_.end()) {
                    needToRemove.emplace_back(numId, oldConf.ShardStrId);
                } else {
                    auto& newConf = it->second;

                    if (oldConf.AggrRules != newConf.AggrRules ||
                        oldConf.IsShardMemOnly != newConf.IsShardMemOnly ||
                        oldConf.MetricNameLabel != newConf.MetricNameLabel)
                    {
                        needToUpdate.emplace_back(newConf.ShardNumId);
                    }
                }
            }

            for (const auto& [numId, _]: ShardConfigs_) {
                if (!oldShardConfigs.contains(numId)) {
                    needToUpdate.emplace_back(numId);
                }
            }

            for (const auto& shardId: needToUpdate) {
                const auto& conf = ShardConfigs_[shardId];

                if (!conf.ShardStrId.StartsWith(YasmPrefix_)) {
                    continue;
                }

                if (auto it = ShardMap_.find(shardId); it != ShardMap_.end()) {
                    Send(it->second, new TShardActorEvents::TUpdateConfig(conf));
                } // else: the shard is new -- we will create it later if it matches with slices/assignments
            }

            for (const auto& info: needToRemove) {
                RemoveShard(info.ShardNumId, info.ShardStrId);
            }

            // TODO(msherbakov): remove later
            if (IsStandalone_) {
                MakeAssignments();
            }

            if (LastLocalSlices_) {
                MatchShardConfigsWithSlices();
            } else if (LastShardAssignment_) {
                MatchShardConfigsWithAssignments();
            }

            MON_DEBUG(ShardManager, "Shards online: " << ShardMap_.size());
        }

        void OnSelfMon(NSelfMon::TEvPageDataReq::TPtr& ev) {
            if (auto idStr = ev->Get()->Param("id")) {
                TNumId id;
                if (TryFromString(idStr, id)) {
                    if (auto it = ShardConfigs_.find(id); it != ShardConfigs_.end()) {
                        yandex::monitoring::selfmon::Page page;
                        Send(ev->Sender, MakeHolder<NSelfMon::TEvPageDataResp>(ShardConfigPage(it->second)));
                        return;
                    } else {
                        yandex::monitoring::selfmon::Page page;
                        page.mutable_component()->mutable_value()->set_string(TString{"Unable to find shard "} + idStr);
                        Send(ev->Sender, MakeHolder<NSelfMon::TEvPageDataResp>(std::move(page)));
                        return;
                    }
                }
            }

            yandex::monitoring::selfmon::Page page;
            auto* grid = page.mutable_grid();
            if (auto* r = grid->add_rows()) {
                auto* obj = r->add_columns()->mutable_component()->mutable_object();
                if (auto* f = obj->add_fields()) {
                    f->set_name("CurrentVersion");
                    f->mutable_value()->set_uint64(CurrentVersion_);
                }
                if (auto* f = obj->add_fields()) {
                    f->set_name("Shards");
                    f->mutable_value()->set_uint64(ShardMap_.size());
                }
                if (auto* f = obj->add_fields()) {
                    f->set_name("IsStandalone");
                    f->mutable_value()->set_boolean(IsStandalone_);
                }
                if (auto* f = obj->add_fields()) {
                    f->set_name("YasmPrefix");
                    f->mutable_value()->set_string(YasmPrefix_);
                }

            }
            if (auto* r = grid->add_rows()) {
                auto* table = r->add_columns()->mutable_component()->mutable_table();
                table->set_numbered(true);

                auto* numIdColumn = table->add_columns();
                numIdColumn->set_title("NumId");
                auto* numIdValues = numIdColumn->mutable_reference();

                auto* strIdColumn = table->add_columns();
                strIdColumn->set_title("StrId");
                auto* strIdValues = strIdColumn->mutable_string();

                auto* projectColumn = table->add_columns();
                projectColumn->set_title("Project");
                auto* projectValues = projectColumn->mutable_string();

                auto* clusterColumn = table->add_columns();
                clusterColumn->set_title("Cluster");
                auto* clusterValues = clusterColumn->mutable_string();

                auto* serviceColumn = table->add_columns();
                serviceColumn->set_title("Service");
                auto* serviceValues = serviceColumn->mutable_string();

                for (const auto& [id, _]: ShardMap_) {
                    auto idStr = ToString(id);
                    if (auto* ref = numIdValues->add_values()) {
                        ref->set_title(idStr);
                        ref->set_page("/shards");
                        ref->set_args("id=" + idStr);
                    }

                    const auto& shardConfig = ShardConfigs_[id];
                    strIdValues->add_values(shardConfig.ShardStrId);
                    projectValues->add_values(shardConfig.Key.ProjectId);
                    clusterValues->add_values(shardConfig.Key.ClusterName);
                    serviceValues->add_values(shardConfig.Key.ServiceName);
                }
            }

            Send(ev->Sender, MakeHolder<NSelfMon::TEvPageDataResp>(std::move(page)));
        }

        void MatchShardConfigsWithSlices() {
            if (!LastLocalSlices_ || LastLocalSlices_->empty() || ShardConfigs_.empty()) {
                return;
            }

            TVector<TShardId> allNumIds(::Reserve(ShardConfigs_.size()));
            for (const auto& [numId, _]: ShardConfigs_) {
                allNumIds.emplace_back(numId);
            }
            Sort(allNumIds);

            auto matchInfo = MatchSlicesWithNumIds(*LastLocalSlices_, allNumIds);

            for (auto numId: matchInfo.NotMatched) {
                auto strId = ShardConfigs_[numId].ShardStrId;
                RemoveShard(numId, strId);
            }

            for (auto numId: matchInfo.Matched) {
                ProcessShard(numId);
            }
        }

        void OnCollectMemoryMetering() {
            Schedule(MemoryMeteringCollectionInterval_, new TEvCollectMemoryMetering);

            for (const auto& [numId, actorId]: ShardMap_) {
                Send(actorId, new TShardActorEvents::TCalcSizeBytes());
            }
        }

        void OnGetLoadInfo(TLoadInfoEvents::TGetLoadInfo::TPtr& ev) {
            auto responsePtr = std::make_unique<LoadInfoResponse>();
            TLoadInfoCollector collector{responsePtr.get()};
            MeteringMetricsRepo_->Visit(collector);

            GatherHostInfo(responsePtr.get());

            if (LastLocalSlices_) {
                for (const auto& slice: *LastLocalSlices_) {
                    responsePtr->add_slices_starts(slice.Start);
                    responsePtr->add_slices_ends(slice.End);
                }
            }

            Send(
                    ev->Get()->ReplyTo,
                    new TLoadInfoEvents::TGetLoadInfoResult{std::move(responsePtr)});
        }

        // TODO(msherbakov): remove later
        void MakeAssignments() {
            THashMap<TString, TVector<TShardId>> assignments;
            THashSet<TShardId> localAssignments;
            for (const auto& [numId, shardConfig]: ShardConfigs_) {
                auto&& host = Assigner_.Assign(shardConfig.ShardStrId);
                assignments[host.Fqdn].push_back(numId);
            }

            auto&& localShards = assignments[ClusterMap_.Local().Fqdn];
            Copy(localShards.begin(), localShards.end(),
                std::inserter(localAssignments, localAssignments.end()));

            Assignments_ = std::move(assignments);
            LastShardAssignment_->LocalShards = std::move(localAssignments);
        }

        ui32 ExecutorPoolForShardActors_;
        THashMap<TNumId, TActorId> ShardMap_;
        THashMap<TString, TShardId> StrToNumId_;

        absl::flat_hash_map<TShardId, TShardConfig> ShardConfigs_;
        TIntrusivePtr<TLocalShardsHolder> LastShardAssignment_;
        std::shared_ptr<TSlices> LastLocalSlices_;

        ui64 CurrentVersion_;
        NSolomon::NDb::IShardConfigDaoPtr ShardDao_;
        NSolomon::NDb::IClusterConfigDaoPtr ClusterDao_;
        NSolomon::NDb::IServiceConfigDaoPtr ServiceDao_;
        NSolomon::NDb::IProjectConfigDaoPtr ProjectDao_;
        NMonitoring::TMetricRegistry& Registry_;
        std::shared_ptr<TShardMeteringMetricsRepository> MeteringMetricsRepo_;
        TDuration MemoryMeteringCollectionInterval_;
        IThreadPool* ProcessorsExecutor_;
        IIdValidatorPtr IdValidator_;
        TNumIdGenerator NumIdGenerator_;
        const IClusterMap& ClusterMap_;
        const bool IsStandalone_{false};
        // TODO(msherbakov): remove once we have real load balancing
        TShardAssigner Assigner_;
        TCreateShardOpts CreateShardOpts_;
        THashMap<TString, TVector<TShardId>> Assignments_;
        std::shared_ptr<IClusterRpc<NMemStore::IMemStoreRpc>> MemStoreRpc_;
        TActorId MemStoreClusterWatcher_;
        IRequestTracerPtr Tracer_;
        TActorId SliceletId_;
        TString YasmPrefix_;
    };

} // namespace

    NActors::IActor* CreateShardManager(
        const IngestorConfig& ingestorConfig,
        ui32 executorPool,
        NSolomon::NDb::IShardConfigDaoPtr shardDao,
        NSolomon::NDb::IClusterConfigDaoPtr clusterDao,
        NSolomon::NDb::IServiceConfigDaoPtr serviceDao,
        NSolomon::NDb::IProjectConfigDaoPtr projectDao,
        NMonitoring::TMetricRegistry& registry,
        std::shared_ptr<TShardMeteringMetricsRepository> meteringMetricsRepo,
        IThreadPool* processorsExecutor,
        const IClusterMap& cluster,
        bool isStandalone,
        std::shared_ptr<IClusterRpc<NMemStore::IMemStoreRpc>> memStoreRpc,
        TActorId memStoreClusterWatcher,
        IRequestTracerPtr tracer,
        TActorId sliceletId,
        TString yasmPrefix)
    {
        return new TShardManager(
            ingestorConfig,
            executorPool,
            std::move(shardDao),
            std::move(clusterDao),
            std::move(serviceDao),
            std::move(projectDao),
            registry,
            std::move(meteringMetricsRepo),
            processorsExecutor,
            cluster,
            isStandalone,
            std::move(memStoreRpc),
            memStoreClusterWatcher,
            std::move(tracer),
            sliceletId,
            std::move(yasmPrefix));
    }

} // namespace NSolomon::NIngestor
