#pragma once

#include <solomon/services/ingestor/api/ingestor_grpc_service.pb.h>
#include <solomon/services/ingestor/config/ingestor_config.pb.h>
#include <solomon/services/ingestor/lib/events/events.h>
#include <solomon/services/ingestor/lib/holders/config_holder.h>
#include <solomon/services/ingestor/lib/holders/local_shards_holder.h>
#include <solomon/services/ingestor/lib/requests_tracer/request_tracer.h>
#include <solomon/services/ingestor/lib/shard/data_processor.h>
#include <solomon/services/ingestor/lib/shard_config/shard_config.h>
#include <solomon/services/ingestor/lib/shard_manager/make_shard_configs.h>

#include <solomon/libs/cpp/actors/events/events.h>
#include <solomon/libs/cpp/clients/memstore/rpc.h>
#include <solomon/libs/cpp/metering/shard_metrics_repo.h>
#include <solomon/libs/cpp/shard_key/shard_key.h>
#include <solomon/libs/cpp/clients/slicer/slicer_client.h>
#include <solomon/protos/configs/rpc/rpc_config.pb.h>

#include <library/cpp/actors/core/actor.h>
#include <library/cpp/actors/core/actorsystem.h>
#include <library/cpp/actors/core/event_local.h>
#include <library/cpp/grpc/client/grpc_common.h>

#include <util/generic/fwd.h>
#include <util/generic/ptr.h>

#include <utility>

namespace NSolomon {
    class IClusterMap;
} // namespace NSolomon

namespace NSolomon::NIngestor {
    struct TCreateShardResult {
        TString LeaderHost;
        TString AssignedToHost;
        TString ShardId;
        ui32 NumId{0};
    };

    struct TShardManagerEvents: private TEventSlot<EEventSpace::Ingestor, IS_SHARD_MANAGER> {
        enum {
            EvOnUpdate = SpaceBegin,
            EvOnUpdateFailure,
            EvRequestUpdateConfig,
            EvUpdateShardAssignments,
            EvCreateShard,
            EvCreateShardResult,
            EvGetAssignedShards,
            EvGetAssignedShardsResult,
            EvGetShardAssignments,
            EvGetShardAssignmentsResult,
            FindShard,
            FindShardResult,
            End,
        };

        static_assert(End < SpaceEnd, "too many event types");

        struct TEvUpdateShardAssignments: NActors::TEventLocal<TEvUpdateShardAssignments, EvUpdateShardAssignments> {
            TEvUpdateShardAssignments(TIntrusivePtr<TLocalShardsHolder> shardAssignments)
                : ShardAssignments(std::move(shardAssignments))
            {
            }

            TIntrusivePtr<TLocalShardsHolder> ShardAssignments;
        };

        struct TEvCreateShard: NActors::TEventLocal<TEvCreateShard, EvCreateShard> {
            TEvCreateShard(NActors::TActorId handlerId, TString projectId, TString clusterName, TString serviceName)
                : ShardKey(std::move(projectId), std::move(clusterName), std::move(serviceName))
                , HandlerId(handlerId)
            {
            }

            TShardKey ShardKey;
            NActors::TActorId HandlerId;
        };

        struct TEvCreateShardResult: NActors::TEventLocal<TEvCreateShardResult, EvCreateShardResult> {
            template <typename... TArgs>
            TEvCreateShardResult(TArgs&&... args)
                : Result(std::forward<TArgs>(args)...)
            {
            }

            TErrorOr<TCreateShardResult, TGenericError> Result;
        };

        struct TEvGetAssignedShards: NActors::TEventLocal<TEvGetAssignedShards, EvGetAssignedShards> {
        };

        struct TEvGetAssignedShardsResult: NActors::TEventLocal<TEvGetAssignedShardsResult, EvGetAssignedShardsResult> {
            TVector<TNumId> NumIds;

            explicit TEvGetAssignedShardsResult(TVector<TNumId> numIds)
                : NumIds{std::move(numIds)}
            {
            }
        };

        struct TEvGetShardAssignments: NActors::TEventLocal<TEvGetShardAssignments, EvGetShardAssignments> {
        };

        struct TEvGetShardAssignmentsResult: NActors::TEventLocal<TEvGetShardAssignmentsResult, EvGetShardAssignmentsResult> {
            using TResult = TErrorOr<THashMap<TString, TVector<ui32>>, TGenericError>;

            TResult Result;
            TString LeaderHost;

            explicit TEvGetShardAssignmentsResult(TResult result, TString leaderHost)
                : Result{std::move(result)}
                , LeaderHost{std::move(leaderHost)}
            {}
        };

        /**
         * Asking ShardManager to tell us if this numId is processed by this exact node
         */
        struct TFindShard: public NActors::TEventLocal<TFindShard, FindShard> {
            TNumId NumId;

            explicit TFindShard(TNumId numId) noexcept
                : NumId{numId}
            {
            }
        };

        struct TFindShardResult: public NActors::TEventLocal<TFindShardResult, FindShardResult> {
            NActors::TActorId ShardActor;

            explicit TFindShardResult(NActors::TActorId shardActor) noexcept
                : ShardActor{shardActor}
            {
            }
        };
    };

    NActors::IActor* CreateShardManager(
        const yandex::monitoring::ingestor::IngestorConfig& ingestorConfig,
        ui32 executorPool,
        NSolomon::NDb::IShardConfigDaoPtr shardDao,
        NSolomon::NDb::IClusterConfigDaoPtr clusterDao,
        NSolomon::NDb::IServiceConfigDaoPtr serviceDao,
        NSolomon::NDb::IProjectConfigDaoPtr projectDao,
        NMonitoring::TMetricRegistry& registry,
        std::shared_ptr<TShardMeteringMetricsRepository> meteringMetricsRepo,
        IThreadPool* processorsExecutor,
        const IClusterMap& clusterMap,
        bool isStandalone,
        std::shared_ptr<IClusterRpc<NMemStore::IMemStoreRpc>> memStoreRpc,
        NActors::TActorId memStoreClusterWatcher,
        IRequestTracerPtr tracer,
        NActors::TActorId sliceletId,
        TString yasmPrefix
    );
} // namespace NSolomon::NIngestor
