#include <solomon/services/ingestor/lib/shard_manager/create_shard.h>
#include <solomon/services/ingestor/lib/shard_manager/shard_manager.h>
#include <solomon/services/ingestor/lib/shard_manager/default_limits.h>
#include <solomon/services/ingestor/lib/shard_manager/assignments.h>

#include <solomon/libs/cpp/actors/test_runtime/actor_runtime.h>
#include <solomon/libs/cpp/conf_db/testlib/db.h>
#include <solomon/libs/cpp/conf_db/db.h>
#include <solomon/libs/cpp/id_validator/validator.h>
#include <solomon/libs/cpp/cluster_map/cluster.h>
#include <solomon/libs/cpp/ydb/config/ydb_config.pb.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <ydb/public/sdk/cpp/client/ydb_table/table.h>
#include <ydb/public/sdk/cpp/client/ydb_driver/driver.h>

#include <util/stream/file.h>
#include <util/system/hostname.h>
#include <util/generic/guid.h>

using namespace NActors;
using namespace NSolomon;
using namespace NSolomon::NDb;
using namespace NSolomon::NDb::NModel;
using namespace NSolomon::NIngestor;
using namespace NSolomon::NTesting;

static constexpr TStringBuf LOGIN = "solomon";

struct TMockNumIdGenerator: INumIdGenerator {
    TShardId Generate() const override {
        return 42;
    }
};

struct TMockFactory {
    static IDbConnectionPtr Create(NMonitoring::TMetricRegistry&) {
        return CreateMockConnection();
    }
};

struct TYdbFactory {
    static IDbConnectionPtr Create(NMonitoring::TMetricRegistry& metrics) {
        auto endpoint = Strip(TFileInput("ydb_endpoint.txt").ReadAll());
        auto database = Strip(TFileInput("ydb_database.txt").ReadAll());

        NDb::TYdbConfig config;
        config.SetAddress(endpoint);
        config.SetDatabase(database);

        NYdb::TDriver driver{NYdb::TDriverConfig{}
                .SetEndpoint(endpoint)
                .SetDatabase(database)};

        return CreateYdbConnection(config, metrics);
    }
};

template <typename TConnectionFactory>
class TCreateShardTest: public ::testing::Test {
public:
    TCreateShardTest()
        : Cluster{CreateSingleNodeCluster()}
        , Assigner{*Cluster}
    {
    }

protected:
    void SetUp() override {
        Metrics = MakeHolder<NMonitoring::TMetricRegistry>();
        auto conn = TConnectionFactory::Create(*Metrics);

        auto guid = CreateGuidAsString();
        Projects = conn->CreateProjectDao("/local/Projects" + guid);
        Services = conn->CreateServiceDao("/local/Services" + guid);
        Clusters = conn->CreateClusterDao("/local/Clusters" + guid);
        Shards = conn->CreateShardDao({
                .ShardTablePath = "/local/Shards" + guid,
                .NumPcsTablePath = "/local/NumPcs" + guid,
                .NumIdPcsTablePath = "/local/NumIdPcs" + guid,
        });

        Clusters->CreateTable().GetValueSync();
        Shards->CreateTable().GetValueSync();
        Services->CreateTable().GetValueSync();
        Projects->CreateTable().GetValueSync();
    }

    TShardKey CreateShard(bool canCreateProject = false) {
        auto edge = Runtime->AllocateEdgeActor();
        auto key = MakeKey();
        auto ctx = MakeContext();
        ctx.Opts.SetCanCreateProject(canCreateProject);

        Runtime->Register(NIngestor::CreateShard(edge, 0, key, ctx));
        return key;
    }

    TShardCreationContext MakeContext() {
        TShardCreationContext ctx;
        ctx.User = "solomon";
        ctx.IdValidator = IdValidator.Get();
        ctx.NumIdGenerator = &NumIdGenerator;
        ctx.ShardDao = Shards;
        ctx.ServiceDao = Services;
        ctx.ClusterDao = Clusters;
        ctx.ProjectDao = Projects;
        ctx.Assigner = &Assigner;

        return ctx;
    }

    TShardKey MakeKey() {
        return TShardKey{"projectId", "clusterName", "serviceName"};
    }

public:
    THolder<NMonitoring::TMetricRegistry> Metrics;
    IIdValidatorPtr IdValidator{CreateIdValidator()};
    TMockNumIdGenerator NumIdGenerator;
    THolder<TTestActorRuntime> Runtime = TTestActorRuntime::CreateInited();
    IProjectConfigDaoPtr Projects;
    IServiceConfigDaoPtr Services;
    IClusterConfigDaoPtr Clusters;
    IShardConfigDaoPtr Shards;
    IClusterMapPtr Cluster;
    TShardAssigner Assigner;
};

using TCreateShardTypes = ::testing::Types<TMockFactory, TYdbFactory>;
TYPED_TEST_SUITE(TCreateShardTest, TCreateShardTypes);

TYPED_TEST(TCreateShardTest, DoesntCreateProjectByDefault) {
    this->CreateShard();
    auto ev = this->Runtime->template GrabEdgeEvent<TShardManagerEvents::TEvCreateShardResult>();
    ASSERT_TRUE(ev->Result.Fail());
}

TYPED_TEST(TCreateShardTest, EntitiesAreCreated) {
    auto key = this->CreateShard(true);
    auto ev = this->Runtime->template GrabEdgeEvent<TShardManagerEvents::TEvCreateShardResult>();
    ASSERT_TRUE(ev->Result.Success());
    auto v = ev->Result.Extract();
    ASSERT_EQ(v.AssignedToHost, this->Cluster->NodeById(0)->Fqdn);
    ASSERT_EQ(v.NumId, 42u);
    ASSERT_EQ(v.ShardId, MakeShardId(key));
    // XXX
    ASSERT_EQ(v.LeaderHost, FQDNHostName());

    auto proj = this->Projects->GetById(key.ProjectId).ExtractValueSync();
    ASSERT_TRUE(proj);
    ASSERT_EQ(proj->Id, key.ProjectId);
    ASSERT_EQ(proj->Name, key.ProjectId);
    ASSERT_EQ(proj->CreatedBy, LOGIN);
    ASSERT_EQ(proj->UpdatedBy, LOGIN);

    auto cluster = this->Clusters->GetByName(key.ClusterName, key.ProjectId).ExtractValueSync();
    ASSERT_TRUE(cluster);
    ASSERT_EQ(cluster->Id, key.ClusterName);
    ASSERT_EQ(cluster->Name, key.ClusterName);
    ASSERT_EQ(cluster->CreatedBy, LOGIN);
    ASSERT_EQ(cluster->UpdatedBy, LOGIN);

    auto service = this->Services->GetByName(key.ServiceName, key.ProjectId).ExtractValueSync();
    ASSERT_TRUE(cluster);
    ASSERT_EQ(service->Id, key.ServiceName);
    ASSERT_EQ(service->Name, key.ServiceName);
    ASSERT_EQ(service->CreatedBy, LOGIN);
    ASSERT_EQ(service->UpdatedBy, LOGIN);

    auto shard = this->Shards->GetById(MakeShardId(key), key.ProjectId).ExtractValueSync();
    ASSERT_TRUE(shard);
    ASSERT_EQ(shard->Id, MakeShardId(key));
    ASSERT_EQ(shard->NumId, 42u);
    ASSERT_EQ(shard->CreatedBy, LOGIN);
    ASSERT_EQ(shard->UpdatedBy, LOGIN);
    ASSERT_EQ(shard->State, NModel::EShardState::Active);
    ASSERT_EQ(shard->MaxMemMetrics, MAX_MEM_METRICS);
    ASSERT_EQ(shard->MaxResponseSizeBytes, MAX_RESPONSE_SIZE_BYTES);
    ASSERT_EQ(shard->MaxFileMetrics, MAX_FILE_METRICS);
    ASSERT_EQ(shard->MaxMemMetrics, MAX_MEM_METRICS);
}

TYPED_TEST(TCreateShardTest, CreateWithExistingProject) {
    TProjectConfig project{
            .Id = "projectId",
            .Name = "My project",
            .Owner = "solomon",
    };

    this->Projects->Insert(project).GetValueSync();

    auto key = this->CreateShard();
    auto ev = this->Runtime->template GrabEdgeEvent<TShardManagerEvents::TEvCreateShardResult>();
    ASSERT_TRUE(ev->Result.Success());
}

TYPED_TEST(TCreateShardTest, CreateWithExistingCluster) {
    TProjectConfig project{
            .Id = "projectId",
            .Name = "My project",
            .Owner = "solomon",
    };

    TClusterConfig cluster{
            .Id = "clusterName",
            .Name = "clusterName",
            .ProjectId = "projectId",
    };

    this->Projects->Insert(project).GetValueSync();
    this->Clusters->Insert(cluster).GetValueSync();

    auto key = this->CreateShard();
    auto ev = this->Runtime->template GrabEdgeEvent<TShardManagerEvents::TEvCreateShardResult>();
    ASSERT_TRUE(ev->Result.Success());
}

TYPED_TEST(TCreateShardTest, InvalidIdIsDisallowed) {
    auto edge = this->Runtime->AllocateEdgeActor();
    auto key = this->MakeKey();
    auto ctx = this->MakeContext();

    {
        auto k = key;
        k.ProjectId = "<something>";
        this->Runtime->Register(CreateShard(edge, 0, k, ctx));
        auto ev = this->Runtime->template GrabEdgeEvent<TShardManagerEvents::TEvCreateShardResult>();
        ASSERT_TRUE(ev->Result.Fail());
    }

    {
        auto k = key;
        k.ClusterName = "";
        this->Runtime->Register(CreateShard(edge, 0, k, ctx));
        auto ev = this->Runtime->template GrabEdgeEvent<TShardManagerEvents::TEvCreateShardResult>();
        ASSERT_TRUE(ev->Result.Fail());
    }

    {
        auto k = key;
        k.ServiceName.resize(254, 'a');
        this->Runtime->Register(CreateShard(edge, 0, k, ctx));
        auto ev = this->Runtime->template GrabEdgeEvent<TShardManagerEvents::TEvCreateShardResult>();
        ASSERT_TRUE(ev->Result.Fail());
    }
}
