#include <library/cpp/testing/gtest/gtest.h>

#include <solomon/services/ingestor/lib/shard_manager/shard_manager.h>

#include <solomon/libs/cpp/actors/test_runtime/actor_runtime.h>
#include <solomon/libs/cpp/cluster_map/cluster.h>
#include <solomon/libs/cpp/conf_db/db.h>
#include <solomon/libs/cpp/clients/slicer/slicelet_listener.h>

#include <util/thread/pool.h>

#include <optional>

using namespace NActors;
using namespace NSolomon::NDb;
using namespace NSolomon::NIngestor;
using namespace NSolomon;
using namespace NThreading;
using namespace testing;
using namespace yandex::monitoring::ingestor;
using namespace yandex::monitoring::slicer;

class TTestShardDao: public IShardConfigDao {
public:
    TAsyncVoid CreateTable() override { return MakeFuture(); }
    TAsyncVoid DropTable()  override { return MakeFuture(); }

    TAsyncVoid Insert(const NModel::TShardConfig&) override {
        return MakeFuture();
    }

    TAsyncVoid Delete(const TString&, const TString&) override {
        return MakeFuture();
    }

    TAsyncShardConfigs GetAll() override {
        return MakeFuture<TAsyncShardConfigs::value_type>(TVector<NModel::TShardConfig>{
                { .Id = "yasm_prefix_shard_1", .NumId = 1u, .ClusterId = "cluster_1", .ProjectId = "project_1", .ServiceId = "service_1", },
                { .Id = "yasm_prefix_shard_2", .NumId = 2u, .ClusterId = "cluster_2", .ProjectId = "project_2", .ServiceId = "service_2", },
                { .Id = "yasm_prefix_shard_3", .NumId = 3u, .ClusterId = "cluster_3", .ProjectId = "project_3", .ServiceId = "service_3", },
                { .Id = "yasm_prefix_shard_4", .NumId = 4u, .ClusterId = "cluster_4", .ProjectId = "project_4", .ServiceId = "service_4", },
                { .Id = "yasm_prefix_shard_5", .NumId = 5u, .ClusterId = "cluster_5", .ProjectId = "project_5", .ServiceId = "service_5", },
        });
    }

    TAsyncShardConfig GetById(const TString&, const TString&) override {
        return MakeFuture<TAsyncShardConfig::value_type>();
    }
};

class TTestClusterDao: public IClusterConfigDao {
public:
    TAsyncVoid CreateTable() override { return MakeFuture(); }
    TAsyncVoid DropTable()  override { return MakeFuture(); }

    TAsyncVoid Insert(const NModel::TClusterConfig&) override {
        return MakeFuture();
    }

    TAsyncVoid Delete(const TString&, const TString&) override {
        return MakeFuture();
    }

    TAsyncClusterConfigs GetAll() override {
        return MakeFuture<TAsyncClusterConfigs::value_type>(TVector<NModel::TClusterConfig>{
                { .Id = "cluster_1", .ProjectId = "project_1", },
                { .Id = "cluster_2", .ProjectId = "project_2", },
                { .Id = "cluster_3", .ProjectId = "project_3", },
                { .Id = "cluster_4", .ProjectId = "project_4", },
                { .Id = "cluster_5", .ProjectId = "project_5", },
        });
    }

    TAsyncClusterConfig GetByProject(const TString&, const TString&) override {
        return MakeFuture<TAsyncClusterConfig::value_type>();
    }

    TAsyncClusterConfig GetByName(const TString&, const TString&) override {
        return MakeFuture<TAsyncClusterConfig::value_type>();
    }
};

class TTestServiceDao: public IServiceConfigDao {
public:
    TAsyncVoid CreateTable() override {
        return MakeFuture();
    }

    TAsyncVoid DropTable() override {
        return MakeFuture();
    }

    TAsyncVoid Insert(const NModel::TServiceConfig&) override {
        return MakeFuture();
    }

    TAsyncVoid Delete(const TString&, const TString&) override {
        return MakeFuture();
    }

    TAsyncServiceConfig GetByProject(const TString&, const TString&) override {
        return MakeFuture<TAsyncServiceConfig::value_type>();
    }

    TAsyncServiceConfig GetByName(const TString&, const TString&) override {
        return MakeFuture<TAsyncServiceConfig::value_type>();
    }

    TAsyncServiceConfigs GetAll() override {
        return MakeFuture<TAsyncServiceConfigs::value_type>(TVector<NModel::TServiceConfig>{
                { .Id = "service_1", .ProjectId = "project_1" },
                { .Id = "service_2", .ProjectId = "project_2" },
                { .Id = "service_3", .ProjectId = "project_3" },
                { .Id = "service_4", .ProjectId = "project_4" },
                { .Id = "service_5", .ProjectId = "project_5" },
        });
    }
};

class TTestProjectDao: public IProjectConfigDao {
public:
    TAsyncVoid CreateTable() override {
        return MakeFuture();
    }

    TAsyncVoid DropTable() override {
        return MakeFuture();
    }

    TAsyncBool Exists(const TString&) override {
        return MakeFuture<TAsyncBool::value_type>();
    }

    TAsyncVoid Insert(const NModel::TProjectConfig&) override {
        return MakeFuture();
    }

    TAsyncVoid Delete(const TString&) override {
        return MakeFuture();
    }

    TAsyncProjectConfig GetById(const TString&) override {
        return MakeFuture<TAsyncProjectConfig::value_type>();
    }

    TAsyncProjectConfigs GetAll() override {
        return MakeFuture<TAsyncProjectConfigs::value_type>(TVector<NModel::TProjectConfig>{
                { .Id = "project_1" },
                { .Id = "project_2" },
                { .Id = "project_3" },
                { .Id = "project_4" },
                { .Id = "project_5" },
        });
    }
};

IShardConfigDaoPtr CreateTestShardDao() {
    return MakeIntrusive<TTestShardDao>();
}

IServiceConfigDaoPtr CreateTestServiceDao() {
    return MakeIntrusive<TTestServiceDao>();
}

IClusterConfigDaoPtr CreateTestClusterDao() {
    return MakeIntrusive<TTestClusterDao>();
}

IProjectConfigDaoPtr CreateTestProjectDao() {
    return MakeIntrusive<TTestProjectDao>();
}

struct TUpdateEvent {
    TList<TShardId> NeedUpdate;
    TList<std::pair<TShardId, TString>> NeedRemove;
};

class TShardManagerTest: public ::testing::Test {
protected:
    void SetUp() override {
        Runtime_ = NSolomon::TTestActorRuntime::CreateInited(1, false, true);
        Runtime_->WaitForBootstrap();

        EdgeId_ = Runtime_->AllocateEdgeActor();
        Runtime_->SetLogPriority(NSolomon::Wal, NActors::NLog::PRI_DEBUG);
    }

    void TearDown() override {
        Runtime_.Reset();
    }

    THolder<NSolomon::TTestActorRuntime> Runtime_;
    NMonitoring::TMetricRegistry Registry_;
    THolder<IThreadPool> ThreadPool_{CreateThreadPool(1)};
    NSolomon::TClusterMapBase Cluster_{};
    TActorId EdgeId_;
};

class TFailingShardConfigDao: public IShardConfigDao {
    TAsyncVoid CreateTable() override {
        return MakeFuture();
    }

    TAsyncVoid DropTable() override {
        return MakeFuture();
    }

    TAsyncVoid Insert(const NModel::TShardConfig&) override {
        return MakeFuture();
    }

    TAsyncVoid Delete(const TString&, const TString&) override {
        return MakeFuture();
    }

    TAsyncShardConfigs GetAll() override {
        ++GetAllCallsCnt;
        auto p = NewPromise<TAsyncShardConfigs::value_type>();
        p.SetException("failure");

        return p.GetFuture();
    }

    TAsyncShardConfig GetById(const TString&, const TString&) override {
        return {};
    }

public:
    size_t GetAllCallsCnt{0};
};

TEST_F(TShardManagerTest, CoremonAssignmentsAreProcessed) {
    auto shardManagerId = Runtime_->Register(CreateShardManager(
            yandex::monitoring::ingestor::IngestorConfig{},
            0,
            CreateTestShardDao(),
            CreateTestClusterDao(),
            CreateTestServiceDao(),
            CreateTestProjectDao(),
            Registry_,
            nullptr, // meteringRepo
            ThreadPool_.Get(), // processorsExecutor
            Cluster_,
            false, // isStandalone
            nullptr, // memStoreRpc
            {}, // memStoreClusterWatcher
            nullptr, // tracer
            TActorId{},
            "yasm_prefix")
    );

    Runtime_->WaitForBootstrap();

    TDispatchOptions opts;
    opts.FinalEvents.emplace_back(TConfigsPullerEvents::TConfigsResponse::EventType, 1);
    Runtime_->DispatchEvents(opts);

    {
        Runtime_->Send(shardManagerId, EdgeId_, MakeHolder<TShardManagerEvents::TEvGetAssignedShards>());
        auto ev = Runtime_->GrabEdgeEvent<TShardManagerEvents::TEvGetAssignedShardsResult>(EdgeId_);
        EXPECT_TRUE(ev->Get()->NumIds.empty());
    }

    {
        auto newAssignments = MakeIntrusive<TLocalShardsHolder>();
        newAssignments->LocalShards = {1, 2, 3};

        Runtime_->Send(shardManagerId, MakeHolder<TShardManagerEvents::TEvUpdateShardAssignments>(std::move(newAssignments)));

        Runtime_->Send(shardManagerId, EdgeId_, MakeHolder<TShardManagerEvents::TEvGetAssignedShards>());
        auto ev = Runtime_->GrabEdgeEvent<TShardManagerEvents::TEvGetAssignedShardsResult>(EdgeId_);
        EXPECT_EQ(ev->Get()->NumIds.size(), 3ul);
        EXPECT_THAT(ev->Get()->NumIds, ElementsAre(1, 2, 3));
    }

    {
        auto newAssignments = MakeIntrusive<TLocalShardsHolder>();
        newAssignments->LocalShards = { 1, 4, 5 };

        Runtime_->Send(shardManagerId, MakeHolder<TShardManagerEvents::TEvUpdateShardAssignments>(std::move(newAssignments)));

        Runtime_->Send(shardManagerId, EdgeId_, MakeHolder<TShardManagerEvents::TEvGetAssignedShards>());
        auto ev = Runtime_->GrabEdgeEvent<TShardManagerEvents::TEvGetAssignedShardsResult>(EdgeId_);
        EXPECT_EQ(ev->Get()->NumIds.size(), 3ul);
        EXPECT_THAT(ev->Get()->NumIds, ElementsAre(1, 4, 5));
    }
}

TEST_F(TShardManagerTest, SlicerAssignmentsAreProcessed) {
    auto shardManagerId = Runtime_->Register(CreateShardManager(
            yandex::monitoring::ingestor::IngestorConfig{},
            0,
            CreateTestShardDao(),
            CreateTestClusterDao(),
            CreateTestServiceDao(),
            CreateTestProjectDao(),
            Registry_,
            nullptr, // meteringRepo
            ThreadPool_.Get(), // processorsExecutor
            Cluster_,
            false, // isStandalone
            nullptr, // memStoreRpc
            {}, // memStoreClusterWatcher
            nullptr, // tracer
            TActorId{},
            "yasm_prefix")
    );

    Runtime_->WaitForBootstrap();

    TDispatchOptions opts;
    opts.FinalEvents.emplace_back(TConfigsPullerEvents::TConfigsResponse::EventType, 1);
    Runtime_->DispatchEvents(opts);

    {
        GetSlicesByHostResponse newSlices;
        newSlices.add_assigned_starts(1);
        newSlices.add_assigned_ends(3);

        Runtime_->Send(shardManagerId, MakeHolder<NSlicerClient::TSliceletListenerEvents::TSlicesUpdate>(
            std::move(newSlices)
        ));

        Runtime_->Send(shardManagerId, EdgeId_, MakeHolder<TShardManagerEvents::TEvGetAssignedShards>());
        auto ev = Runtime_->GrabEdgeEvent<TShardManagerEvents::TEvGetAssignedShardsResult>(EdgeId_);
        EXPECT_EQ(ev->Get()->NumIds.size(), 3ul);
        EXPECT_THAT(ev->Get()->NumIds, ElementsAre(1, 2, 3));
    }

    {
        GetSlicesByHostResponse newSlices;

        newSlices.add_assigned_starts(1);
        newSlices.add_assigned_ends(1);

        newSlices.add_assigned_starts(4);
        newSlices.add_assigned_ends(5);

        Runtime_->Send(shardManagerId, MakeHolder<NSlicerClient::TSliceletListenerEvents::TSlicesUpdate>(
            std::move(newSlices)
        ));

        Runtime_->Send(shardManagerId, EdgeId_, MakeHolder<TShardManagerEvents::TEvGetAssignedShards>());
        auto ev = Runtime_->GrabEdgeEvent<TShardManagerEvents::TEvGetAssignedShardsResult>(EdgeId_);
        EXPECT_EQ(ev->Get()->NumIds.size(), 3ul);
        EXPECT_THAT(ev->Get()->NumIds, ElementsAre(1, 4, 5));
    }
}

TEST_F(TShardManagerTest, ShardManagerRequestsConfigsEvenAfterAnError) {
    auto shardDao = MakeIntrusive<TFailingShardConfigDao>();
    EXPECT_EQ(shardDao->GetAllCallsCnt, 0ul);

    Runtime_->Register(CreateShardManager(
        yandex::monitoring::ingestor::IngestorConfig{},
        0,
        shardDao,
        CreateTestClusterDao(),
        CreateTestServiceDao(),
        CreateTestProjectDao(),
        Registry_,
        nullptr, // meteringRepo
        ThreadPool_.Get(), // processorsExecutor
        Cluster_,
        false, // isStandalone
        nullptr, // memStoreRpc
        {}, // memStoreClusterWatcher
        nullptr, // tracer
        TActorId{},
        "yasm_prefix")
    );

    Runtime_->WaitForBootstrap();
    Runtime_->AdvanceCurrentTime(TDuration::Minutes(1));
    Runtime_->DispatchEvents({}, TDuration::Seconds(1));
    EXPECT_EQ(shardDao->GetAllCallsCnt, 1ul);

    Runtime_->AdvanceCurrentTime(TDuration::Minutes(1));
    Runtime_->DispatchEvents({}, TDuration::Seconds(1));
    EXPECT_GT(shardDao->GetAllCallsCnt, 1ul);
}
