#include "doubles_out.h"

#include <util/generic/yexception.h>
#include <util/string/builder.h>

#include <contrib/libs/double-conversion/double-conversion/double-conversion.h>

namespace NSolomon::NIngestor {
namespace {

using double_conversion::StringBuilder;
using double_conversion::DoubleToStringConverter;

struct TBuilder {
    alignas(StringBuilder) char Store[sizeof(StringBuilder)];
    StringBuilder* SB;

    inline TBuilder(char* buf, size_t len) noexcept
        : SB(new (Store) StringBuilder(buf, len))
    {
    }
};

// As in standard output (https://nda.ya.ru/t/1ywLfXsR3Vthcc)
constexpr ui64 BUF_LEN = 512;
constexpr TStringBuf INF_SYMBOL = "inf";
constexpr TStringBuf NAN_SYMBOL = "NaN";
constexpr char EXP_SYMBOL = 'E';

// TODO: parallel usage?
inline DoubleToStringConverter& GetStringConverter() noexcept {
    struct TCvt: public DoubleToStringConverter {
        inline TCvt() noexcept
            : DoubleToStringConverter(
                EMIT_TRAILING_DECIMAL_POINT | EMIT_TRAILING_ZERO_AFTER_POINT,
                INF_SYMBOL.data(), NAN_SYMBOL.data(), EXP_SYMBOL,
                // [10^-3; 10^7) -- as in Java (https://nda.ya.ru/t/66oQPshK3Vthgx)
                -3, 7,
                // not used in ToShortest mode, so pass zeros
                0, 0
              )
        {}
    };

    return *SingletonWithPriority<TCvt, 0>();
}

/// "1E-7" --> "1.0E-7"
size_t AddTrailingZeroInScientificMode(char* buf, size_t len) {
    char* end = buf + len;
    char* exp = nullptr;

    for (char* c = buf; c != end; ++c) {
        if ('.' == *c) {
            return len;
        }

        if (EXP_SYMBOL == *c) {
            exp = c;
            break;
        }
    }

    if (nullptr == exp) {
        return len;
    }

    size_t lenOfTerminated = (len + 1);
    Y_ENSURE(lenOfTerminated + 2 <= BUF_LEN, "string representation of a double is too big");

    for (char* c = end; c >= exp; --c) {
        *(c + 2) = *c;
    }

    *exp = '.';
    *(exp + 1) = '0';

    return len + 2;
}

/// Tries to mimic Java's Double.toString() as much as possible for backward compatibility
TString DoDoubleToString(double d) {
    char buf[BUF_LEN];
    TBuilder sb(buf, sizeof(buf));

    Y_ENSURE(GetStringConverter().ToShortest(d, sb.SB), "conversion double -> string has failed");

    size_t len = sb.SB->position();
    buf[len] = 0;
    size_t newLen = AddTrailingZeroInScientificMode(buf, len);

    return TString{buf, newLen};
}

} // namespace

TString DoubleToString(double value) {
    double floatingPointPart = value - static_cast<i64>(value);
    i64* bits = reinterpret_cast<i64*>(&floatingPointPart);

    if (*bits == 0ULL) {
        return TStringBuilder() << static_cast<i64>(value);
    }

    return DoDoubleToString(value);
}

} // namespace NSolomon::NIngestor
