#include <solomon/services/ingestor/lib/type_utils/histogram_utils.h>
#include <solomon/services/ingestor/lib/type_utils/doubles_out.h>

#include <library/cpp/monlib/metrics/labels.h>
#include <library/cpp/testing/gtest/gtest.h>

#include <cmath>

using namespace NMonitoring;
using namespace NSolomon::NIngestor;
using namespace NHistogramUtils;

TEST(TBoundToLabelTest, InvalidInput) {
    {
        double nan_bound = std::nan("");
        ASSERT_THROW(BoundToLabel(nan_bound), yexception);
    }

    if constexpr (std::numeric_limits<double>::has_quiet_NaN) {
        double nan_bound = std::numeric_limits<double>::quiet_NaN();
        ASSERT_THROW(BoundToLabel(nan_bound), yexception);
    }

    if constexpr (std::numeric_limits<double>::has_signaling_NaN) {
        double nan_bound = std::numeric_limits<double>::signaling_NaN();
        ASSERT_THROW(BoundToLabel(nan_bound), yexception);
    }

    if constexpr (std::numeric_limits<double>::has_infinity) {
        double inf_bound = std::numeric_limits<double>::infinity();
        ASSERT_THROW(BoundToLabel(inf_bound), yexception);

        double negative_inf_bound = -std::numeric_limits<double>::infinity();
        ASSERT_THROW(BoundToLabel(negative_inf_bound), yexception);
    }
}

TEST(TBoundToLabelTest, ValidInput) {
    {
        double zero_bound = 0 ;
        auto label = BoundToLabel(zero_bound);

        ASSERT_EQ(label.Value(), "0");
    }

    {
        double neg_zero_bound = -0.0;
        auto label = BoundToLabel(neg_zero_bound);

        ASSERT_EQ(label.Value(), "-0.0");
    }

    {
        double value = 1.234;
        auto label = BoundToLabel(value);

        ASSERT_EQ(label.Value(), "1.234");
        ASSERT_EQ(DoubleToString(value), "1.234");
    }

    {
        double value = 1.000004;
        auto label = BoundToLabel(value);

        ASSERT_EQ(label.Value(), "1.000004");
        ASSERT_EQ(DoubleToString(value), "1.000004");
    }

    // Check that values that are outside of [10^-3, 10^7) are represented in scientific notation and vice versa

    {
        double value = std::pow(10, -3);
        double outside = value / 10;
        double inside = value + 1;

        {
            auto label = BoundToLabel(value);
            ASSERT_EQ(label.Value(), "0.001");
        }

        {
            auto label = BoundToLabel(outside);
            ASSERT_EQ(label.Value(), "1.0E-4");
        }

        {
            auto label = BoundToLabel(inside);
            ASSERT_EQ(label.Value(), "1.001");
        }
    }

    // 10^7 should be represented in scientific notation, but since it has no fractional part it's represented
    // as an integer
    {
        double value = std::pow(10, 7);
        double inside = value - 0.9;
        double outside = value + 0.9;

        {
            auto label = BoundToLabel(value);
            ASSERT_EQ(label.Value(), "10000000");
        }

        {
            auto label = BoundToLabel(outside);
            ASSERT_EQ(label.Value(), "1.00000009E7");
        }

        {
            auto label = BoundToLabel(inside);
            ASSERT_EQ(label.Value(), "9999999.1");
        }
    }

    {
        double value = -std::pow(10, -3);
        double inside = value - 1;
        double outside = value / 10;

        {
            auto label = BoundToLabel(value);
            ASSERT_EQ(label.Value(), "-0.001");
        }

        {
            auto label = BoundToLabel(outside);
            ASSERT_EQ(label.Value(), "-1.0E-4");
        }

        {
            auto label = BoundToLabel(inside);
            ASSERT_EQ(label.Value(), "-1.001");
        }
    }

    {
        double value = -std::pow(10, 7);
        double outside = value - 0.9;
        double inside = value + 0.9;

        {
            auto label = BoundToLabel(value);
            ASSERT_EQ(label.Value(), "-10000000");
        }

        {
            auto label = BoundToLabel(outside);
            ASSERT_EQ(label.Value(), "-1.00000009E7");
        }

        {
            auto label = BoundToLabel(inside);
            ASSERT_EQ(label.Value(), "-9999999.1");
        }
    }
}

TEST(TSplitBucketsTest, SameBuckets) {
    TBucketBounds buckets{0.0, 1.1, 1.1};
    TBucketValues values{   1,   2,   3};
    auto snapshot = ExplicitHistogramSnapshot(buckets, values);
    auto ht = SplitBuckets(std::move(snapshot));

    ASSERT_EQ(ht.size(), 2u);
    ASSERT_EQ(ht.at(TLabel{"bin", "0"}), 1u);
    ASSERT_EQ(ht.at(TLabel{"bin", "1.1"}), 3u);
}

TEST(TSplitBucketsTest, HappyPath) {
    TBucketBounds buckets{0.0, 1.1, 2.3456, 4};
    TBucketValues values{   3,   1,      2, 8};
    auto snapshot = ExplicitHistogramSnapshot(buckets, values);
    auto binLabelToValue = SplitBuckets(std::move(snapshot));

    ASSERT_EQ(binLabelToValue.size(), 4u);
    ASSERT_TRUE(binLabelToValue.contains({"bin", "0"}));
    ASSERT_TRUE(binLabelToValue.contains({"bin", "1.1"}));
    ASSERT_TRUE(binLabelToValue.contains({"bin", "2.3456"}));
    ASSERT_TRUE(binLabelToValue.contains({"bin", "4"}));

    ASSERT_EQ(binLabelToValue.at(TLabel{"bin", "0"}), 3u);
    ASSERT_EQ(binLabelToValue.at(TLabel{"bin", "1.1"}), 1u);
    ASSERT_EQ(binLabelToValue.at(TLabel{"bin", "2.3456"}), 2u);
    ASSERT_EQ(binLabelToValue.at(TLabel{"bin", "4"}), 8u);
}
