#include "http.h"

#include <solomon/services/memstore/config/memstore_config.pb.h>
#include <solomon/services/memstore/lib/api/service.h>
#include <solomon/services/memstore/lib/config/config.h>
#include <solomon/services/memstore/lib/index/index_limiter.h>
#include <solomon/services/memstore/lib/index/memory_tools.h>
#include <solomon/services/memstore/lib/index/shard_manager.h>
#include <solomon/services/memstore/lib/storage/cluster.h>
#include <solomon/services/memstore/lib/wal/init.h>

#include <solomon/libs/cpp/actors/metrics/actor_runtime_metrics.h>
#include <solomon/libs/cpp/actors/runtime/actor_runtime.h>
#include <solomon/libs/cpp/clients/kikimr/rpc.h>
#include <solomon/libs/cpp/cluster_membership/actor.h>
#include <solomon/libs/cpp/cluster_membership/grpc_service.h>
#include <solomon/libs/cpp/conf_db/db.h>
#include <solomon/libs/cpp/conf_db/puller/puller.h>
#include <solomon/libs/cpp/config/units.h>
#include <solomon/libs/cpp/grpc/executor/limits.h>
#include <solomon/libs/cpp/grpc/server/server.h>
#include <solomon/libs/cpp/host_resolver/host_resolver.h>
#include <solomon/libs/cpp/load_info_service/load_info_service.h>
#include <solomon/libs/cpp/local_shard_provider/local_shard_provider.h>
#include <solomon/libs/cpp/logging/logging.h>
#include <solomon/libs/cpp/metering/metric_suppliers.h>
#include <solomon/libs/cpp/metering/shard_metrics_repo.h>
#include <solomon/libs/cpp/secrets/secrets.h>
#include <solomon/libs/cpp/selfmon/selfmon.h>
#include <solomon/libs/cpp/selfmon/service/proto_page.h>
#include <solomon/libs/cpp/signals/signals.h>
#include <solomon/libs/cpp/clients/slicer/slicelet_listener.h>
#include <solomon/libs/cpp/ydb/driver.h>
#include <solomon/libs/cpp/minidump/minidump.h>

#include <library/cpp/actors/core/process_stats.h>
#include <library/cpp/getoptpb/getoptpb.h>
#include <library/cpp/svnversion/svnversion.h>

#include <util/stream/output.h>
#include <util/system/getpid.h>
#include <util/system/hostname.h>

#include <utility>

using yandex::solomon::config::http::HttpServerConfig;
using namespace NSolomon::NDb;
using namespace NSolomon::NLoadInfo;
using namespace NSolomon::NSlicerClient;
using namespace NSolomon::NMemStore::NIndex;
using namespace yandex::monitoring::memstore;
using namespace NActors;

namespace NSolomon::NMemStore {
namespace {

constexpr size_t MEMORY_LIMIT = 45ull * 1024 * 1024 * 1024;

void ResolveAllAddresses(MemStoreConfig& config) {
    auto replace = [r = CachingResolver(ConductorGroupResolver())](auto* addresses) {
        TAddressSet resolved;
        for (const TString& a: *addresses) {
            Cerr << "resolving " << a << Endl;
            r->Resolve(a, &resolved);
        }
        addresses->Clear();
        for (const TString& a: resolved) {
            addresses->Add()->assign(a);
        }
    };

    replace(config.mutable_ydb_client()->mutable_grpcconfig()->mutable_addresses());
}

NActors::TActorId CreateStorage(
    TActorRuntime& runtime,
    const MemStoreConfig& config,
    NMonitoring::TMetricRegistry& metrics)
{
    auto kikimrRpc = NSolomon::NKikimr::CreateClusterGrpc(
            config.ydb_client().GetGrpcConfig(),
            metrics,
            config.client_id());

    auto storage = StorageCluster({
            .NodeId = NConfig::GetNodeId(config.cluster()),
            .Rpc = kikimrRpc,
            .VolumePath = config.volume_path(),
            .WatchDelay = TDuration::Seconds(5),
            .MetricRegistry = metrics
    });

    auto storageId = runtime.Register(std::move(storage));
    NSelfMon::RegisterPage(runtime.ActorSystem(), "/storage", "Storage", storageId);
    return storageId;
}

struct TConfigsPuller {
    IDbConnectionPtr DbConnection; // need to be present in order for a shards dao to operate properly
    TActorId Actor;
};

TConfigsPuller CreateConfigsPullerActor(
    TActorRuntime& runtime,
    const MemStoreConfig& appConfig,
    NSecrets::ISecretProvider& secretProvider,
    NMonitoring::TMetricRegistry& registry)
{
    const auto& config = appConfig.configs_puller();
    auto updateInterval = FromProtoTime(config.update_interval(), TDuration::Minutes(3));
    auto maxUpdateInterval = FromProtoTime(config.max_update_interval(), TDuration::Minutes(12));

    TVector<NDb::NModel::TShardConfig> localShardsForTesting{::Reserve(config.local_shards_for_testing_size())};
    for (auto& localShard: config.local_shards_for_testing()) {
        localShardsForTesting.emplace_back(NDb::NModel::TShardConfig{
            .Id = localShard.project() + "_" + localShard.cluster() + "_" + localShard.service(),
            .NumId = localShard.num_id(),
            .ClusterName = localShard.cluster(),
            .ProjectId = localShard.project(),
            .ServiceName = localShard.service(),
            .State = NModel::EShardState::Active,
        });
    }

    IDbConnectionPtr dbConnection;
    NSolomon::NDb::IShardConfigDaoPtr shardDao;
    TActorId configsPuller;

    if (localShardsForTesting.empty()) {
        auto driver = CreateDriver(appConfig.ydb_config(), secretProvider);
        dbConnection = CreateYdbConnection(std::move(driver), registry);
        const auto& pathPrefix = appConfig.config_path_prefix();
        shardDao = dbConnection->CreateShardDao(TShardTables{
            .ShardTablePath = pathPrefix + "Shard",
            .NumPcsTablePath = pathPrefix + "ShardPcsKey",
            .NumIdPcsTablePath = pathPrefix + "ShardPcsNumId",
        });
    }

    auto actor = CreateConfigsPuller({
            .ShardDao = shardDao,
            .UpdateInterval = updateInterval,
            .MaxUpdateInterval = maxUpdateInterval,
            .Registry = registry,
            .ConfigsForTesting = TConfigs{.Shards = localShardsForTesting},
    });

    const auto& executorName = config.executor();
    if (executorName.empty()) {
        return TConfigsPuller{
            .DbConnection = std::move(dbConnection),
            .Actor = runtime.Register(actor.release()),
        };
    }

    auto executor = runtime.FindExecutorByName(executorName);
    return TConfigsPuller{
        .DbConnection = std::move(dbConnection),
        .Actor = runtime.Register(actor.release(), NActors::TMailboxType::Simple, executor),
    };
}

TActorId CreateShardManager(
        std::shared_ptr<TShardMeteringMetricsRepository> shardMeteringMetricsRepository,
        TActorRuntime& runtime,
        const MemStoreConfig& config,
        TActorId localShardProvider,
        std::shared_ptr<IIndexWriteLimiter> indexLimiter,
        const std::shared_ptr<NMonitoring::TMetricRegistry>& metrics)
{
    NIndex::TShardManagerConfig indexConfig(
            FromProtoTime(config.index().chunk_length()),
            FromProtoTime(config.index().index_capacity()),
            config.index().subshard_count(),
            std::move(shardMeteringMetricsRepository),
            localShardProvider);

    const auto& mainPoolName = config.index().main_pool();
    Y_ENSURE(mainPoolName, "index main pool is not specified");
    ui32 mainPool = runtime.FindExecutorByName(mainPoolName);

    auto findPoolId = [&](const TString& poolName) {
        return poolName ? runtime.FindExecutorByName(poolName) : mainPool;
    };

    indexConfig.RequestPool = findPoolId(config.index().request_pool());
    indexConfig.FtsPool = findPoolId(config.index().fts_pool());

    auto limiter = CreateIndexLimiter(TIndexLimiterConfig::Default(), metrics);
    auto limiterId = runtime.Register(std::move(limiter), NActors::TMailboxType::HTSwap, mainPool);

    auto shardManager = NIndex::CreateShardManager(indexConfig, limiterId, std::move(indexLimiter), metrics);
    auto shardManagerId = runtime.Register(std::move(shardManager), NActors::TMailboxType::HTSwap, mainPool);
    return shardManagerId;
}

} // namespace

int Main(const MemStoreConfig& config, std::shared_ptr<NSecrets::ISecretProvider>&& secretProvider) {
    NSolomon::NMemStore::NIndex::SetMemoryLimit(MEMORY_LIMIT);
    NSolomon::NGrpc::SetThreadsLimit(2);

    // metrics
    auto registry = std::make_shared<NMonitoring::TMetricRegistry>();
    registry->IntGauge({{"sensor", "version"}, {"revision", GetArcadiaLastChange()}})->Set(1);

    // actor system
    auto actorRuntime = TActorRuntime::Create(config.actor_system(), registry);
    actorRuntime->Start();

    AddActorRuntimeMetrics(*actorRuntime, *registry);

    auto shardMeteringMetricsRepository = std::make_shared<TShardMeteringMetricsRepository>();
    auto shardMetricsSupplier = CreateShardMetricsSupplier(shardMeteringMetricsRepository);
    auto hostMetricsSupplier = CreateHostMetricsSupplier(shardMeteringMetricsRepository);

    // http server (also used for exporting metrics, so start as soon as possible)
    auto httpServer = config.has_mon_server()
            ? StartHttpServer(*actorRuntime, *shardMetricsSupplier, *hostMetricsSupplier, registry, config.mon_server())
            : std::unique_ptr<NSolomon::NHttp::THttpServer>();

    NSelfMon::RegisterPage(
            actorRuntime->ActorSystem(),
            "/config", "Config",
            NSelfMon::ProtoPage(std::make_shared<const MemStoreConfig>(config)));

    // components
    auto storage = CreateStorage(*actorRuntime, config, *registry);

    TActorId sliceletListener;
    if (config.has_slicelet_config()) {
        TStringBuilder localAddress;
        localAddress << FQDNHostName() << ':' << config.grpc_server().GetPort(0);

        auto& sliceletConfig = config.slicelet_config();

        if (sliceletConfig.has_local_slices_for_testing()) {
            NSlicer::NApi::TSlices slices;
            auto& slicesProto = sliceletConfig.local_slices_for_testing();

            for (int i = 0; i != slicesProto.assigned_starts().size(); ++i) {
                slices.emplace(slicesProto.assigned_starts(i), slicesProto.assigned_ends(i));
            }

            sliceletListener = actorRuntime->Register(CreateSliceletListenerActorForTesting(std::move(slices)));
        } else {
            sliceletListener = actorRuntime->Register(CreateSliceletListenerActor(ConstructSliceletListenerOptions(
                sliceletConfig,
                *registry,
                localAddress)));
        }
    }

    auto configsPuller = CreateConfigsPullerActor(*actorRuntime, config, *secretProvider, *registry);
    auto localShardProvider = actorRuntime->Register(
            CreateLocalShardProvider(sliceletListener, configsPuller.Actor).release());

    auto indexLimiter = CreateIndexWriteLimiter(TIndexLimiterConfig::Default(), registry);
    auto index = CreateShardManager(
            shardMeteringMetricsRepository,
            *actorRuntime,
            config,
            localShardProvider,
            indexLimiter,
            registry);
    auto walManager = NWal::CreateWalManager(
            actorRuntime->ActorSystem(),
            actorRuntime->FindExecutorByName("wal"),
            index,
            config,
            indexLimiter,
            *registry);

    actorRuntime->ActorSystem().Send(index, new NIndex::TShardManagerEvents::TEnableSnapshots{walManager});

    TActorId clusterMembershipActorId;
    if (config.has_cluster_membership_config()) {
        auto clusterMembershipActor = NClusterMembership::CreateClusterMembershipActor(
            config.has_slicelet_config() ? config.slicelet_config().service() : "memstore",
            config.cluster_membership_config(),
            config.grpc_server().GetPort(0),
            *registry,
            {},
            config.client_id(),
            NClusterMembership::EMembershipMode::Passive);
        clusterMembershipActorId = actorRuntime->Register(clusterMembershipActor.Release());
    }

    // gRPC server
    auto apiServer = NSolomon::NGrpc::MakeGRpcServerFromConfig(actorRuntime->ActorSystem(), *registry, config.grpc_server());
    auto apiExecutor = actorRuntime->FindExecutorByName("api");

    if (config.has_cluster_membership_config()) {
        MON_DEBUG_C(actorRuntime->ActorSystem(), MemStore, "setting the ClusterMembership service up");

        apiServer->AddService(NClusterMembership::CreateClusterMembershipService(
            actorRuntime->ActorSystem(),
            *registry,
            apiExecutor,
            clusterMembershipActorId));
    }

    apiServer->AddService(NApi::CreateMemStoreService(
            actorRuntime->ActorSystem(),
            *registry,
            apiExecutor,
            walManager,
            index,
            indexLimiter));
    apiServer->AddService(CreateLoadInfoService(actorRuntime->ActorSystem(), *registry, apiExecutor, index));
    apiServer->Start();

    // main loop
    sigset_t blockMask;
    SigEmptySet(&blockMask);

    while (true) {
        SigSuspend(&blockMask);

        if (NeedTerminate) {
            actorRuntime->EmergencyLog("MemStore was terminated");

            apiServer->Stop();
            apiServer = nullptr;

            actorRuntime->AsyncPoison(walManager).GetValueSync();
            actorRuntime->AsyncPoison({storage, index}, TDuration::Seconds(5))
                    .GetValueSync();
            actorRuntime = nullptr;

            httpServer.reset();
            return 0;
        } else if (NeedReopenLog) {
            NeedReopenLog = 0;
            actorRuntime->ReopenLog();
            TLogBackend::ReopenAllBackends();
        }
    }
}

} // namespace NSolomon::NMemStore

int main(int argc, const char* argv[]) {
    using namespace NSolomon::NMemStore;
    using namespace NSolomon::NSecrets;

    NLastGetopt::TOpts opts;
    opts.AddLongOption("config", "path to configuration file")
        .Required()
        .RequiredArgument("<path>")
        .Completer(NLastGetopt::NComp::File("*.conf"));
    opts.AddLongOption("secrets", "path to secrets file")
        .Completer(NLastGetopt::NComp::File("*.secrets"));
    opts.AddVersionOption();
    opts.AddHelpOption();
    opts.AddCompletionOption("memstore");
    opts.SetFreeArgsNum(0);

    try {
        NLastGetopt::TOptsParseResult r(&opts, argc, argv);

        std::shared_ptr<ISecretProvider> secretProvider;
        if (r.Has("secrets")) {
            TFsPath secrets(r.Get("secrets"));
            secretProvider = secrets.Exists() ? FileSecretProvider(secrets.GetPath()) : EmptySecretProvider();
        } else {
            secretProvider = EmptySecretProvider();
        }

        NSolomon::InitSignals();
        Cerr << "Starting MemStore (pid: " << GetPID() << ", SVN revision: " << GetArcadiaLastChange() << ')' << Endl;

        auto config = NConfig::Load(r.Get("config"));
        ResolveAllAddresses(config);
        NSolomon::SetupMinidump(config.minidump_config());
        return Main(config, std::move(secretProvider));
    } catch (const NLastGetopt::TUsageException&) {
        opts.PrintUsage("memstore");
    } catch (const NConfig::TConfigParseException& e) {
        Cerr << "Cannot parse config: " << e.AsStrBuf() << Endl;
    } catch (const NConfig::TConfigValidationException& e) {
        Cerr << "Config is not valid: " << e.AsStrBuf() << Endl;
    } catch (...) {
        Cerr << "Unhandled exception: " << CurrentExceptionMessage() << ". Process will terminate" << Endl;
    }

    return 1;
}
