#include "proto.h"

#include <solomon/services/memstore/api/memstore_service.pb.h>
#include <solomon/services/memstore/lib/index/shard.h>
#include <solomon/services/memstore/lib/index/shard_manager.h>

#include <solomon/libs/cpp/grpc/server/handler.h>

using namespace NActors;

using yandex::monitoring::memstore::FindRequest;
using yandex::monitoring::memstore::FindResponse;

namespace NSolomon::NMemStore::NApi {

template <>
void ToProto<NIndex::TShardEvents::TFindResponse, FindResponse>(
        NIndex::TShardEvents::TFindResponse& shardResp,
        FindResponse* resp) noexcept
{
    for (auto& m: shardResp.Metrics) {
        auto* metricProto = resp->add_metrics();
        metricProto->set_type(ToProto(m.Type));

        int labelsSize = 2 * static_cast<int>(m.Labels.size());
        auto* labelsProto = metricProto->mutable_labels_idx();
        labelsProto->Reserve(labelsSize);
        ui32* labelProto = labelsProto->AddNAlreadyReserved(labelsSize);
        for (auto& label: m.Labels) {
            *labelProto++ = label.Key();
            *labelProto++ = label.Value();
        }
    }

    if (shardResp.Strings.Size() != 0) {
        shardResp.Strings.ToProto(
                yandex::solomon::common::StringPool_Compression_LZ4,
                resp->mutable_string_pool());
    }

    resp->set_total_count(shardResp.TotalCount);
    resp->set_truncated(shardResp.TotalCount > shardResp.Metrics.size());
}

namespace {

class TFindHandler: public NGrpc::TBaseHandler<TFindHandler> {
public:
    explicit TFindHandler(NActors::TActorId shardManager) noexcept
        : ShardManager_(shardManager)
    {
    }

public:
    STATEFN(HandleEvent) {
        switch (ev->GetTypeRewrite()) {
            hFunc(NIndex::TShardManagerEvents::TFindShardResponse, OnFindShardResponse);
            hFunc(NIndex::TShardEvents::TFindResponse, OnFindResponse);
        }
    }

    EMode HandleRequest(NGrpc::TRequestState* req) {
        auto* msg = req->GetMessage<FindRequest>();
        Send(ShardManager_, new NIndex::TShardManagerEvents::TFindShard{msg->num_id()}, 0, req->ToCookie());
        return EMode::Async;
    }

    void OnFindShardResponse(NIndex::TShardManagerEvents::TFindShardResponse::TPtr& ev) {
        auto req = NGrpc::TRequestState::FromCookie(ev->Cookie);
        if (req->IsReplied()) {
            // request was already expired
            return;
        }
        auto* msg = req->GetMessage<FindRequest>();

        // TODO: cache shard actor id
        auto& shardActor = ev->Get()->ShardActor;
        if (!shardActor) {
            req->SendError(grpc::NOT_FOUND, TStringBuilder{} << "unknown shard, id=" << msg->num_id());
            return;
        }

        TSelectors selectors;
        try {
            selectors = NSolomon::ParseSelectors(msg->selectors());

            // TODO: make sure that sorting is useless here
            SortBy(selectors.begin(), selectors.end(), [](const TSelector& selector) {
                return selector.Key();
            });
        } catch (TInvalidSelectorsFormat& err) {
            req->SendError(grpc::INVALID_ARGUMENT, TString("invalid selectors: ") + err.what());
            return;
        }

        auto* event = new NIndex::TShardEvents::TFind{msg->num_id(), std::move(selectors), msg->limit()};
        Send(shardActor, event, 0, req.release()->ToCookie());
    }

    void OnFindResponse(NIndex::TShardEvents::TFindResponse::TPtr& ev) {
        if (auto req = NGrpc::TRequestState::FromCookie(ev->Cookie); !req->IsReplied()) {
            auto* resp = google::protobuf::Arena::CreateMessage<FindResponse>(req->Arena());
            ToProto(*ev->Get(), resp);
            req->SendReply(resp);
        }
    }

private:
    NActors::TActorId ShardManager_;
};

} // namespace

IActor* CreateFindHandler(TActorId shardManager) {
    return new TFindHandler{shardManager};
}

} // namespace NSolomon::NMemStore::NApi
