#include "proto.h"

#include <solomon/services/memstore/api/memstore_service.pb.h>
#include <solomon/services/memstore/lib/index/shard.h>
#include <solomon/services/memstore/lib/index/shard_manager.h>

#include <solomon/libs/cpp/grpc/server/handler.h>

using namespace NActors;

using yandex::monitoring::memstore::LabelKeysRequest;
using yandex::monitoring::memstore::LabelKeysResponse;

namespace NSolomon::NMemStore::NApi {

template <>
void ToProto<NIndex::TShardEvents::TLabelKeysResponse, LabelKeysResponse>(
        NIndex::TShardEvents::TLabelKeysResponse& shardResp,
        LabelKeysResponse* resp) noexcept
{
    if (shardResp.Strings.Size() != 0) {
        shardResp.Strings.ToProto(
                yandex::solomon::common::StringPool_Compression_LZ4,
                resp->mutable_string_pool());
    }

    int keysSize = static_cast<int>(shardResp.Keys.size());
    auto* keysProto = resp->mutable_keys_idx();
    keysProto->Reserve(keysSize);
    std::copy(shardResp.Keys.begin(), shardResp.Keys.end(),
              keysProto->AddNAlreadyReserved(keysSize));
}

namespace {

class TLabelKeysHandler: public NGrpc::TBaseHandler<TLabelKeysHandler> {
public:
    explicit TLabelKeysHandler(NActors::TActorId shardManager) noexcept
        : ShardManager_(shardManager)
    {
    }

public:
    STATEFN(HandleEvent) {
        switch (ev->GetTypeRewrite()) {
            hFunc(NIndex::TShardManagerEvents::TFindShardResponse, OnFindShardResponse);
            hFunc(NIndex::TShardEvents::TLabelKeysResponse, OnLabelKeysResponse);
        }
    }

    EMode HandleRequest(NGrpc::TRequestState* req) {
        auto* msg = req->GetMessage<LabelKeysRequest>();
        Send(ShardManager_, new NIndex::TShardManagerEvents::TFindShard{msg->num_id()}, 0, req->ToCookie());
        return EMode::Async;
    }

    void OnFindShardResponse(NIndex::TShardManagerEvents::TFindShardResponse::TPtr& ev) {
        auto req = NGrpc::TRequestState::FromCookie(ev->Cookie);
        if (req->IsReplied()) {
            // request was already expired
            return;
        }
        auto* msg = req->GetMessage<LabelKeysRequest>();

        // TODO: cache shard actor id
        auto& shardActor = ev->Get()->ShardActor;
        if (!shardActor) {
            req->SendError(grpc::NOT_FOUND, TStringBuilder{} << "unknown shard, id=" << msg->num_id());
            return;
        }

        TSelectors selectors;
        try {
            selectors = NSolomon::ParseSelectors(msg->selectors());

            // TODO: make sure that sorting is useless here
            SortBy(selectors.begin(), selectors.end(), [](const TSelector& selector) {
                return selector.Key();
            });
        } catch (TInvalidSelectorsFormat& err) {
            req->SendError(grpc::INVALID_ARGUMENT, TString("invalid selectors: ") + err.what());
            return;
        }

        auto* event = new NIndex::TShardEvents::TLabelKeys{std::move(selectors)};
        Send(shardActor, event, 0, req.release()->ToCookie());
    }

    void OnLabelKeysResponse(NIndex::TShardEvents::TLabelKeysResponse::TPtr& ev) {
        if (auto req = NGrpc::TRequestState::FromCookie(ev->Cookie); !req->IsReplied()) {
            auto* resp = google::protobuf::Arena::CreateMessage<LabelKeysResponse>(req->Arena());
            ToProto(*ev->Get(), resp);
            req->SendReply(resp);
        }
    }

private:
    NActors::TActorId ShardManager_;
};

} // namespace

IActor* CreateLabelKeysHandler(TActorId shardManager) {
    return new TLabelKeysHandler{shardManager};
}

} // namespace NSolomon::NMemStore::NApi
