#include "proto.h"

#include <solomon/services/memstore/api/memstore_service.pb.h>
#include <solomon/services/memstore/lib/index/shard.h>
#include <solomon/services/memstore/lib/index/shard_manager.h>

#include <solomon/libs/cpp/grpc/server/handler.h>

using namespace NActors;

using yandex::monitoring::memstore::LabelValuesRequest;
using yandex::monitoring::memstore::LabelValuesResponse;

namespace NSolomon::NMemStore::NApi {

template <>
void ToProto<NIndex::TShardEvents::TLabelValuesResponse, LabelValuesResponse>(
        NIndex::TShardEvents::TLabelValuesResponse& shardResp,
        LabelValuesResponse* resp) noexcept
{
    if (shardResp.Strings.Size() != 0) {
        shardResp.Strings.ToProto(
                yandex::solomon::common::StringPool_Compression_LZ4,
                resp->mutable_string_pool());
    }

    auto* labelsProto = resp->mutable_labels();
    labelsProto->Reserve(static_cast<int>(shardResp.Labels.size()));

    for (const auto& label: shardResp.Labels) {
        auto* labelProto = labelsProto->Add();
        labelProto->set_key_idx(label.Key);

        int valuesSize = static_cast<int>(label.Values.size());
        auto* valuesProto = labelProto->mutable_values_idx();
        valuesProto->Reserve(valuesSize);
        std::copy(label.Values.begin(), label.Values.end(),
                  valuesProto->AddNAlreadyReserved(valuesSize));

        labelProto->set_total_count(label.MetricCount);
        labelProto->set_truncated(label.Truncated);
    }

    resp->set_metric_count(shardResp.MetricCount);
}

namespace {

class TLabelValuesHandler: public NGrpc::TBaseHandler<TLabelValuesHandler> {
public:
    TLabelValuesHandler(NActors::TActorId shardManager)
        : ShardManager_(shardManager)
    {
    }

public:
    STATEFN(HandleEvent) {
        switch (ev->GetTypeRewrite()) {
            hFunc(NIndex::TShardManagerEvents::TFindShardResponse, OnFindShardResponse);
            hFunc(NIndex::TShardEvents::TLabelValuesResponse, OnLabelValuesResponse);
        }
    }

    EMode HandleRequest(NGrpc::TRequestState* req) {
        auto* msg = req->GetMessage<LabelValuesRequest>();
        Send(ShardManager_, new NIndex::TShardManagerEvents::TFindShard{msg->num_id()}, 0, req->ToCookie());
        return EMode::Async;
    }

    void OnFindShardResponse(NIndex::TShardManagerEvents::TFindShardResponse::TPtr& ev) {
        auto req = NGrpc::TRequestState::FromCookie(ev->Cookie);
        if (req->IsReplied()) {
            // request was already expired
            return;
        }
        auto* msg = req->GetMessage<LabelValuesRequest>();

        // TODO: cache shard actor id
        auto& shardActor = ev->Get()->ShardActor;
        if (!shardActor) {
            req->SendError(grpc::NOT_FOUND, TStringBuilder{} << "unknown shard, id=" << msg->num_id());
            return;
        }

        TSelectors selectors;
        try {
            selectors = NSolomon::ParseSelectors(msg->selectors());

            // TODO: make sure that sorting is useless here
            SortBy(selectors.begin(), selectors.end(), [](const TSelector& selector) {
                return selector.Key();
            });
        } catch (TInvalidSelectorsFormat& err) {
            req->SendError(grpc::INVALID_ARGUMENT, TString("invalid selectors: ") + err.what());
            return;
        }

        auto* event = new NIndex::TShardEvents::TLabelValues{
            std::move(selectors),
            std::vector<TString>{msg->keys().begin(),  msg->keys().end()}, // TODO: avoid strings copy
            msg->text_filter(),
            msg->limit()};
        Send(shardActor, event, 0, req.release()->ToCookie());
    }

    void OnLabelValuesResponse(NIndex::TShardEvents::TLabelValuesResponse::TPtr& ev) {
        if (auto req = NGrpc::TRequestState::FromCookie(ev->Cookie); !req->IsReplied()) {
            auto* resp = google::protobuf::Arena::CreateMessage<LabelValuesResponse>(req->Arena());
            ToProto(*ev->Get(), resp);
            req->SendReply(resp);
        }
    }

private:
    NActors::TActorId ShardManager_;
};

} // namespace

IActor* CreateLabelValuesHandler(TActorId shardManager) {
    return new TLabelValuesHandler{shardManager};
}

} // namespace NSolomon::NMemStore::NApi
