#include "proto.h"

#include <solomon/services/memstore/api/memstore_service.pb.h>
#include <solomon/services/memstore/lib/index/shard.h>
#include <solomon/services/memstore/lib/index/shard_manager.h>

#include <solomon/libs/cpp/grpc/server/handler.h>
#include <solomon/libs/cpp/ts_math/error.h>
#include <solomon/libs/cpp/ts_math/proto.h>
#include <solomon/libs/cpp/stockpile_codec/metric_archive.h>

using namespace NActors;

using yandex::monitoring::memstore::ReadManyRequest;
using yandex::monitoring::memstore::ReadManyResponse;

namespace NSolomon::NMemStore::NApi {

template <>
void ToProto<NIndex::TShardEvents::TReadManyResponse, ReadManyResponse>(
        NIndex::TShardEvents::TReadManyResponse& shardResp,
        ReadManyResponse* resp) noexcept
{
    auto* metricsProto = resp->mutable_metrics();
    metricsProto->Reserve(static_cast<int>(shardResp.Metrics.size()));

    for (auto& metric: shardResp.Metrics) {
        auto* metricProto = metricsProto->Add();

        // (1) fill metadata
        metricProto->set_type(NTsModel::ToProto(metric.Type));

        if (metric.Labels.Defined()) {
            int size = 2 * static_cast<int>(metric.Labels->size());
            auto* labelsProto = metricProto->mutable_labels_idx();
            labelsProto->Reserve(size);
            ui32* labelProto = labelsProto->AddNAlreadyReserved(size);

            for (auto& label: *metric.Labels) {
                *labelProto++ = label.Key();
                *labelProto++ = label.Value();
            }
        }

        // (2) fill time series
        auto* tsProto = metricProto->mutable_time_series();
        tsProto->set_format_version(static_cast<ui32>(shardResp.Format));

        auto type = ToProto(metric.Type);
        auto numPoints = static_cast<ui32>(metric.NumPoints);

        auto* chunk = tsProto->add_chunks();
        chunk->set_from_millis(metric.WindowBegin.MilliSeconds());
        chunk->set_to_millis(metric.WindowEnd.MilliSeconds());
        chunk->set_point_count(numPoints);

        NStockpile::TMetricHeader header;
        header.Type = type;
        header.Owner.ShardId = shardResp.ShardId;

        NStockpile::TMetricArchive archive{
            header,
            shardResp.Format,
            metric.Columns.ToColumnSet(metric.Type),
            numPoints,
            std::move(metric.Data)};

        NStockpile::TCodecOutput output{archive.Data().SizeBytes()};
        NStockpile::TMetricArchiveCodec codec{shardResp.Format};
        codec.Encode(archive, &output);

        // TODO: support encoding metric archive into string, to avoid copy
        auto buffer = output.TakeBuffer();
        chunk->set_content(buffer.Data(), buffer.Size());
    }

    if (shardResp.Strings.Size() != 0) {
        shardResp.Strings.ToProto(
                yandex::solomon::common::StringPool_Compression_LZ4,
                resp->mutable_string_pool());
    }
}

namespace {

class TReadManyHandler: public NGrpc::TBaseHandler<TReadManyHandler> {
public:
    explicit TReadManyHandler(NActors::TActorId shardManager) noexcept
        : ShardManager_(shardManager)
    {
    }

public:
    STATEFN(HandleEvent) {
        switch (ev->GetTypeRewrite()) {
            hFunc(NIndex::TShardManagerEvents::TFindShardResponse, OnFindShardResponse)
            hFunc(NIndex::TShardEvents::TReadManyResponse, OnReadManyResponse);
            hFunc(NIndex::TShardEvents::TReadError, OnReadError);
        }
    }

    EMode HandleRequest(NGrpc::TRequestState* req) {
        auto* msg = req->GetMessage<ReadManyRequest>();
        Send(ShardManager_, new NIndex::TShardManagerEvents::TFindShard{msg->num_id()}, 0, req->ToCookie());
        return EMode::Async;
    }

    void OnFindShardResponse(NIndex::TShardManagerEvents::TFindShardResponse::TPtr& ev) {
        auto req = NGrpc::TRequestState::FromCookie(ev->Cookie);
        if (req->IsReplied()) {
            // request was already expired
            return;
        }
        auto* msg = req->GetMessage<ReadManyRequest>();

        // TODO: cache shard actor id
        auto& shardActor = ev->Get()->ShardActor;
        if (!shardActor) {
            req->SendError(grpc::NOT_FOUND, TStringBuilder{} << "unknown shard, id=" << msg->num_id());
            return;
        }

        auto event = std::make_unique<NIndex::TShardEvents::TReadMany>();

        // (1) parse format
        try {
            event->Format = NStockpile::FormatFromInt(msg->max_timeseries_format());
        } catch (const yexception& e) {
            req->SendError(grpc::INVALID_ARGUMENT, TString{e.AsStrBuf()});
            return;
        }

        // (2) parse pipeline
        try {
            for (auto& operation: msg->operations()) {
                event->Pipeline.Add(NTsMath::FromProto(operation));
            }
        } catch (NTsMath::TException& err) {
            req->SendError(err.Code(), TString{err.AsStrBuf()});
            return;
        }

        // (3) parse from/to boundaries
        event->From = TInstant::MilliSeconds(msg->from_millis());
        event->To = TInstant::MilliSeconds(msg->to_millis());;

        if (msg->has_lookup()) {
            // (4) parse lookup
            event->Lookup = std::make_unique<NIndex::TShardEvents::TReadMany::TLookup>();
            event->Lookup->Limit = msg->lookup().limit();

            try {
                event->Lookup->Selectors = ParseSelectors(msg->lookup().selectors());

                // TODO: make sure that sorting is useless here
                SortBy(event->Lookup->Selectors, [](const TSelector& selector) {
                    return selector.Key();
                });
            } catch (TInvalidSelectorsFormat& err) {
                req->SendError(grpc::INVALID_ARGUMENT, TString("invalid selectors: ") + err.what());
                return;
            }
        } else if (msg->has_resolved_keys()) {
            // (5) parse resolved keys
            if (msg->resolved_keys().metric_labels_size() == 0) {
                req->SendReply(google::protobuf::Arena::CreateMessage<ReadManyResponse>(req->Arena()));
                return;
            }

            event->Resolved = std::make_unique<NIndex::TShardEvents::TReadMany::TResolved>();

            try {
                event->Resolved->Pool = NStringPool::TStringPool{msg->resolved_keys().string_pool()};
            } catch (yexception& err) {
                req->SendError(grpc::INVALID_ARGUMENT, TString("invalid strings pool: ") + err.what());
                return;
            }

            if (int size = msg->resolved_keys().common_labels_idx_size(); size > 0) {
                event->Resolved->CommonLabels.resize(size);
                MemCopy(event->Resolved->CommonLabels.data(), msg->resolved_keys().common_labels_idx().data(), size);
            }

            event->Resolved->ResolvedKeys.reserve(msg->resolved_keys().metric_labels_size());
            for (auto& metric: msg->resolved_keys().metric_labels()) {
                if (int size = metric.labels_idx_size(); size > 0) {
                    auto& labels = event->Resolved->ResolvedKeys.emplace_back();
                    labels.resize(size);
                    MemCopy(labels.data(), metric.labels_idx().data(), size);
                }
            }
        } else {
            req->SendError(grpc::FAILED_PRECONDITION, "Query selectors or resolved keys should be specified");
            return;
        }

        Send(shardActor, event.release(), 0, req.release()->ToCookie());
    }

    static void OnReadManyResponse(NIndex::TShardEvents::TReadManyResponse::TPtr& ev) {
        if (auto req = NGrpc::TRequestState::FromCookie(ev->Cookie); !req->IsReplied()) {
            auto* resp = google::protobuf::Arena::CreateMessage<ReadManyResponse>(req->Arena());
            ToProto(*ev->Get(), resp);
            req->SendReply(resp);
        }
    }

    static void OnReadError(NIndex::TShardEvents::TReadError::TPtr& ev) {
        if (auto req = NGrpc::TRequestState::FromCookie(ev->Cookie); !req->IsReplied()) {
            req->SendError(ev->Get()->Status, ev->Get()->Message);
        }
    }

private:
    NActors::TActorId ShardManager_;
};

} // namespace

IActor* CreateReadManyHandler(TActorId shardManager) {
    return new TReadManyHandler{shardManager};
}

} // namespace NSolomon::NMemStore::NApi
