#include "proto.h"

#include <solomon/services/memstore/api/memstore_service.pb.h>
#include <solomon/services/memstore/lib/index/shard.h>
#include <solomon/services/memstore/lib/index/shard_manager.h>

#include <solomon/libs/cpp/grpc/server/handler.h>
#include <solomon/libs/cpp/ts_math/error.h>
#include <solomon/libs/cpp/ts_math/proto.h>
#include <solomon/libs/cpp/stockpile_codec/metric_archive.h>

using namespace NActors;

using yandex::monitoring::memstore::ReadOneRequest;
using yandex::monitoring::memstore::ReadOneResponse;

namespace NSolomon::NMemStore::NApi {

template <>
void ToProto<NIndex::TShardEvents::TReadOneResponse, ReadOneResponse>(
        NIndex::TShardEvents::TReadOneResponse& shardResp,
        ReadOneResponse* resp) noexcept
{
    auto& metric = shardResp.Metric;
    if (metric.Type == NTsModel::EPointType::Unknown) {
        return;
    }

    auto format = shardResp.Format;
    auto type = ToProto(metric.Type);
    auto numPoints = static_cast<ui32>(metric.NumPoints);

    resp->set_type(type);
    resp->mutable_time_series()->set_format_version(static_cast<ui32>(format));

    auto* chunk = resp->mutable_time_series()->add_chunks();
    chunk->set_from_millis(metric.WindowBegin.MilliSeconds());
    chunk->set_to_millis(metric.WindowEnd.MilliSeconds());
    chunk->set_point_count(numPoints);

    NStockpile::TMetricHeader header;
    header.Type = type;
    header.Owner.ShardId = shardResp.ShardId;

    NStockpile::TMetricArchive archive{
            header,
            format,
            metric.Columns.ToColumnSet(metric.Type),
            numPoints,
            std::move(metric.Data)};

    NStockpile::TCodecOutput output{archive.Data().SizeBytes()};
    NStockpile::TMetricArchiveCodec codec{format};
    codec.Encode(archive, &output);

    // TODO: support encoding metric archive into string, to avoid copy
    auto buffer = output.TakeBuffer();
    chunk->set_content(buffer.Data(), buffer.Size());
}

namespace {

class TReadOneHandler: public NGrpc::TBaseHandler<TReadOneHandler> {
public:
    explicit TReadOneHandler(NActors::TActorId shardManager) noexcept
        : ShardManager_(shardManager)
    {
    }

public:
    STATEFN(HandleEvent) {
        switch (ev->GetTypeRewrite()) {
            hFunc(NIndex::TShardManagerEvents::TFindShardResponse, OnFindShardResponse)
            hFunc(NIndex::TShardEvents::TReadOneResponse, OnReadOneResponse);
            hFunc(NIndex::TShardEvents::TReadError, OnReadError);
        }
    }

    EMode HandleRequest(NGrpc::TRequestState* req) {
        auto* msg = req->GetMessage<ReadOneRequest>();
        Send(ShardManager_, new NIndex::TShardManagerEvents::TFindShard{msg->num_id()}, 0, req->ToCookie());
        return EMode::Async;
    }

    void OnFindShardResponse(NIndex::TShardManagerEvents::TFindShardResponse::TPtr& ev) {
        auto req = NGrpc::TRequestState::FromCookie(ev->Cookie);
        if (req->IsReplied()) {
            // request was already expired
            return;
        }
        auto* msg = req->GetMessage<ReadOneRequest>();

        // TODO: cache shard actor id
        auto& shardActor = ev->Get()->ShardActor;
        if (!shardActor) {
            req->SendError(grpc::NOT_FOUND, TStringBuilder{} << "unknown shard, id=" << msg->num_id());
            return;
        }

        // (1) parse format
        NStockpile::EFormat format;
        try {
            format = NStockpile::FormatFromInt(msg->max_timeseries_format());
        } catch (const yexception& e) {
            req->SendError(grpc::INVALID_ARGUMENT, TString{e.AsStrBuf()});
            return;
        }

        // (2) parse labels
        if (int labelsSize = msg->labels_size(); labelsSize % 2 != 0) {
            req->SendError(grpc::INVALID_ARGUMENT, TStringBuilder{} << "got an odd number of labels: " << labelsSize);
            return;
        }
        auto labels = NSolomon::NLabels::TLabels::OwnedStringsList(msg->labels().begin(), msg->labels().end());

        // (3) parse pipeline
        NTsMath::TOperationPipeline pipeline;
        try {
            for (auto& operation: msg->operations()) {
                pipeline.Add(NTsMath::FromProto(operation));
            }
        } catch (NTsMath::TException& err) {
            req->SendError(err.Code(), TString{err.AsStrBuf()});
            return;
        }

        // (4) parse from/to boundaries
        TInstant from = TInstant::MilliSeconds(msg->from_millis());
        TInstant to = TInstant::MilliSeconds(msg->to_millis());

        auto* event = new NIndex::TShardEvents::TReadOne{
            std::move(labels),
            format,
            std::move(pipeline),
            from, to};
        Send(shardActor, event, 0, req.release()->ToCookie());
    }

    static void OnReadOneResponse(NIndex::TShardEvents::TReadOneResponse::TPtr& ev) {
        if (auto req = NGrpc::TRequestState::FromCookie(ev->Cookie); !req->IsReplied()) {
            auto* resp = google::protobuf::Arena::CreateMessage<ReadOneResponse>(req->Arena());
            ToProto(*ev->Get(), resp);
            req->SendReply(resp);
        }
    }

    static void OnReadError(NIndex::TShardEvents::TReadError::TPtr& ev) {
        if (auto req = NGrpc::TRequestState::FromCookie(ev->Cookie); !req->IsReplied()) {
            req->SendError(ev->Get()->Status, ev->Get()->Message);
        }
    }

private:
    NActors::TActorId ShardManager_;
};

} // namespace

IActor* CreateReadOneHandler(TActorId shardManager) {
    return new TReadOneHandler{shardManager};
}

} // namespace NSolomon::NMemStore::NApi
