#include "proto.h"

#include <solomon/services/memstore/api/memstore_service.pb.h>
#include <solomon/services/memstore/lib/index/shard.h>
#include <solomon/services/memstore/lib/index/shard_manager.h>

#include <solomon/libs/cpp/grpc/server/handler.h>

using namespace NActors;

using yandex::monitoring::memstore::UniqueLabelsRequest;
using yandex::monitoring::memstore::UniqueLabelsResponse;

namespace NSolomon::NMemStore::NApi {

template <>
void ToProto<NIndex::TShardEvents::TUniqueLabelsResponse, UniqueLabelsResponse>(
        NIndex::TShardEvents::TUniqueLabelsResponse& shardResp,
        UniqueLabelsResponse* resp) noexcept
{
    if (shardResp.Strings.Size() != 0) {
        shardResp.Strings.ToProto(
                yandex::solomon::common::StringPool_Compression_LZ4,
                resp->mutable_string_pool());
    }

    auto* labelsListProto = resp->mutable_labels();
    labelsListProto->Reserve(static_cast<int>(shardResp.Labels.size()));

    for (const auto& labels: shardResp.Labels) {
        auto* labelsProto = labelsListProto->Add();

        int labelsSize = 2 * static_cast<int>(labels.size());
        auto* labelsIdxProto = labelsProto->mutable_labels_idx();
        labelsIdxProto->Reserve(labelsSize);
        ui32* ptr = labelsIdxProto->AddNAlreadyReserved(labelsSize);
        for (auto& label: labels) {
            *ptr++ = label.Key();
            *ptr++ = label.Value();
        }
    }
}

namespace {

class TUniqueLabelsHandler: public NGrpc::TBaseHandler<TUniqueLabelsHandler> {
public:
    explicit TUniqueLabelsHandler(TActorId shardManager) noexcept
        : ShardManager_{shardManager}
    {
    }

public:
    STATEFN(HandleEvent) {
        switch (ev->GetTypeRewrite()) {
            hFunc(NIndex::TShardManagerEvents::TFindShardResponse, OnFindShardResponse);
            hFunc(NIndex::TShardEvents::TUniqueLabelsResponse, OnUniqueLabelsResponse);
        }
    }

    EMode HandleRequest(NGrpc::TRequestState* req) {
        auto* msg = req->GetMessage<UniqueLabelsRequest>();
        Send(ShardManager_, new NIndex::TShardManagerEvents::TFindShard{msg->num_id()}, 0, req->ToCookie());
        return EMode::Async;
    }

    void OnFindShardResponse(NIndex::TShardManagerEvents::TFindShardResponse::TPtr& ev) {
        auto req = NGrpc::TRequestState::FromCookie(ev->Cookie);
        if (req->IsReplied()) {
            // request was already expired
            return;
        }
        auto* msg = req->GetMessage<UniqueLabelsRequest>();

        // TODO: cache shard actor id
        auto& shardActor = ev->Get()->ShardActor;
        if (!shardActor) {
            req->SendError(grpc::NOT_FOUND, TStringBuilder{} << "unknown shard, id=" << msg->num_id());
            return;
        }

        TSelectors selectors;
        try {
            selectors = NSolomon::ParseSelectors(msg->selectors());

            // TODO: make sure that sorting is useless here
            SortBy(selectors.begin(), selectors.end(), [](const TSelector& selector) {
                return selector.Key();
            });
        } catch (TInvalidSelectorsFormat& err) {
            req->SendError(grpc::INVALID_ARGUMENT, TString("invalid selectors: ") + err.what());
            return;
        }

        auto* event = new NIndex::TShardEvents::TUniqueLabels{
            std::move(selectors),
            std::vector<TString>{msg->keys().begin(),  msg->keys().end()}, // TODO: avoid strings copy
        };
        Send(shardActor, event, 0, req.release()->ToCookie());
    }

    void OnUniqueLabelsResponse(NIndex::TShardEvents::TUniqueLabelsResponse::TPtr& ev) {
        if (auto req = NGrpc::TRequestState::FromCookie(ev->Cookie); !req->IsReplied()) {
            auto* resp = google::protobuf::Arena::CreateMessage<UniqueLabelsResponse>(req->Arena());
            ToProto(*ev->Get(), resp);
            req->SendReply(resp);
        }
    }

private:
    TActorId ShardManager_;
};

} // namespace

IActor* CreateUniqueLabelsHandler(TActorId shardManager) {
    return new TUniqueLabelsHandler{shardManager};
}

} // namespace NSolomon::NMemStore::NApi
