#include <solomon/services/memstore/api/memstore_service.pb.h>
#include <solomon/services/memstore/lib/index/index_limiter.h>
#include <solomon/services/memstore/lib/index/shard.h>
#include <solomon/services/memstore/lib/index/shard_manager.h>
#include <solomon/services/memstore/lib/wal/wal_events.h>

#include <solomon/libs/cpp/grpc/server/handler.h>

#include <solomon/libs/cpp/logging/logging.h>
#include <solomon/libs/cpp/slog/slog.h>
#include <solomon/libs/cpp/slog/unresolved_meta/header.h>

#include <library/cpp/monlib/metrics/metric_registry.h>

#include <utility>

using namespace NActors;

using yandex::monitoring::memstore::WriteRequest;
using yandex::monitoring::memstore::WriteResponse;

namespace NSolomon::NMemStore::NApi {
namespace {

// TODO: move to a better place
//NMonitoring::IRate* BytesWritten = NMonitoring::TMetricRegistry::Instance()->Rate(
//    {{"sensor", "api.write.bytes.rate"}});
//NMonitoring::IRate* PointsWritten = NMonitoring::TMetricRegistry::Instance()->Rate(
//    {{"sensor", "api.write.points.rate"}});
//NMonitoring::IRate* MetricsWritten = NMonitoring::TMetricRegistry::Instance()->Rate(
//    {{"sensor", "api.write.metrics.rate"}});

constexpr i64 MAX_INFLIGHT_MESSAGES_COUNT = 100'000;

class TWriteHandler: public NGrpc::TBaseHandler<TWriteHandler> {
public:
    TWriteHandler(
            TActorId walManager,
            TActorId index,
            std::shared_ptr<NSolomon::NMemStore::NIndex::IIndexWriteLimiter> indexLimiter,
            NMonitoring::TMetricRegistry& registry) noexcept
        : WalManager_{walManager}
        , ShardManager_{index}
        , Registry_{registry}
        , IndexLimiter_{std::move(indexLimiter)}
    {
        WriteInflightBytesMetric_ = Registry_.IntGauge({{"sensor", "api.write.inflight_bytes"}});
        WriteInflightMessagesMetric_ = Registry_.IntGauge({{"sensor", "api.write.inflight_message_count"}});
        CanAddMessageOpenMetric_ = Registry_.Rate({
                {"sensor", "api.write.can_add_message"},
                {"state", "open"}});
        CanAddMessageCloseMetric_ = Registry_.Rate({
                {"sensor", "api.write.can_add_message"},
                {"state", "close"}});
        WalNotInitializedMetric_ = Registry_.Rate({{"sensor", "api.write.wal_not_initialized"}});
    }

public:
    STATEFN(HandleEvent) {
        switch (ev->GetTypeRewrite()) {
            hFunc(NIndex::TShardEvents::TIndexDone, OnIndexDone);
            hFunc(NWal::TWalEvents::TAddLogRecordResult, OnLogAdded);
            hFunc(NIndex::TShardManagerEvents::TFindShardResponse, OnFindShardResponse);
            sFunc(NWal::TWalEvents::TWalInitialized, OnWalInitialized)
        }
    }

    EMode HandleRequest(NGrpc::TRequestState* req) {
        auto* msg = req->GetMessage<WriteRequest>();
        if (msg->num_id() == 0) {
            req->SendError(grpc::StatusCode::INVALID_ARGUMENT, "numId cannot be zero");
            return EMode::Sync;
        }

        if (IsFirstRequest_) {
            WalNotInitializedMetric_->Inc();
            Send(WalManager_, new NWal::TWalEvents::TSetWriteHandler(SelfId()));
            IsFirstRequest_ = false;
        }

        if (!IsWalInitialized_) {
            WalNotInitializedMetric_->Inc();
            req->SendError(grpc::StatusCode::UNAVAILABLE, TStringBuilder() << "memstore is initializing");
            return EMode::Sync;
        }

        if (WriteMessagesLimiter_.IsLocked()) {
            CanAddMessageCloseMetric_->Inc();
            req->SendError(grpc::StatusCode::RESOURCE_EXHAUSTED, TStringBuilder() << "In flight requests count is over the limit");
            return EMode::Sync;
        }
        CanAddMessageOpenMetric_->Inc();

        if (!IndexLimiter_->CanAddIndexMessage()) {
            req->SendError(grpc::StatusCode::RESOURCE_EXHAUSTED, TStringBuilder() << "In flight requests count is over the limit");
            return EMode::Sync;
        }

        const i64 inflightBytes = static_cast<i64>(msg->metadata().size() + msg->data().size());
        const auto cookie = req->ToCookie();
        WriteInflightBytesMetric_->Add(inflightBytes);
        InflightBytesMap_[cookie] = inflightBytes;
        WriteMessagesLimiter_.AddValue((i64)1);
        Send(ShardManager_, new NIndex::TShardManagerEvents::TFindShard{msg->num_id()}, 0, cookie);
        return EMode::Async;
    }

    void OnWalInitialized() {
        IsWalInitialized_ = true;
    }

    void OnIndexDone(NIndex::TShardEvents::TIndexDone::TPtr& ev) {
        if (auto req = NGrpc::TRequestState::FromCookie(ev->Cookie); !req->IsReplied()) {
            switch (ev->Get()->Status) {
                case NIndex::TShardEvents::TIndexDone::OK: {
                    WriteResponse reply;
                    req->SendReply(&reply);
                    break;
                }
                case NIndex::TShardEvents::TIndexDone::MemoryExhausted: {
                    req->SendError(
                            grpc::StatusCode::RESOURCE_EXHAUSTED,
                            TStringBuilder() << "Memory usage is over the limit");
                    break;
                }
                case NIndex::TShardEvents::TIndexDone::ShardClosed: {
                    req->SendError(
                            grpc::StatusCode::NOT_FOUND,
                            TStringBuilder() << "Shard is closed");
                    break;
                }
            }
        }
        OnRequestProcessed(ev->Cookie);
    }

    void OnLogAdded(NWal::TWalEvents::TAddLogRecordResult::TPtr& ev) {
        auto req = NGrpc::TRequestState::FromCookie(ev->Cookie);
        if (ev->Get()->Fail()) {
            // TIndexDone message for that request will never be sent
            if (!req->IsReplied()) {
                auto error = ev->Get()->ExtractError();
                req->SendError(error.GetStatus(), error.GetMessage());
            }
            OnRequestProcessed(ev->Cookie);
            return;
        }
        // We expect that TIndexDone message will be sent
        req.release();
    }

    std::unique_ptr<NGrpc::TRequestState> CheckFindShardResponse(NIndex::TShardManagerEvents::TFindShardResponse::TPtr& ev) {
        auto req = NGrpc::TRequestState::FromCookie(ev->Cookie);
        if (req->IsReplied()) {
            // request was already expired
            return nullptr;
        }

        auto* msg = req->GetMessage<WriteRequest>();

        if (!ev->Get()->ShardActor) {
            req->SendError(
                    grpc::StatusCode::NOT_FOUND,
                    TStringBuilder() << "shard with numId=" << msg->num_id() << " is not processed by this node");
            return nullptr;
        }

        try {
            TStringInput meta{msg->metadata()};
            auto header = NSlog::NUnresolvedMeta::ReadHeader(&meta);
            Y_UNUSED(header);
            //        Points_ = header.PointsCount;
            //        Metrics_ = header.MetricsCount;
        } catch (NSlog::TDecodeError& err) {
            req->SendError(grpc::INVALID_ARGUMENT, err.what());
            return nullptr;
        }

        try {
            TStringInput data{msg->data()};
            auto header = NSlog::CreateLogDataIterator(&data);
            Y_UNUSED(header);
        } catch (NSlog::TDecodeError& err) {
            req->SendError(grpc::INVALID_ARGUMENT, err.what());
            return nullptr;
        }

        return req;
    }

    void OnFindShardResponse(NIndex::TShardManagerEvents::TFindShardResponse::TPtr& ev) {
        auto req = CheckFindShardResponse(ev);
        if (!req) {
            OnRequestProcessed(ev->Cookie);
            return;
        }

        auto* msg = req->GetMessage<WriteRequest>();
        const auto& shardKey = msg->shard_key();
        auto addLogRecordMsg = std::make_unique<NWal::TWalEvents::TAddLogRecord>(
                msg->num_id(),
                TShardKey{shardKey.project(), shardKey.cluster(), shardKey.service()},
                msg->metadata(),
                msg->data());
        Send(WalManager_, addLogRecordMsg.release(), 0, req.release()->ToCookie());
    }

    bool ProcessWriteRequest(NGrpc::TRequestState* req) {
        auto* msg = req->GetMessage<WriteRequest>();
        try {
            TStringInput meta{msg->metadata()};
            auto header = NSlog::NUnresolvedMeta::ReadHeader(&meta);
            Y_UNUSED(header);
        } catch (NSlog::TDecodeError& err) {
            req->SendError(grpc::INVALID_ARGUMENT, err.what());
            return false;
        }

        try {
            TStringInput data{msg->data()};
            auto header = NSlog::CreateLogDataIterator(&data);
            Y_UNUSED(header);
        } catch (NSlog::TDecodeError& err) {
            req->SendError(grpc::INVALID_ARGUMENT, err.what());
            return false;
        }

        const auto& shardKey = msg->shard_key();
        auto addLogRecordMsg = std::make_unique<NWal::TWalEvents::TAddLogRecord>(
                msg->num_id(),
                TShardKey{shardKey.project(), shardKey.cluster(), shardKey.service()},
                msg->metadata(),
                msg->data());
        Send(WalManager_, addLogRecordMsg.release(), 0, req->ToCookie());
        return true;
    }

private:

    void OnRequestProcessed(ui64 cookie) {
        WriteMessagesLimiter_.AddValue((i64)-1);
        auto it = InflightBytesMap_.find(cookie);
        Y_VERIFY_DEBUG(it != InflightBytesMap_.end());
        if (it != InflightBytesMap_.end()) {
            WriteInflightBytesMetric_->Add(-it->second);
            InflightBytesMap_.erase(it);
        }
    }

private:
    TActorId WalManager_;
    TActorId ShardManager_;
    bool IsWalInitialized_{false};
    bool IsFirstRequest_{true};
    NMonitoring::TMetricRegistry& Registry_;
    absl::flat_hash_map<ui64, i64> InflightBytesMap_;
    NMonitoring::IIntGauge* WriteInflightBytesMetric_{0};
    NMonitoring::IIntGauge* WriteInflightMessagesMetric_{0};
    NMonitoring::IRate* CanAddMessageOpenMetric_{0};
    NMonitoring::IRate* CanAddMessageCloseMetric_{0};
    NMonitoring::IRate* WalNotInitializedMetric_{0};
    NSolomon::NMemStore::NIndex::TSimpleLimiter  WriteMessagesLimiter_{MAX_INFLIGHT_MESSAGES_COUNT, MAX_INFLIGHT_MESSAGES_COUNT / 2};
    std::shared_ptr<NSolomon::NMemStore::NIndex::IIndexWriteLimiter> IndexLimiter_;

    // TODO: move to a better place
    //    size_t Bytes_ = 0;
    //    size_t Points_ = 0;
    //    size_t Metrics_ = 0;
};

} // namespace

IActor* CreateWriteHandler(
        TActorId walManager,
        TActorId shardManager,
        std::shared_ptr<NSolomon::NMemStore::NIndex::IIndexWriteLimiter> indexLimiter,
        NMonitoring::TMetricRegistry& metricRegistry) {
    return new TWriteHandler{walManager, shardManager, std::move(indexLimiter), metricRegistry};
}

} // namespace NSolomon::NMemStore::NApi
