#include "service.h"

#include <solomon/services/memstore/api/memstore_service.grpc.pb.h>

#include <solomon/libs/cpp/grpc/server/service.h>

using namespace NActors;
using namespace NSolomon::NMemStore::NIndex;
using yandex::monitoring::memstore::MemStoreService;

namespace NSolomon::NMemStore::NApi {

IActor* CreateFindHandler(TActorId shardManager);
IActor* CreateLabelKeysHandler(TActorId shardManager);
IActor* CreateLabelValuesHandler(TActorId shardManager);
IActor* CreateListShardsHandler(TActorId shardManager);
IActor* CreateReadManyHandler(TActorId shardManager);
IActor* CreateReadOneHandler(TActorId shardManager);
IActor* CreateUniqueLabelsHandler(TActorId shardManager);
IActor* CreateWriteHandler(
        TActorId walManager,
        TActorId shardManager,
        std::shared_ptr<IIndexWriteLimiter> indexLimiter,
        NMonitoring::TMetricRegistry&);

namespace {

/// Main GRPC service that provides all public API.
class TMemStoreService: public NSolomon::NGrpc::TService<MemStoreService> {
public:
    TMemStoreService(
            TActorSystem& actorSystem,
            NMonitoring::TMetricRegistry& registry,
            ui32 executorPool,
            TActorId walManager,
            TActorId shardManager,
            std::shared_ptr<IIndexWriteLimiter> indexLimiter)
        : TService{actorSystem, registry, executorPool}
        , WalManager_{walManager}
        , ShardManager_{shardManager}
        , IndexLimiter_(std::move(indexLimiter))
    {
    }

public:
    void InitHandlers(grpc::ServerCompletionQueue* cq) override {
        using AsyncService = MemStoreService::AsyncService;

        auto writeHandler = CreateWriteHandler(WalManager_, ShardManager_, IndexLimiter_, GetMetricRegistry());
        RegisterSingleHandler(cq, &AsyncService::RequestWrite, "Write", writeHandler);

        RegisterSingleHandler(cq, &AsyncService::RequestListShards, "ListShards", CreateListShardsHandler(ShardManager_));

        RegisterSingleHandler(cq, &AsyncService::RequestFind, "Find", CreateFindHandler(ShardManager_));
        RegisterSingleHandler(cq, &AsyncService::RequestLabelKeys, "LabelKeys", CreateLabelKeysHandler(ShardManager_));
        RegisterSingleHandler(cq, &AsyncService::RequestLabelValues, "LabelValues", CreateLabelValuesHandler(ShardManager_));
        RegisterSingleHandler(cq, &AsyncService::RequestUniqueLabels, "UniqueLabels", CreateUniqueLabelsHandler(ShardManager_));

        RegisterSingleHandler(cq, &AsyncService::RequestReadMany, "ReadMany", CreateReadManyHandler(ShardManager_));
        RegisterSingleHandler(cq, &AsyncService::RequestReadOne, "ReadOne", CreateReadOneHandler(ShardManager_));
    }

private:
    TActorId WalManager_;
    TActorId ShardManager_;
    std::shared_ptr<IIndexWriteLimiter> IndexLimiter_;
};

} // namespace

TIntrusivePtr<::NGrpc::IGRpcService> CreateMemStoreService(
        TActorSystem& actorSystem,
        NMonitoring::TMetricRegistry& registry,
        ui32 executorPool,
        TActorId walManager,
        TActorId shardManager,
        const std::shared_ptr<IIndexWriteLimiter>& indexLimiter)
{
    return MakeIntrusive<TMemStoreService>(actorSystem, registry, executorPool, walManager, shardManager, indexLimiter);
}

} // namespace NSolomon::NMemStore
