#include <library/cpp/testing/gtest/gtest.h>

#include <solomon/services/memstore/api/memstore_service.pb.h>
#include <solomon/services/memstore/lib/api/proto.h>
#include <solomon/services/memstore/lib/index/shard.h>

#include <solomon/libs/cpp/stockpile_codec/metric_archive.h>

using namespace NSolomon;
using namespace NMemStore;

using yandex::monitoring::memstore::ReadManyResponse;
using yandex::solomon::model::MetricType;

TEST(TReadManyHandlerTest, ToProtoEmpty) {
    NIndex::TShardEvents::TReadManyResponse shardResp{123, NStockpile::EFormat::DELETED_SHARDS_39};

    ReadManyResponse protoResp;
    NApi::ToProto(shardResp, &protoResp);

    ASSERT_FALSE(protoResp.has_string_pool());
    ASSERT_EQ(protoResp.metrics_size(), 0);
}

TEST(TReadManyHandlerTest, ToProtoNonEmpty) {
    ReadManyResponse protoResp;

    const auto from = TInstant::ParseIso8601("2021-09-14T01:02:03Z");
    const auto to = from + TDuration::Minutes(1);

    {
        NStringPool::TStringPoolBuilder strings;
        std::vector<NTsMath::TTimeSeriesEncoded> metrics;

        auto& metric = metrics.emplace_back();
        metric.Labels.ConstructInPlace();
        metric.Labels->emplace_back(strings.Put("host"), strings.Put("memstore-sas-00"));
        metric.Labels->emplace_back(strings.Put("sensor"), strings.Put("mem_usage"));

        metric.NumPoints = 5;
        metric.Type = NTsModel::EPointType::DGauge;
        metric.WindowBegin = from;
        metric.WindowEnd = to;
        metric.Data = NTs::TBitBuffer(TBuffer{"FakeData", 8});

        NIndex::TShardEvents::TReadManyResponse shardResp{
            123,
            NStockpile::EFormat::DELETED_SHARDS_39,
            std::move(metrics),
            strings.Build()};
        NApi::ToProto(shardResp, &protoResp);
    }

    ASSERT_TRUE(protoResp.has_string_pool());

    NStringPool::TStringPool strings{protoResp.string_pool()};
    ASSERT_EQ(strings.Size(), 4u);
    ASSERT_EQ(strings[0], "");
    ASSERT_EQ(strings[1], "host");
    ASSERT_EQ(strings[2], "memstore-sas-00");
    ASSERT_EQ(strings[3], "sensor");
    ASSERT_EQ(strings[4], "mem_usage");

    ASSERT_EQ(protoResp.metrics_size(), 1);

    auto& metric = protoResp.metrics(0);
    ASSERT_EQ(metric.type(), MetricType::DGAUGE);

    ASSERT_EQ(metric.labels_idx_size(), 4);
    auto& labels = metric.labels_idx();
    ASSERT_EQ(strings[labels[0]], "host");
    ASSERT_EQ(strings[labels[1]], "memstore-sas-00");
    ASSERT_EQ(strings[labels[2]], "sensor");
    ASSERT_EQ(strings[labels[3]], "mem_usage");

    ASSERT_TRUE(metric.has_time_series());
    auto& timeSeries = metric.time_series();
    ASSERT_EQ(timeSeries.format_version(), static_cast<ui32>(NStockpile::EFormat::DELETED_SHARDS_39));

    ASSERT_EQ(timeSeries.chunks_size(), 1);
    auto& chunk = timeSeries.chunks(0);
    ASSERT_EQ(chunk.from_millis(), from.MilliSeconds());
    ASSERT_EQ(chunk.to_millis(), to.MilliSeconds());
    ASSERT_EQ(chunk.point_count(), 5u);

    ASSERT_FALSE(chunk.content().empty());
    NStockpile::TMetricArchiveCodec codec{NStockpile::FormatFromInt(timeSeries.format_version())};
    NStockpile::TCodecInput input{chunk.content().data(), chunk.content().size()};
    NStockpile::TMetricArchive archive = codec.Decode(&input);

    ASSERT_EQ(archive.Format(), NStockpile::EFormat::DELETED_SHARDS_39);
    ASSERT_EQ(archive.Header().Type, MetricType::DGAUGE);
    ASSERT_EQ(archive.Header().Owner.ShardId, 123u);
    ASSERT_EQ(archive.Data(), NTs::TBitSpan("FakeData"));
}
