#include <library/cpp/testing/gtest/gtest.h>

#include <solomon/services/memstore/api/memstore_service.pb.h>
#include <solomon/services/memstore/lib/api/proto.h>
#include <solomon/services/memstore/lib/index/shard.h>

#include <solomon/libs/cpp/stockpile_codec/metric_archive.h>

using namespace NSolomon;
using namespace NMemStore;

using yandex::monitoring::memstore::ReadOneResponse;
using yandex::solomon::model::MetricType;

TEST(TReadOneHandlerTest, ToProtoEmpty) {
    NIndex::TShardEvents::TReadOneResponse shardResp{123, NStockpile::EFormat::DELETED_SHARDS_39};

    ReadOneResponse protoResp;
    NApi::ToProto(shardResp, &protoResp);

    ASSERT_EQ(protoResp.type(), MetricType::METRIC_TYPE_UNSPECIFIED);
    ASSERT_FALSE(protoResp.has_time_series());
}

TEST(TReadOneHandlerTest, ToProtoNonEmpty) {
    ReadOneResponse protoResp;

    const auto from = TInstant::ParseIso8601("2021-09-14T01:02:03Z");
    const auto to = from + TDuration::Minutes(1);

    {
        NStringPool::TStringPoolBuilder strings;

        NTsMath::TTimeSeriesEncoded metric;
        metric.Labels.ConstructInPlace();
        metric.Labels->emplace_back(strings.Put("host"), strings.Put("memstore-sas-00"));
        metric.Labels->emplace_back(strings.Put("sensor"), strings.Put("mem_usage"));

        metric.NumPoints = 5;
        metric.Type = NTsModel::EPointType::DGauge;
        metric.WindowBegin = from;
        metric.WindowEnd = to;
        metric.Data = NTs::TBitBuffer(TBuffer{"FakeData", 8});

        NIndex::TShardEvents::TReadOneResponse shardResp{
            123,
            NStockpile::EFormat::DELETED_SHARDS_39,
            std::move(metric),
            strings.Build()};
        NApi::ToProto(shardResp, &protoResp);
    }

    ASSERT_EQ(protoResp.type(), MetricType::DGAUGE);
    ASSERT_TRUE(protoResp.has_time_series());

    auto& timeSeries = protoResp.time_series();
    ASSERT_EQ(timeSeries.format_version(), static_cast<ui32>(NStockpile::EFormat::DELETED_SHARDS_39));

    ASSERT_EQ(timeSeries.chunks_size(), 1);
    auto& chunk = timeSeries.chunks(0);
    ASSERT_EQ(chunk.from_millis(), from.MilliSeconds());
    ASSERT_EQ(chunk.to_millis(), to.MilliSeconds());
    ASSERT_EQ(chunk.point_count(), 5u);

    ASSERT_FALSE(chunk.content().empty());
    NStockpile::TMetricArchiveCodec codec{NStockpile::FormatFromInt(timeSeries.format_version())};
    NStockpile::TCodecInput input{chunk.content().data(), chunk.content().size()};
    NStockpile::TMetricArchive archive = codec.Decode(&input);

    ASSERT_EQ(archive.Format(), NStockpile::EFormat::DELETED_SHARDS_39);
    ASSERT_EQ(archive.Header().Type, MetricType::DGAUGE);
    ASSERT_EQ(archive.Header().Owner.ShardId, 123u);
    ASSERT_EQ(archive.Data(), NTs::TBitSpan("FakeData"));
}
