#include <library/cpp/testing/gtest/gtest.h>

#include <solomon/services/memstore/api/memstore_service.pb.h>
#include <solomon/services/memstore/lib/api/proto.h>
#include <solomon/services/memstore/lib/index/shard.h>

using namespace NSolomon;
using namespace NMemStore;

using yandex::monitoring::memstore::UniqueLabelsResponse;

TEST(TUniqueLabelsHandlerTest, ToProtoEmpty) {
    NIndex::TShardEvents::TUniqueLabelsResponse shardResp;

    UniqueLabelsResponse protoResp;
    NApi::ToProto(shardResp, &protoResp);

    ASSERT_FALSE(protoResp.has_string_pool());
    ASSERT_EQ(protoResp.labels_size(), 0);
}

void CompareStringPoolEq(const NStringPool::TStringPool& stringPool, TVector<TStringBuf> strings) {
    TVector<TStringBuf> stringPoolStrings;

    for (size_t id = 0; id <= stringPool.Size(); ++id) {
        stringPoolStrings.push_back(stringPool[id]);
    }
    Sort(stringPoolStrings);
    Sort(strings);

    EXPECT_EQ(stringPoolStrings, strings);
}

void CompareLabelsEq(
        const UniqueLabelsResponse& protoResp,
        const NStringPool::TStringPool& stringPool,
        TVector<TVector<TStringBuf>> labels)
{
    TVector<TVector<TStringBuf>> protoLabels;

    for (size_t labelId = 0; labelId < protoResp.labelsSize(); ++labelId) {
        auto& label = protoLabels.emplace_back();
        auto& protoLabel = protoResp.labels(labelId);
        for (int id = 0; id < protoLabel.labels_idx_size(); ++id) {
            label.push_back(stringPool[protoLabel.labels_idx(id)]);
        }
    }
    Sort(labels);
    Sort(protoLabels);

    EXPECT_EQ(labels, protoLabels);
}

TEST(TUniqueLabelsHandlerTest, ToProtoNonEmpty) {
    UniqueLabelsResponse protoResp;

    {
        NStringPool::TStringPoolBuilder strings;
        absl::flat_hash_set<TLabels, THash<TLabels>> labels;
        labels.emplace(TLabels{
            {
                strings.Put("host"),
                strings.Put("memstore-sas-00"),
            },
            {
                strings.Put("sensor"),
                strings.Put("mem_usage"),
            }
        });
        labels.emplace(TLabels{
            {
                strings.Put("host"),
                strings.Put("memstore-sas-01"),
            },
            {
                strings.Put("sensor"),
                strings.Put("cpu_usage"),
            }
        });

        NIndex::TShardEvents::TUniqueLabelsResponse shardResp{
            strings.Build(),
            std::move(labels),
        };
        NApi::ToProto(shardResp, &protoResp);
    }

    ASSERT_TRUE(protoResp.has_string_pool());
    NStringPool::TStringPool strings{protoResp.string_pool()};
    ASSERT_EQ(strings.Size(), 6u);
    CompareStringPoolEq(
            strings,
            {
                "",
                "host",
                "memstore-sas-00",
                "sensor",
                "mem_usage",
                "memstore-sas-01",
                "cpu_usage"
            });

    ASSERT_EQ(protoResp.labels_size(), 2);

    CompareLabelsEq(
            protoResp,
            strings,
            {
                {
                    "host",
                    "memstore-sas-00",
                    "sensor",
                    "mem_usage"
                },
                {
                    "host",
                    "memstore-sas-01",
                    "sensor",
                    "cpu_usage"
                }
            });
}
