#include "config.h"

#include <solomon/libs/cpp/config/units.h>
#include <solomon/libs/cpp/config_includes/config_includes.h>
#include <solomon/libs/cpp/yasm/constants/interval.h>

#include <util/generic/size_literals.h>
#include <util/system/hostname.h>

namespace NSolomon::NMemStore::NConfig {

void VerifyMemStoreConfig(TMemStoreConfig& config) {
    // TODO: verify actor system config and grpc server config

    // YDB client

//    auto& ydbClientConfig = *config.mutable_ydb_client();

//    if (ydbClientConfig.solomon_volume_path().empty()) {
//        ythrow TConfigValidationException() << "ydb_client.solomon_volume_path is required";
//    }
//
//    if (ydbClientConfig.request_timeout().value() == 0) {
//        ydbClientConfig.mutable_request_timeout()->set_value(10000);
//        ydbClientConfig.mutable_request_timeout()->set_unit(yandex::solomon::config::MILLISECONDS);
//    }
//
//    if (ydbClientConfig.reconnect_time_min().value() == 0) {
//        ydbClientConfig.mutable_reconnect_time_min()->set_value(200);
//        ydbClientConfig.mutable_reconnect_time_min()->set_unit(yandex::solomon::config::MILLISECONDS);
//    }
//
//    if (ydbClientConfig.reconnect_time_max().value() == 0) {
//        ydbClientConfig.mutable_reconnect_time_max()->set_value(5000);
//        ydbClientConfig.mutable_reconnect_time_max()->set_unit(yandex::solomon::config::MILLISECONDS);
//    }
//
//    if (ydbClientConfig.max_retries() == 0) {
//        ydbClientConfig.set_max_retries(5);
//    }
//
//    if (ydbClientConfig.retry_backoff_time().value() == 0) {
//        ydbClientConfig.mutable_retry_backoff_time()->set_value(100);
//        ydbClientConfig.mutable_retry_backoff_time()->set_unit(yandex::solomon::config::MILLISECONDS);
//    }
//
//    if (ydbClientConfig.retry_backoff_time_max().value() == 0) {
//        ydbClientConfig.mutable_retry_backoff_time_max()->set_value(5000);
//        ydbClientConfig.mutable_retry_backoff_time_max()->set_unit(yandex::solomon::config::MILLISECONDS);
//    }
//
//    if (ydbClientConfig.retry_backoff_factor() == 0) {
//        ydbClientConfig.set_retry_backoff_factor(1.5);
//    } else if (ydbClientConfig.retry_backoff_factor() < 1) {
//        ythrow TConfigValidationException() << "ydb_client.retry_backoff_factor should be no less than 1";
//    }
//
//    if (ydbClientConfig.host_ping_interval().value() == 0) {
//        ydbClientConfig.mutable_host_ping_interval()->set_value(10000);
//        ydbClientConfig.mutable_host_ping_interval()->set_unit(yandex::solomon::config::MILLISECONDS);
//    }
//
//    if (ydbClientConfig.host_down_ping_interval().value() == 0) {
//        ydbClientConfig.mutable_host_down_ping_interval()->set_value(3000);
//        ydbClientConfig.mutable_host_down_ping_interval()->set_unit(yandex::solomon::config::MILLISECONDS);
//    }

    // Write queue

    auto& writeQueueConfig = *config.mutable_write_queue();

    if (writeQueueConfig.max_queue_size().value() == 0) {
        writeQueueConfig.mutable_max_queue_size()->set_value(50);
        writeQueueConfig.mutable_max_queue_size()->set_unit(yandex::solomon::config::MEGABYTES);
    }

    if (writeQueueConfig.max_global_queue_size().value() == 0) {
        writeQueueConfig.mutable_max_global_queue_size()->set_value(100);
        writeQueueConfig.mutable_max_global_queue_size()->set_unit(yandex::solomon::config::MEGABYTES);
    }

    if (writeQueueConfig.batch_size().value() == 0) {
        writeQueueConfig.mutable_batch_size()->set_value(20);
        writeQueueConfig.mutable_batch_size()->set_unit(yandex::solomon::config::MEGABYTES);
    }

    if (writeQueueConfig.write_timeout().value() == 0) {
        writeQueueConfig.mutable_write_timeout()->set_value(8000);
        writeQueueConfig.mutable_write_timeout()->set_unit(yandex::solomon::config::MILLISECONDS);
    }
//    if (FromProtoTime(writeQueueConfig.write_timeout()) >= FromProtoTime(ydbClientConfig.request_timeout())) {
//        ythrow TConfigValidationException() << "write_queue.write_timeout should be less than ydb_client.request_timeout";
//    }

    if (writeQueueConfig.write_backoff().value() == 0) {
        writeQueueConfig.mutable_write_backoff()->set_value(300);
        writeQueueConfig.mutable_write_backoff()->set_unit(yandex::solomon::config::MILLISECONDS);
    }

    // Index

    auto& indexConfig = *config.mutable_index();

    constexpr ui32 chunkPeriodMinutes = 5;
    if (indexConfig.chunk_length().value() == 0) {
        indexConfig.mutable_chunk_length()->set_value(chunkPeriodMinutes);
        indexConfig.mutable_chunk_length()->set_unit(yandex::solomon::config::MINUTES);
    }

    static_assert(NSolomon::NYasm::MEMSTORE_IN_MEMORY_PERIOD.Minutes() % chunkPeriodMinutes == 0);
    if (indexConfig.index_capacity().value() == 0) {
        indexConfig.mutable_index_capacity()->set_value(NSolomon::NYasm::MEMSTORE_IN_MEMORY_PERIOD.Minutes());
        indexConfig.mutable_index_capacity()->set_unit(yandex::solomon::config::MINUTES);
    }

    if (indexConfig.subshard_count() == 0) {
        indexConfig.set_subshard_count(64);
    }
}

TMemStoreConfig Load(const TString& filePath) {
    TMemStoreConfig config = ParseTextProto(filePath);
    try {
        NSolomon::MergeIncludes(config);
    } catch (...) {
        throw TConfigParseException() << CurrentExceptionMessage() << Endl;
    }
    VerifyMemStoreConfig(config);
    return config;
}

TMemStoreConfig ParseTextProto(const TString& filePath) {
    TMemStoreConfig config;
    static EParseFromTextFormatOptions options; // no flags -- all bits set to 0
    try {
        ParseFromTextFormat(filePath, config, options);
    } catch (...) {
        // using options above, we should catch an error if a config has unknown fields
        throw TConfigParseException() << CurrentExceptionMessage() << Endl;
    }
    return config;
}

void Validate(const TMemStoreConfig& config) {
    auto configCopy = config; // TODO: do not copy config
    VerifyMemStoreConfig(configCopy);
}

TMemStoreBenchmarkConfig LoadBenchmarkConfig(const TString& filePath) {
    TMemStoreBenchmarkConfig config;

    static auto options = EParseFromTextFormatOption::AllowUnknownField;
    try {
        ParseFromTextFormat(filePath, config, options);
        NSolomon::MergeIncludes(config);
    } catch (...) {
        throw TConfigParseException() << CurrentExceptionMessage() << Endl;
    }

    VerifyMemStoreConfig(*config.mutable_memstore_config());

    if (config.concurrency() == 0) {
        config.set_concurrency(1);
    }

    if (config.delay().value() == 0) {
        config.mutable_delay()->set_value(1);
        config.mutable_delay()->set_unit(yandex::solomon::config::SECONDS);
    }

    if (config.write_request_weight() == 0) {
        config.set_write_request_weight(10'000);
    }

    if (config.increase_delay() == 0.) {
        config.set_increase_delay(1.);
    }

    return config;
}

TNodeId GetNodeId(const yandex::monitoring::config::StaticClusterMapping& config) {
    const auto& localHostName = FQDNHostName();

    std::optional<TNodeId> localId;
    for (const auto& node: config.nodes()) {
        if (node.host() == localHostName) {
            return node.node_id();
        } else if (node.host() == TStringBuf{"localhost"}) {
            // localhost has lower priority, but remembered as fallback
            localId = node.node_id();
        }
    }

    if (localId.has_value()) {
        return *localId;
    }

    ythrow TConfigValidationException() << "host " << localHostName << " not found in the cluster map";
}

} // namespace NSolomon::NMemStore::NConfig
