#include "index.h"

#include <solomon/services/memstore/lib/index/subshard.h>

#include <solomon/libs/cpp/logging/logging.h>
#include <solomon/libs/cpp/search/lsm_tree.h>
#include <solomon/libs/cpp/search/bitmap_index.h>
#include <solomon/libs/cpp/steady_timer/steady_timer.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/hfunc.h>

#include <util/generic/deque.h>

using namespace NActors;

namespace NSolomon::NMemStore {
namespace {

struct TTreeAndSize {
    TIntrusivePtr<NSolomon::NSearch::IStorage> Tree;
    size_t Size;
};

constexpr auto FlushDelay = TDuration::Seconds(10);

class TFtsIndex final: public TActorBootstrapped<TFtsIndex> {
public:
    TFtsIndex(
            ui32 numOfSubShards,
            NMonitoring::IIntGauge* memoryFts,
            NMonitoring::IIntGauge* memoryFtsAdd,
            std::shared_ptr<IResourceUsageContext> shardMeteringContext) noexcept
        : NumOfSubShards_{numOfSubShards}
        , ShardMeteringContext_{std::move(shardMeteringContext)}
        , MemoryFts_(memoryFts)
        , MemoryFtsAdd_(memoryFtsAdd)
    {
    }

    TIntrusivePtr<NSolomon::NSearch::IStorage> NewTree() {
        return NSolomon::NSearch::CreateLsmTree(NSolomon::NSearch::TLsmLimits{});
    }

    void Bootstrap() {
        Schedule(FlushDelay, new TEvents::TEvWakeup);
        Become(&TThis::StateFunc);
    }

    STATEFN(StateFunc) {
        Timer_.Reset();

        Y_DEFER {
            if (ShardMeteringContext_) {
                ShardMeteringContext_->AddCpuTime(Timer_.Step());
            }
        };

        switch (ev->GetTypeRewrite()) {
            hFunc(TFtsIndexEvents::TAdd, OnAdd);
            hFunc(TFtsIndexEvents::TFindReq, OnFind);
            hFunc(TFtsIndexEvents::TReadReq, OnRead);
            hFunc(TFtsIndexEvents::TAddFrame, OnAddFrame);
            hFunc(TFtsIndexEvents::TDropFrame, OnDropFrame);
            sFunc(TEvents::TEvWakeup, OnWakeup);
            hFunc(TEvents::TEvPoison, OnPoison);
        }
    }

private:
    void OnAdd(const TFtsIndexEvents::TAdd::TPtr& ev) {
        auto* req = ev->Get();
        auto& lsm = GetLsm(req->FrameIdx);

        // TODO: switch to NSolomon::NLabels::TLabels in libs/search calls
        TVector<std::pair<NMonitoring::ILabelsPtr, ui32>> toAdd;
        for (const auto& [key, metricId]: req->Metrics) {
            NMonitoring::ILabelsPtr labels = MakeIntrusive<NMonitoring::TLabels>();
            for (auto kv = key.Begin(); kv != key.End(); ++kv) {
                labels->Add(kv->first, kv->second);
            }
            toAdd.emplace_back(std::move(labels), metricId.Pack());
        }
        MON_TRACE(Fts, "OnAdd frameId: " << req->FrameIdx << "; metric count to add/new: " << req->Metrics.size() << "/" << toAdd.size());
        lsm.Tree->Add(toAdd);
        lsm.Size += toAdd.size();
        if (lsm.Size > FlushLsmThreshold) {
            FlushLsm(lsm);
        }

        Send(req->ReplyTo, MakeHolder<NSolomon::NMemStore::NIndex::TSubShardEvents::TWriteToFrameDone>(req->Range), 0, req->ReplyCookie);

        if (MemoryFtsAdd_) {
            MemoryFtsAdd_->Add(-static_cast<i64>(req->AllocatedSize));
        }
    }

    void OnFind(TFtsIndexEvents::TFindReq::TPtr ev) {
        auto* req = ev->Get();
        MON_TRACE(Fts, "TFindReq sender = " << ev->Sender << " Selector size:" << req->Selectors.size());
        auto limit = req->Limit != 0 ? req->Limit : Max<ui32>();
        auto [labelIdxs, totalSize] = GetLabelIdxs(req->Selectors, limit);
        auto response = new TFtsIndexEvents::TFindResp(std::move(labelIdxs));
        response->TotalCount = totalSize;

        MON_TRACE(Fts, "TFindReq totalSize = " << response->TotalCount);
        Send(ev->Sender, response, 0, ev->Cookie);
    }

    void OnRead(const TFtsIndexEvents::TReadReq::TPtr& ev) {
        auto* req = ev->Get();
        auto [labelIdxs, _] = GetLabelIdxs(req->Selectors, req->FramesOnWindow, req->Limit);

        auto response = new TFtsIndexEvents::TReadResp(std::move(labelIdxs));

        Send(ev->Sender, response, 0, ev->Cookie);
    }

    void OnAddFrame(TFtsIndexEvents::TAddFrame::TPtr ev) {
        MON_DEBUG(Fts, "Create new frame #" << ev->Get()->Frame);
        Y_VERIFY(ev->Get()->Frame == FirstFrame_ + Lsm_.size());
        Lsm_.push_back({NewTree(), 0});
    }

    void OnDropFrame(const TFtsIndexEvents::TDropFrame::TPtr& ev) {
        auto frameIdx = ev->Get()->Frame;
        MON_DEBUG(Fts, "Drop frame #" << frameIdx);
        Y_VERIFY(frameIdx == FirstFrame_, "frame mismatch on drop");
        Lsm_.pop_front();
        FirstFrame_++;
        // TODO: clean if we have enough empty metrics and notify Shard if subshard is empty
    }

    void OnWakeup() {
        Schedule(FlushDelay, new TEvents::TEvWakeup);
        for (size_t lsmId = 0; lsmId < Lsm_.size(); ++lsmId) {
            if (Lsm_[lsmId].Size > 0) {
                Lsm_[lsmId].Tree->Flush();
            }
        }
        ReportSizeBytes(false);
    }

    void OnPoison(TEvents::TEvPoison::TPtr& ev) {
        Send(ev->Sender, new TEvents::TEvPoisonTaken);
        ReportSizeBytes(true);
        PassAway();
    }

    TTreeAndSize& GetLsm(TFrameIdx idx) {
        Y_VERIFY(idx >= FirstFrame_);
        Y_VERIFY(idx - FirstFrame_ < Lsm_.size());
        return Lsm_[idx - FirstFrame_];
    }

    void FlushLsm(TTreeAndSize& lsm) {
        lsm.Tree->Flush();
        if (lsm.Tree->NeedsOptimize()) {
            lsm.Tree->Optimize();
        }
        lsm.Size = 0;
    }

    struct LabelsAndSize {
        TVector<TVector<ui32>> Labels;
        size_t TotalSize;
    };

    LabelsAndSize GetLabelIdxs(const TSelectors& selectors, ui32 limit = Max<ui32>()) {
        TVector<TFrameIdx> allFrames;
        for (size_t id = 0; id < Lsm_.size(); ++ id) {
            allFrames.push_back(FirstFrame_ + id);
        }
        return GetLabelIdxs(selectors, allFrames, limit);
    }

    LabelsAndSize GetLabelIdxs(
            const TSelectors& selectors,
            const std::vector<TFrameIdx>& frames,
            ui32 limit = Max<ui32>())
    {
        TVector<NSearch::ISearchResult::TPtr> results;
        TVector<TVector<ui32>> unpacked(NumOfSubShards_);
        // TODO: need to optimize totalSize calculation:
        for (auto frameId: frames) {
            FlushLsm(GetLsm(frameId));
            auto& [tree, _] = GetLsm(frameId);
            results.push_back(tree->Search(selectors, Max<ui32>()));
        }
        auto result = NSearch::CombineSegmentResults(results, Max<ui32>());
        size_t totalCount = result->Size();
        result = NSearch::Truncate(result, limit);
        TVector<ui32> packed = result->ToVector();
        for (ui32 id: packed) {
            auto&& tmp = TMetricId::Unpack(id);
            unpacked[tmp.ShardNo].push_back(tmp.TsId);
        }
        for (auto& subshard: unpacked) {
            SortUnique(subshard);
        }
        return {unpacked, totalCount};
    }

    size_t CalcLsmSize() const {
        size_t lsmSize = Lsm_.size() * sizeof(TTreeAndSize);
        for (const auto& lsm: Lsm_) {
            lsmSize += lsm.Tree->AllocatedBytes();
        }
        return lsmSize;
    }

    void ReportSizeBytes(bool onPoison) {
        if (MemoryFts_) {
            const size_t memSizeBytes = onPoison ? 0 : CalcLsmSize();
            MemoryFts_->Add(static_cast<i64>(memSizeBytes) - static_cast<i64>(MemoryFtsSize_));
            MemoryFtsSize_ = memSizeBytes;
        }
    }

private:
    TDeque<TTreeAndSize> Lsm_;
    TFrameIdx FirstFrame_ = 0;
    ui32 NumOfSubShards_;
    std::shared_ptr<IResourceUsageContext> ShardMeteringContext_;
    NMonitoring::IIntGauge* const MemoryFts_;
    NMonitoring::IIntGauge* const MemoryFtsAdd_;
    size_t MemoryFtsSize_{0};
    TSteadyTimer Timer_{};

    static constexpr size_t FlushLsmThreshold = 1024;
};

} // namespace

std::unique_ptr<IActor> FtsIndex(
        ui32 numOfSubShards,
        NMonitoring::IIntGauge* memoryFtsMetric,
        NMonitoring::IIntGauge* memoryFtsAddMetric,
        std::shared_ptr<IResourceUsageContext> shardMeteringContext)
{
    return std::make_unique<TFtsIndex>(numOfSubShards, memoryFtsMetric, memoryFtsAddMetric, std::move(shardMeteringContext));
};

} // namespace NSolomon::NMemStore
