#pragma once

#include "metric_id.h"

#include <solomon/services/memstore/lib/event_slots.h>
#include <solomon/services/memstore/lib/time_series/points_range.h>
#include <solomon/services/memstore/lib/types.h>

#include <solomon/libs/cpp/actors/events/events.h>
#include <solomon/libs/cpp/actors/fwd.h>
#include <solomon/libs/cpp/labels/labels.h>
#include <solomon/libs/cpp/metering/shard_metrics_repo.h>
#include <solomon/libs/cpp/selectors/selectors.h>

#include <library/cpp/actors/core/event_local.h>

namespace NSolomon::NMemStore {

class TFtsIndexEvents: private TEventSlot<EEventSpace::MemStore, ES_FTS_INDEX> {
    enum {
        Add = SpaceBegin,
        FindReq,
        FindResp,
        ReadReq,
        ReadResp,
        AddFrame,
        DropFrame,
        End,
    };
    static_assert(End < SpaceEnd, "too many event types");

public:
    struct TAdd: public NActors::TEventLocal<TAdd, Add> {
        TFrameIdx FrameIdx;
        TVector<std::pair<NLabels::TLabels, TMetricId>> Metrics;
        NActors::TActorId ReplyTo;
        ui64 ReplyCookie;
        TPointsRange Range;
        size_t AllocatedSize{0};

        explicit TAdd(TFrameIdx frameIdx, TVector<std::pair<NLabels::TLabels, TMetricId>> metrics = {}) noexcept
            : FrameIdx{frameIdx}
            , Metrics{std::move(metrics)}
        {
        }

        size_t SizeBytes() const {
            using namespace NSolomon::NLabels;
            size_t size = sizeof(TAdd) + Metrics.capacity() * sizeof(std::pair<NLabels::TLabels, TMetricId>);
            for (const auto& [labels, _]: Metrics) {
                size += labels.Size() * sizeof(TLabels::TPair);
                for (const auto& label: labels) {
                    size += strlen((char*)label.first) + strlen((char*)label.second);
                }
            }
            return size;
        }
    };

    struct TFindReq: public NActors::TEventLocal<TFindReq, FindReq> {
        // Should be sorted lexicographically by keys!
        TSelectors Selectors;
        ui32 Limit;
        //Should count all metrics matches Selector, even they are out of Limit
        // TotalCount = 0, if NeedTotalCount set to false
        bool NeedTotalCount = false;

        TFindReq(TSelectors selectors, ui32 limit) noexcept
            : Selectors{std::move(selectors)}
            , Limit{limit}
        {
        }
    };

    struct TFindResp: public NActors::TEventLocal<TFindResp, FindResp> {
        // subshardId -> Vector<TsId>
        TVector<TVector<ui32>> ResultIds;
        ui32 TotalCount = 0;

        explicit TFindResp(TVector<TVector<ui32>> resultIds = {}) noexcept
            : ResultIds{std::move(resultIds)}
        {
        }
    };

    struct TReadReq: public NActors::TEventLocal<TReadReq, ReadReq> {
        // Should be sorted lexicographically by keys!
        TSelectors Selectors;
        std::vector<TFrameIdx> FramesOnWindow;
        ui32 Limit;

        TReadReq(TSelectors selectors, ui32 limit, std::vector<TFrameIdx> framesOnWindow) noexcept
            : Selectors{std::move(selectors)}
            , FramesOnWindow{std::move(framesOnWindow)}
            , Limit{limit}
        {
        }
    };

    struct TReadResp: public NActors::TEventLocal<TReadResp, ReadResp> {
        // subshardId -> Vector<TsId>
        TVector<TVector<ui32>> MetricIdxs;

        explicit TReadResp(TVector<TVector<ui32>> metricIdxs) noexcept
            : MetricIdxs{std::move(metricIdxs)}
        {
        }
    };

    struct TAddFrame: public NActors::TEventLocal<TAddFrame, AddFrame> {
        TFrameIdx Frame;

        explicit TAddFrame(TFrameIdx frame) noexcept
            : Frame{frame}
        {
        }
    };

    struct TDropFrame: public NActors::TEventLocal<TDropFrame, DropFrame> {
        TFrameIdx Frame;

        explicit TDropFrame(TFrameIdx frame) noexcept
            : Frame{frame}
        {
        }
    };
};

/**
 * In-memory FTS index implementation for a single shard.
 */
std::unique_ptr<NActors::IActor> FtsIndex(
        ui32 numOfSubShards,
        NMonitoring::IIntGauge* memoryFtsMetric = nullptr,
        NMonitoring::IIntGauge* memoryFtsAddMetric = nullptr,
        std::shared_ptr<IResourceUsageContext> shardMeteringContext = nullptr);

} // namespace NSolomon::NMemStore
