#include <solomon/libs/cpp/actors/events/common.h>
#include <solomon/libs/cpp/actors/test_runtime/actor_runtime.h>

#include <solomon/services/memstore/lib/fts/index.h>
#include <solomon/services/memstore/lib/index/metrics.h>

#include <library/cpp/actors/core/actor.h>
#include <library/cpp/actors/core/actorsystem.h>
#include <library/cpp/testing/gtest/gtest.h>

#include <util/generic/hash.h>
#include <util/random/fast.h>
#include <util/random/easy.h>
#include <util/random/shuffle.h>

using namespace NSolomon;
using namespace NMemStore;
using namespace NMonitoring;
using namespace NActors;

class TIndexTest: public ::testing::Test {
protected:
    void SetUp() override {
        Test::SetUp();

        Runtime_ = NSolomon::TTestActorRuntime::CreateInited();
        Runtime_->SetLogPriority(NSolomon::Index, NActors::NLog::PRI_DEBUG);

        Fts_ = Runtime_->Register(FtsIndex(NumOfSubShards_).release());
        Runtime_->WaitForBootstrap();
    }

    void TearDown() override {
        Runtime_.Reset();

        Test::TearDown();
    }

protected:
    constexpr static ui32 NumOfSubShards_ = 64;
    constexpr static ui32 ShardNum_ = 10;

    THolder<NSolomon::TTestActorRuntime> Runtime_;
    TActorId Fts_;
};

TEST_F(TIndexTest, ReadWrite) {
    ui64 frameIdx = 0;
    Runtime_->Send(Fts_, MakeHolder<TFtsIndexEvents::TAddFrame>(frameIdx));
    NLabels::TLabels label = NLabels::TLabels::OwnedStorage({{"host", "xxa"}});
    auto event = new TFtsIndexEvents::TAdd{frameIdx};
    event->Metrics.emplace_back(label, NMemStore::TMetricId{ShardNum_, 1});
    Runtime_->Send(Fts_, THolder(event));

    TSelectors Selectors;

    auto apiActor = Runtime_->AllocateEdgeActor();
    Runtime_->Send(new IEventHandle(Fts_, apiActor, new TFtsIndexEvents::TFindReq{Selectors, 100}));

    auto response = Runtime_->GrabEdgeEvent<TFtsIndexEvents::TFindResp>(apiActor);

    EXPECT_EQ(response->Get()->TotalCount, 1u);
}

TEST_F(TIndexTest, DropOne) {
    ui64 frameIdx = 0;
    Runtime_->Send(Fts_, MakeHolder<TFtsIndexEvents::TAddFrame>(frameIdx));
    NLabels::TLabels label_xxa = NLabels::TLabels::OwnedStorage({{"host", "xxa"}});
    NLabels::TLabels label_xxb = NLabels::TLabels::OwnedStorage({{"host", "xxb"}});
    NLabels::TLabels label_xxc = NLabels::TLabels::OwnedStorage({{"host", "xxc"}});
    {
        auto event = new TFtsIndexEvents::TAdd{frameIdx};
        event->Metrics.emplace_back(label_xxa, NMemStore::TMetricId{1, 1});
        event->Metrics.emplace_back(label_xxb, NMemStore::TMetricId{1, 2});
        Runtime_->Send(Fts_, THolder(event));
    }

    TSelectors Selectors;

    auto apiActor = Runtime_->AllocateEdgeActor();
    {
        Runtime_->Send(new IEventHandle(Fts_, apiActor, new TFtsIndexEvents::TFindReq{Selectors, 100}));
        auto response = Runtime_->GrabEdgeEvent<TFtsIndexEvents::TFindResp>(apiActor);
        EXPECT_EQ(response->Get()->TotalCount, 2u);
    }

    ++frameIdx;

    Runtime_->Send(Fts_, MakeHolder<TFtsIndexEvents::TAddFrame>(frameIdx));
    {
        auto event = new TFtsIndexEvents::TAdd{frameIdx};
        event->Metrics.emplace_back(label_xxc, NMemStore::TMetricId{1, 3});
        event->Metrics.emplace_back(label_xxb, NMemStore::TMetricId{1, 2});
        Runtime_->Send(Fts_, THolder(event));
    }
    {
        Runtime_->Send(new IEventHandle(Fts_, apiActor, new TFtsIndexEvents::TFindReq{Selectors, 100}));
        auto response = Runtime_->GrabEdgeEvent<TFtsIndexEvents::TFindResp>(apiActor);
        EXPECT_EQ(response->Get()->TotalCount, 3u);
    }

    Runtime_->Send(Fts_, MakeHolder<TFtsIndexEvents::TDropFrame>(0));
    {
        Runtime_->Send(new IEventHandle(Fts_, apiActor, new TFtsIndexEvents::TFindReq{Selectors, 100}));
        auto response = Runtime_->GrabEdgeEvent<TFtsIndexEvents::TFindResp>(apiActor);
        EXPECT_EQ(response->Get()->TotalCount, 2u);
    }
}

TEST_F(TIndexTest, ReadWriteWithDrop) {
    constexpr ui32 numOfMetrics = 1000;

    for (ui64 frameIdx = 0; frameIdx < 100u; ++ frameIdx) {
        Runtime_->Send(Fts_, MakeHolder<TFtsIndexEvents::TAddFrame>(frameIdx));

        for (ui32 hostId = 0; hostId < numOfMetrics; ++hostId) {
            NLabels::TLabels label = NLabels::TLabels::OwnedStrings({{"host", ToString(frameIdx * numOfMetrics + hostId).c_str()}});
            auto event = new TFtsIndexEvents::TAdd{frameIdx};
            event->Metrics.emplace_back(label, NMemStore::TMetricId{ShardNum_, static_cast<ui32>(frameIdx * numOfMetrics + hostId)});
            Runtime_->Send(Fts_, THolder(event));
        }
        TSelectors Selectors;
        auto apiActor = Runtime_->AllocateEdgeActor();
        Runtime_->Send(new IEventHandle(Fts_, apiActor, new TFtsIndexEvents::TFindReq{Selectors, 100}));

        auto response = Runtime_->GrabEdgeEvent<TFtsIndexEvents::TFindResp>(apiActor);

        if (frameIdx >= 6) {
            Runtime_->Send(Fts_, MakeHolder<TFtsIndexEvents::TDropFrame>(frameIdx - 6u));
        }
        EXPECT_EQ(response->Get()->TotalCount, Min(frameIdx + 1, static_cast<ui64>(7)) * numOfMetrics);
    }
}
