#include "add_point.h"
#include "metrics.h"
#include "shard.h"
#include "subshard.h"

#include <solomon/libs/cpp/logging/logging.h>
#include <solomon/libs/cpp/steady_timer/steady_timer.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/hfunc.h>

using namespace NActors;

namespace NSolomon::NMemStore::NIndex {
namespace {

class TAddPointActor: public TActorBootstrapped<TAddPointActor> {
public:
    TAddPointActor(
            TActorId shard,
            TVector<TActorId> subShards,
            TFrameIdx frame,
            TVector<TAddPointRequest> requests,
            std::shared_ptr<TMetrics> metrics,
            TActorId indexLimiterId,
            TActorId subscriber = {},
            std::shared_ptr<IResourceUsageContext> shardMeteringContext = nullptr)
        : Shard_{shard}
        , SubShards_(std::move(subShards))
        , Frame_{frame}
        , Requests_(std::move(requests))
        , Subscriber_(subscriber)
        , ShardMeteringContext_(std::move(shardMeteringContext))
        , Metrics_{std::move(metrics)}
        , IndexLimiterId_{indexLimiterId}
    {
        Metrics_->WriteInflight->Inc();
    }

    void Bootstrap() {
        Timer_.Reset();

        Y_DEFER {
            if (ShardMeteringContext_) {
                ShardMeteringContext_->AddCpuTime(Timer_.Step());
            }
        };

        THashMap<size_t, TVector<NSLog::TTsParser>> dataParsers;
        ReqSize_.reserve(Requests_.size());
        size_t requestsDataSize = 0;

        for (auto& request: Requests_) {
            ReqSize_.emplace_back(request.SizeBytes());
            try {
                requestsDataSize += request.Meta.size() + request.Data.size();
                NSLog::TParser parser{std::move(request.Meta), std::move(request.Data)};
                ParsersSizeBytes_ += parser.GetStorageSizeBytes();
                while (parser.HasNext()) {
                    auto dataParser = parser.Next();
                    ParsersSizeBytes_ += dataParser.SizeBytes();
                    dataParsers[dataParser.GetLabels().Hash() % SubShards_.size()].push_back(std::move(dataParser));
                }
            } catch (NSLog::TParsingError& e) {
                MON_WARN(Index, "point dropped: " << e.what());
                Metrics_->WriteErrors->Inc();
                Metrics_->WriteErrorsParsing->Inc();
            }
        }
        ParsersSizeBytes_ += SizeBytes();
        Metrics_->MemoryParsers->Add(static_cast<i64>(ParsersSizeBytes_));
        if (requestsDataSize != 0) {
            Metrics_->DataExpansion->Record(static_cast<double>(ParsersSizeBytes_) / requestsDataSize);
        }

        TIndexLimiterData limiterData;
        limiterData.AddParsersSizeBytes(static_cast<i64>(ParsersSizeBytes_));
        Send(IndexLimiterId_, new TIndexLimiterEvents::TAddParameterValues(
                limiterData.DetachDelta(),
                /* needReply: */false));

        for (auto& [subShardIdx, request]: dataParsers) {
            Send(SubShards_[subShardIdx], MakeHolder<TSubShardEvents::TWriteToFrame>(Frame_, std::move(request)));
        }

        Inflight_ = dataParsers.size();

        if (!Inflight_) {
            Done();
        }

        Become(&TThis::StateFunc);
    }

    STATEFN(StateFunc) {
        Timer_.Reset();

        Y_DEFER {
            if (ShardMeteringContext_) {
                ShardMeteringContext_->AddCpuTime(Timer_.Step());
            }
        };

        switch (ev->GetTypeRewrite()) {
            hFunc(TSubShardEvents::TWriteToFrameDone, OnWriteToFrameDone)
        }
    }

private:
    void OnWriteToFrameDone(TSubShardEvents::TWriteToFrameDone::TPtr& ev) {
        Range_.Update(ev->Get()->Range);
        if (--Inflight_ == 0) {
            Done();
        }
    }

    void Done() {
        Send(Shard_, MakeHolder<TShardEvents::TAddPointDone>(Frame_, Range_));

        ui64 sizeTotal = 0;
        for (size_t i = 0; i < Requests_.size(); ++i) {
            Send(Requests_[i].ReplyTo, new TShardEvents::TIndexDone{ReqSize_[i]}, 0, Requests_[i].Cookie);
            sizeTotal += ReqSize_[i];
        }

        Send(Subscriber_, MakeHolder<TShardEvents::TIndexDone>(sizeTotal));

        Metrics_->WriteInflight->Dec();
        Metrics_->WriteComplete->Inc();
        Metrics_->MemoryParsers->Add(-static_cast<i64>(ParsersSizeBytes_));

        TIndexLimiterData limiterData;
        limiterData.AddParsersSizeBytes(-static_cast<i64>(ParsersSizeBytes_));
        Send(IndexLimiterId_, new TIndexLimiterEvents::TAddParameterValues(
                limiterData.DetachDelta(),
                /* needReply: */false));

        ParsersSizeBytes_ = 0;

        PassAway();
    }

    size_t SizeBytes() const {
        return sizeof(TAddPointActor)
                + SubShards_.capacity() * sizeof(TActorId)
                + Requests_.capacity() * sizeof(TAddPointRequest)
                + ReqSize_.capacity() * sizeof(ui64);
    }

private:
    TActorId Shard_;
    TVector<TActorId> SubShards_;
    TFrameIdx Frame_;
    TVector<TAddPointRequest> Requests_;
    TVector<ui64> ReqSize_;
    TPointsRange Range_;
    size_t Inflight_{0};
    TActorId Subscriber_;
    std::shared_ptr<IResourceUsageContext> ShardMeteringContext_;
    TSteadyTimer Timer_{};
    std::shared_ptr<TMetrics> Metrics_;
    size_t ParsersSizeBytes_{0};
    TActorId IndexLimiterId_;
};

} // namespace

std::unique_ptr<IActor> CreateAddPointActor(
        TActorId shard,
        TVector<TActorId> subShards,
        TFrameIdx frame,
        TVector<TAddPointRequest> requests,
        std::shared_ptr<TMetrics> metrics,
        TActorId indexLimiterId,
        TActorId subscriber,
        std::shared_ptr<IResourceUsageContext> shardMeteringContext)
{
    return std::make_unique<TAddPointActor>(
            shard,
            std::move(subShards),
            frame,
            std::move(requests),
            std::move(metrics),
            indexLimiterId,
            subscriber,
            std::move(shardMeteringContext));
}

} // NSolomon::NMemStore::NIndex
