#include "find.h"
#include "metrics.h"
#include "shard.h"
#include "subshard.h"

#include <solomon/services/memstore/lib/fts/index.h>
#include <solomon/services/memstore/lib/metric/metric.h>

#include <solomon/libs/cpp/steady_timer/steady_timer.h>
#include <solomon/libs/cpp/string_pool/string_pool.h>

#include <library/cpp/actors/core/actor.h>
#include <library/cpp/actors/core/hfunc.h>

using namespace NActors;

namespace NSolomon::NMemStore::NIndex {
namespace {

class TFind: public TActor<TFind> {
public:
    TFind(TVector<TActorId> subShards, TActorId fts, std::shared_ptr<IResourceUsageContext> shardMeteringContext, std::shared_ptr<TMetrics> metrics)
        : TActor<TFind>{&TThis::StateFunc}
        , SubShards_(std::move(subShards))
        , Fts_(fts)
        , ShardMeteringContext_(std::move(shardMeteringContext))
        , SelfMetrics_{std::move(metrics)}
    {
        Y_VERIFY(!SubShards_.empty(), "SubShards array cannot be empty!");
        SelfMetrics_->ReadInflight->Inc();
    }

    STATEFN(StateFunc) {
        Timer_.Reset();

        Y_DEFER {
            if (ShardMeteringContext_) {
                ShardMeteringContext_->AddCpuTime(Timer_.Step());
            }
        };

        switch (ev->GetTypeRewrite()) {
            hFunc(TShardEvents::TFind, OnFindReq);
            hFunc(TFtsIndexEvents::TFindResp, OnFtsResponse);
            hFunc(TSubShardEvents::TFindMetricsResponse, OnFindMetricsResponse)
        }
    }

private:
    void OnFindReq(TShardEvents::TFind::TPtr& ev) {
        Req_ = ev;
        Send(Fts_, new TFtsIndexEvents::TFindReq{Req_->Get()->Selectors, Req_->Get()->Limit});
    }

    void OnFtsResponse(TFtsIndexEvents::TFindResp::TPtr& ev) {
        auto& metricList = ev->Get()->ResultIds;
        Y_VERIFY(metricList.size() == SubShards_.size(),
                 "Unexpected num of sub shards from FTS index: %lu. Expected: %lu",
                 metricList.size(),
                 SubShards_.size());

        TotalCount_ = ev->Get()->TotalCount;

        auto& selectors = Req_->Get()->Selectors;
        for (size_t subShardId = 0; subShardId < SubShards_.size(); ++subShardId) {
            if (!metricList.empty() && metricList[subShardId].empty()) {
                continue;
            }

            RequestsInFlight_++;
            Send(SubShards_[subShardId], new TSubShardEvents::TFindMetrics{
                SelfId(),
                selectors,
                std::move(metricList[subShardId])});
        }

        if (RequestsInFlight_ == 0) {
            ReplyAndDie();
        }
    }

    void OnFindMetricsResponse(TSubShardEvents::TFindMetricsResponse::TPtr& ev) {
        auto limit = Req_->Get()->Limit;

        for (auto& [type, labels]: ev->Get()->Found) {
            if (limit > 0 && static_cast<ui32>(Metrics_.size()) >= limit) {
                break;
            }

            auto& m = Metrics_.emplace_back();
            m.Type = type;

            m.Labels.reserve(labels.Size());
            for (auto [k, v]: labels) {
                m.Labels.emplace_back(Pool_.Put(k), Pool_.Put(v));
            }
        }

        if (--RequestsInFlight_ == 0) {
            ReplyAndDie();
        }
    }

    void ReplyAndDie() {
        auto* event = new TShardEvents::TFindResponse{Pool_.Build(), std::move(Metrics_), TotalCount_};
        Send(Req_->Sender, event, 0, Req_->Cookie);

        SelfMetrics_->ReadInflight->Dec();
        SelfMetrics_->ReadComplete->Inc();

        PassAway();
    }

private:
    TVector<TActorId> SubShards_;
    TActorId Fts_;
    std::shared_ptr<IResourceUsageContext> ShardMeteringContext_;
    TSteadyTimer Timer_{};
    TShardEvents::TFind::TPtr Req_;
    NStringPool::TStringPoolBuilder Pool_;
    std::vector<TMetric> Metrics_;
    ui32 TotalCount_ = 0;
    ui32 RequestsInFlight_ = 0;
    std::shared_ptr<TMetrics> SelfMetrics_;
};

} // namespace

std::unique_ptr<IActor> CreateFindActor(
        TVector<TActorId> subShards,
        TActorId fts,
        std::shared_ptr<IResourceUsageContext> shardMeteringContext,
        std::shared_ptr<TMetrics> metrics)
{
    return std::make_unique<TFind>(std::move(subShards), fts, std::move(shardMeteringContext), std::move(metrics));
}

} // NSolomon::NMemStore::NIndex
