#include "index_limiter.h"
#include "memory_tools.h"

#include <solomon/libs/cpp/logging/logging.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/containers/absl_flat_hash/flat_hash_set.h>

#include <contrib/libs/tcmalloc/tcmalloc/malloc_extension.h>

using namespace NActors;

namespace NSolomon::NMemStore::NIndex {

constexpr size_t MAX_PARSERS_SIZE_BYTES = 12ull * 1024 * 1024 * 1024;
constexpr size_t MAX_TIME_SERIES_SIZE_BYTES = 40ull * 1024 * 1024 * 1024;
constexpr size_t MAX_INDEX_MESSAGE_COUNT = 10'000ull;
constexpr size_t MEMORY_LIMIT_LOCK = 47ull * 1024 * 1024 * 1024;
constexpr size_t MEMORY_LIMIT_UNLOCK = MEMORY_LIMIT_LOCK - 1;
static_assert(MEMORY_LIMIT_UNLOCK < MEMORY_LIMIT_LOCK);

TIndexLimiterConfig TIndexLimiterConfig::Default() {
    return TIndexLimiterConfig{
            .MaxTimeSeriesSize = MAX_TIME_SERIES_SIZE_BYTES,
            .MaxParsersSize = MAX_PARSERS_SIZE_BYTES,
            .MaxIndexMessageCount = MAX_INDEX_MESSAGE_COUNT
    };
}

namespace {

class TLimiter {
public:
    void AddValue(i64 delta) {
        Value_ += delta;
    }

    i64 GetValue() const {
        return Value_;
    }

    std::tuple<bool, i64> IsLocked(i64 upperLimit, i64 lowerLimit) {
        const i64 value = Value_;
        const bool isLocked = IsLocked_;
        const bool shouldBeLocked = value > upperLimit;
        const bool shouldBeUnlocked = value < lowerLimit;

        if (shouldBeLocked) {
            if (!isLocked) {
                IsLocked_ = true;
            }
            return {true, value};
        }

        if (shouldBeUnlocked) {
            if (isLocked) {
                IsLocked_ = false;
            }
            return {false, value};
        }

        return {isLocked, value};
    }

private:
    std::atomic<i64> Value_{0};
    std::atomic<bool> IsLocked_{false};
};

class TIndexWriteLimiter : public IIndexWriteLimiter {
public:
    TIndexWriteLimiter(TIndexLimiterConfig config, std::shared_ptr<NMonitoring::TMetricRegistry> registry)
        : Config_(std::move(config))
        , Registry_(std::move(registry))
    {
        if (Registry_) {
            IndexMessageQuotaMetric_ = Registry_->IntGauge({{"sensor", "index.limiter.quota.inflight_message.count"}});
            CanAddMessageOpenMetric_ = Registry_->Rate({
                    {"sensor", "index.limiter.can_add_message"},
                    {"state", "open"}});
            CanAddMessageCloseMetric_ = Registry_->Rate({
                    {"sensor", "index.limiter.can_add_message"},
                    {"state", "close"}});
        }
    }

    void AddIndexMessageCount(i64 delta) override {
        InflightMessagesLimiter_.AddValue(delta);
    }

    bool CanAddIndexMessage() override {
        const auto [isInflightMessagesLocked, inflightMessagesCount] = InflightMessagesLimiter_.IsLocked(
                static_cast<i64>(Config_.MaxIndexMessageCount),
                static_cast<i64>(Config_.MaxIndexMessageCount) / 2);

        const bool canAdd = !isInflightMessagesLocked;
        if (Registry_) {
            if (canAdd) {
                CanAddMessageOpenMetric_->Inc();
            } else {
                CanAddMessageCloseMetric_->Inc();
            }
            const i64 indexMessagesQuota = isInflightMessagesLocked ?
                    0 : static_cast<i64>(Config_.MaxIndexMessageCount) - inflightMessagesCount;
            IndexMessageQuotaMetric_->Set(indexMessagesQuota);
        }

        return canAdd;
    }

private:
    const TIndexLimiterConfig Config_;
    TLimiter InflightMessagesLimiter_;
    std::shared_ptr<NMonitoring::TMetricRegistry> Registry_;
    NMonitoring::IIntGauge* IndexMessageQuotaMetric_{nullptr};
    NMonitoring::IRate* CanAddMessageOpenMetric_{nullptr};
    NMonitoring::IRate* CanAddMessageCloseMetric_{nullptr};
};


/**
* ----------------------------------------------------------------------------------------------------------------------
*/

class TIndexLimiterActor: public TActorBootstrapped<TIndexLimiterActor> {
public:
    TIndexLimiterActor(TIndexLimiterConfig config, std::shared_ptr<NMonitoring::TMetricRegistry> registry)
        : Config_(std::move(config))
        , Registry_(std::move(registry))
    {
        if (Registry_) {
            ParsersQuotaMetric_ = Registry_->IntGauge({{"sensor", "index.limiter.quota.parsers.size_bytes"}});
            TimeSeriesQuotaMetric_ = Registry_->IntGauge({{"sensor", "index.limiter.quota.time_series.size_bytes"}});
            TimeSeriesOpenMetric_ = Registry_->Rate({{"sensor", "index.limiter.time_series"}, {"state", "open"}});
            TimeSeriesCloseMetric_ = Registry_->Rate({{"sensor", "index.limiter.time_series"}, {"state", "close"}});;
            ParsersOpenMetric_ = Registry_->Rate({{"sensor", "index.limiter.parsers"}, {"state", "open"}});
            ParsersCloseMetric_ = Registry_->Rate({{"sensor", "index.limiter.parsers"}, {"state", "close"}});
            MemoryOpenMetric_ = Registry_->Rate({{"sensor", "index.limiter.memory"}, {"state", "open"}});
            MemoryCloseMetric_ = Registry_->Rate({{"sensor", "index.limiter.memory"}, {"state", "close"}});
        }
    }

    void Bootstrap() {
        Schedule(WakeUpCycle_, new TEvents::TEvWakeup);
        Become(&TThis::StateFunc);
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            sFunc(TEvents::TEvWakeup, OnWakeUp)
            hFunc(TIndexLimiterEvents::TAddParameterValues, OnAddParameterValues)
            hFunc(TIndexLimiterEvents::TSubscribe, OnSubscribe)
            hFunc(TIndexLimiterEvents::TUnsubscribe, OnUnsubscribe)
        }
    }

    void OnAddParameterValues(TIndexLimiterEvents::TAddParameterValues::TPtr& ev) {
        Params_.AddDelta(ev->Get()->ParamsDelta);
        if (ev->Get()->NeedReply) {
            UpdateState();
            Send(ev->Sender, new TIndexLimiterEvents::TReportLimiterState{State_});
        }
    }

    void OnWakeUp() {
        UpdateState();

        if (!State_.IsEqual(PrevState_)) {
            for (const auto& actorId: Subscribers_) {
                Send(actorId, new TIndexLimiterEvents::TReportLimiterState{State_});
            }
            PrevState_ = State_;
        }
        Schedule(WakeUpCycle_, new TEvents::TEvWakeup);
    }

    void OnSubscribe(TIndexLimiterEvents::TSubscribe::TPtr& ev) {
        Subscribers_.insert(ev->Get()->SubscriberId);
    }

    void OnUnsubscribe(TIndexLimiterEvents::TUnsubscribe::TPtr& ev) {
        auto it = Subscribers_.find(ev->Get()->SubscriberId);
        if (it != Subscribers_.end()) {
            Subscribers_.erase(it);
        }
    }

private:
    void UpdateState() {
        MemoryCommitted_ = GetMemoryCommitted();
        const i64 maxTimeSeriesSize = CalcMaxTimeSeriesSize();
        State_.SetTimeSeriesSizeOverLimit(IsLocked(
                Params_.TimeSeriesSizeBytes,
                maxTimeSeriesSize,
                maxTimeSeriesSize / 2,
                State_.IsTimeSeriesSizeOverLimit()));

        State_.SetParsersSizeOverLimit(IsLocked(
                Params_.ParsersSizeBytes,
                static_cast<i64>(Config_.MaxParsersSize),
                static_cast<i64>(Config_.MaxParsersSize) / 2,
                State_.IsParsersSizeOverLimit()));

        State_.SetIndexMessageCountOverLimit(IsLocked(
                Params_.IndexMessageCount,
                static_cast<i64>(Config_.MaxIndexMessageCount),
                static_cast<i64>(Config_.MaxIndexMessageCount) / 2,
                State_.IsIndexMessageCountOverLimit()));

        if (MemoryCommitted_) {
            State_.SetMemoryExhausted(IsLocked(
                    MemoryCommitted_.value(),
                    static_cast<i64>(MEMORY_LIMIT_LOCK),
                    static_cast<i64>(MEMORY_LIMIT_UNLOCK),
                    State_.IsMemoryExhausted()));
        } else {
            State_.SetMemoryExhausted(false);
        }

        if (Registry_) {
            const i64 parsersQuota = State_.IsParsersSizeOverLimit() ? 0 : Config_.MaxParsersSize - Params_.ParsersSizeBytes;
            ParsersQuotaMetric_->Set(parsersQuota);
            const i64 timeSeriesQuota = State_.IsTimeSeriesSizeOverLimit() ? 0 : maxTimeSeriesSize - Params_.TimeSeriesSizeBytes;
            TimeSeriesQuotaMetric_->Set(timeSeriesQuota);

            if (State_.IsParsersSizeOverLimit()) {
                ParsersCloseMetric_->Inc();
            } else {
                ParsersOpenMetric_->Inc();
            }

            if (State_.IsTimeSeriesSizeOverLimit()) {
                TimeSeriesCloseMetric_->Inc();
            } else {
                TimeSeriesOpenMetric_->Inc();
            }

            if (State_.IsMemoryExhausted()) {
                MemoryCloseMetric_->Inc();
            } else {
                MemoryOpenMetric_->Inc();
            }
        }
    }

    i64 CalcMaxTimeSeriesSize() const {
        i64 maxTimeSeriesSize =
                static_cast<i64>(Config_.MaxTimeSeriesSize)
                - Params_.LabelsSizeBytes
                - Params_.ParsersSizeBytes;

        if (MemoryCommitted_) {
            i64 memoryShortage = Max((i64)0, MemoryCommitted_.value() - static_cast<i64>(MEMORY_LIMIT_LOCK));
            maxTimeSeriesSize -= memoryShortage;
        }
        return maxTimeSeriesSize;
    }

    static bool IsLocked(i64 value, i64 upperLimit, i64 lowerLimit, bool isLocked) {
        if (value > upperLimit) {
            return true;
        }
        if (value < lowerLimit) {
            return false;
        }
        return isLocked;
    }

private:
    const TIndexLimiterConfig Config_;
    absl::flat_hash_set<TActorId, THash<TActorId>> Subscribers_;
    TIndexLimiterParameters Params_;
    TIndexLimiterState State_;
    TIndexLimiterState PrevState_;
    std::optional<i64> MemoryCommitted_;
    TDuration WakeUpCycle_ = TDuration::Seconds(1);
    std::shared_ptr<NMonitoring::TMetricRegistry> Registry_;
    NMonitoring::IIntGauge* ParsersQuotaMetric_{nullptr};
    NMonitoring::IIntGauge* TimeSeriesQuotaMetric_{nullptr};
    NMonitoring::IRate* TimeSeriesOpenMetric_{nullptr};
    NMonitoring::IRate* TimeSeriesCloseMetric_{nullptr};
    NMonitoring::IRate* ParsersOpenMetric_{nullptr};
    NMonitoring::IRate* ParsersCloseMetric_{nullptr};
    NMonitoring::IRate* MemoryOpenMetric_{nullptr};
    NMonitoring::IRate* MemoryCloseMetric_{nullptr};
};

}

std::shared_ptr<IIndexWriteLimiter> CreateIndexWriteLimiter(
        TIndexLimiterConfig config,
        std::shared_ptr<NMonitoring::TMetricRegistry> registry)
{
    return std::make_shared<TIndexWriteLimiter>(std::move(config), std::move(registry));
}

std::unique_ptr<IActor> CreateIndexLimiter(
        TIndexLimiterConfig config,
        std::shared_ptr<NMonitoring::TMetricRegistry> registry) {
    return std::make_unique<TIndexLimiterActor>(std::move(config), std::move(registry));
}

}
