#pragma once

#include <memory>

#include <solomon/services/memstore/lib/event_slots.h>

#include <library/cpp/actors/core/actor.h>
#include <library/cpp/actors/core/event_local.h>

#include <util/system/types.h>

namespace NMonitoring {

class TMetricRegistry;

}

namespace NSolomon::NMemStore::NIndex {

class TSimpleLimiter {
public:

    TSimpleLimiter(i64 lockLimit, i64 unlockLimit)
        : LockLimit_(lockLimit)
        , UnlockLimit_(unlockLimit)
    {
        Y_VERIFY(unlockLimit < lockLimit);
    }

    void AddValue(i64 delta) {
        Value_ += delta;
        UpdateState();
    }

    bool IsLocked() const {
        return IsLocked_;
    }

private:
    void UpdateState() {
        if (Value_ > LockLimit_) {
            IsLocked_ = true;
        } else if (Value_ < UnlockLimit_) {
            IsLocked_ = false;
        }
    }

private:
    const i64 LockLimit_;
    const i64 UnlockLimit_;
    i64 Value_{0};
    bool IsLocked_{false};
};

struct TIndexLimiterConfig {
    /**
     * Max size in bytes of persistent time series data in memory.
     */
    size_t MaxTimeSeriesSize;
    /**
     * Max size in bytes of temporary parsers of data and metadata
     */
    size_t MaxParsersSize;

    /**
     * Max number of in flight messages TShardEvents::TIndex
     */
    size_t MaxIndexMessageCount;

    static TIndexLimiterConfig Default();
};

class IIndexWriteLimiter {
public:
    virtual ~IIndexWriteLimiter() = default;

    /**
     * Add or subtract the number of in flight messages TShardEvents::TIndex
     */
    virtual void AddIndexMessageCount(i64) = 0;

    /**
     * Is it possible to process at least one more TIndex message
     * @return
     */
    virtual bool CanAddIndexMessage() = 0;
};

std::shared_ptr<IIndexWriteLimiter> CreateIndexWriteLimiter(
        TIndexLimiterConfig config,
        std::shared_ptr<NMonitoring::TMetricRegistry> registry = {});


/**
 * ---------------------------------------------------------------------------------------------------------------------
 */

struct TIndexLimiterParameters {
    i64 ParsersSizeBytes{0};
    i64 TimeSeriesSizeBytes{0};
    i64 LabelsSizeBytes{0};
    i64 IndexMessageCount{0};

    void AddDelta(const TIndexLimiterParameters& delta) {
        ParsersSizeBytes += delta.ParsersSizeBytes;
        TimeSeriesSizeBytes += delta.TimeSeriesSizeBytes;
        LabelsSizeBytes += delta.LabelsSizeBytes;
        IndexMessageCount += delta.IndexMessageCount;
    }
};

class TIndexLimiterState {
public:
    bool IsParsersSizeOverLimit() const {
        return Flags_.test(ParsersSizeBytes);
    }
    void SetParsersSizeOverLimit(bool isOverLimit) {
        Flags_[ParsersSizeBytes] = isOverLimit;
    }

    bool IsTimeSeriesSizeOverLimit() const {
        return Flags_.test(TimeSeriesSizeBytes);
    }
    void SetTimeSeriesSizeOverLimit(bool isOverLimit) {
        Flags_[TimeSeriesSizeBytes] = isOverLimit;
    }

    bool IsIndexMessageCountOverLimit() const {
        return Flags_.test(IndexMessageCount);
    }
    void SetIndexMessageCountOverLimit(bool isOverLimit) {
        Flags_[IndexMessageCount] = isOverLimit;
    }

    bool IsMemoryExhausted() const {
        return Flags_.test(MemoryExhausted);
    }
    void SetMemoryExhausted(bool isExhausted) {
        Flags_[MemoryExhausted] = isExhausted;
    }

    bool IsEqual(const TIndexLimiterState& other) const {
        return Flags_ == other.Flags_;
    }

private:
    enum {
        ParsersSizeBytes = 0,
        TimeSeriesSizeBytes,
        IndexMessageCount,
        MemoryExhausted,
        ParametersCount
    };

private:
    std::bitset<ParametersCount> Flags_;
};

struct TIndexLimiterEvents : private TEventSlot<EEventSpace::MemStore, ES_INDEX_WRITE_LIMITER> {
    enum {
        AddParameterValues = SpaceBegin,
        ReportLimiterState,
        Subscribe,
        Unsubscribe,
        End
    };
    static_assert(End < SpaceEnd, "too many event types");

    struct TAddParameterValues: public NActors::TEventLocal<TAddParameterValues, AddParameterValues> {
        TIndexLimiterParameters ParamsDelta;
        bool NeedReply;

        explicit TAddParameterValues(TIndexLimiterParameters delta, bool needReply = true)
            : ParamsDelta(std::move(delta))
            , NeedReply(needReply)
        {
        }
    };

    struct TReportLimiterState: public NActors::TEventLocal<TReportLimiterState, ReportLimiterState> {
        TIndexLimiterState State;

        explicit TReportLimiterState(const TIndexLimiterState& state)
            : State(state)
        {
        }
    };

    struct TSubscribe: public NActors::TEventLocal<TSubscribe, Subscribe> {
        NActors::TActorId SubscriberId;

        explicit TSubscribe(NActors::TActorId subscriberId)
            : SubscriberId(subscriberId)
        {
        }
    };

    struct TUnsubscribe: public NActors::TEventLocal<TUnsubscribe, Unsubscribe> {
        NActors::TActorId SubscriberId;

        explicit TUnsubscribe(NActors::TActorId subscriberId)
            : SubscriberId(subscriberId)
        {
        }
    };
};

class TIndexLimiterData {
public:

    void AddParsersSizeBytes(i64 delta) {
        AccDelta_.ParsersSizeBytes += delta;
    }

    void AddTimeSeriesSizeBytes(i64 delta) {
        AccDelta_.TimeSeriesSizeBytes += delta;
    }

    void AddLabelsSizeBytes(i64 delta) {
        AccDelta_.LabelsSizeBytes += delta;
    }

    void AddIndexMessageCount(i64 delta) {
        AccDelta_.IndexMessageCount += delta;
    }
    void AddDelta(const TIndexLimiterParameters& delta) {
        AccDelta_.AddDelta(delta);
    }

    const TIndexLimiterParameters& GetDelta() const {
        return AccDelta_;
    }

    TIndexLimiterParameters DetachDelta() {
        TIndexLimiterParameters delta{AccDelta_};
        AccDelta_ = TIndexLimiterParameters{};
        return delta;
    }

    TIndexLimiterState GetState() const {
        return State_;
    }

    void SetState(TIndexLimiterState state) {
        State_ = std::move(state);
    }

private:
    // Accumulated parameters delta
    TIndexLimiterParameters AccDelta_;
    TIndexLimiterState State_;
};

std::unique_ptr<NActors::IActor> CreateIndexLimiter(
        TIndexLimiterConfig config,
        std::shared_ptr<NMonitoring::TMetricRegistry> registry = {});

}
