#include "label_keys.h"
#include "metrics.h"
#include "shard.h"
#include "subshard.h"

#include <solomon/libs/cpp/steady_timer/steady_timer.h>

#include <library/cpp/actors/core/actor.h>
#include <library/cpp/actors/core/hfunc.h>

using namespace NActors;

namespace NSolomon::NMemStore::NIndex {
namespace {

class TLabelKeys: public TActor<TLabelKeys> {
public:
    TLabelKeys(TVector<TActorId> subShards, TActorId fts, std::shared_ptr<IResourceUsageContext> shardMeteringContext, std::shared_ptr<TMetrics> metrics)
        : TActor<TLabelKeys>{&TThis::StateFunc}
        , SubShards_{std::move(subShards)}
        , Fts_{fts}
        , ShardMeteringContext_{std::move(shardMeteringContext)}
        , Metrics_(std::move(metrics))
    {
        Y_VERIFY(!SubShards_.empty());
        Metrics_->ReadInflight->Inc();
    }

    STATEFN(StateFunc) {
        Timer_.Reset();

        Y_DEFER {
            if (ShardMeteringContext_) {
                ShardMeteringContext_->AddCpuTime(Timer_.Step());
            }
        };

        switch (ev->GetTypeRewrite()) {
            hFunc(TShardEvents::TLabelKeys, OnLabelKeyRequest)
            hFunc(TFtsIndexEvents::TFindResp, OnFtsResponse)
            hFunc(TSubShardEvents::TFindMetricsResponse, OnFindMetricsResponse)
        }
    }

private:
    void OnLabelKeyRequest(TShardEvents::TLabelKeys::TPtr& ev) {
        Req_ = ev;
        Send(Fts_, new TFtsIndexEvents::TFindReq{Req_->Get()->Selectors, Max<ui32>()});
    }

    void OnFtsResponse(TFtsIndexEvents::TFindResp::TPtr& ev) {
        auto& res = ev->Get()->ResultIds;
        Y_VERIFY(res.size() == SubShards_.size(),
                 "Unexpected num of subshards from FTS index: %lu. Expected: %lu",
                 res.size(),
                 SubShards_.size());

        auto& selectors = Req_->Get()->Selectors;
        for (ui32 subShardId = 0; subShardId < SubShards_.size(); ++subShardId) {
            if (auto& metricId = res[subShardId]; !metricId.empty()) {
                auto& subShard = SubShards_[subShardId];
                Send(subShard, new TSubShardEvents::TFindMetrics{SelfId(), selectors, std::move(metricId)});
                RequestsInFlight_++;
            }
        }

        if (RequestsInFlight_ == 0) {
            ReplyAndDie();
        }
    }

    void OnFindMetricsResponse(TSubShardEvents::TFindMetricsResponse::TPtr& ev) {
        for (auto& [type, labels]: ev->Get()->Found) {
            for (auto [k, v]: labels) {
                // string pool adds new strings in order, so the cheapest way
                // to collect only unique keys would be to add only those ids
                // which are greater than the last one.
                ui32 idx = Pool_.Put(k);
                if (Keys_.empty() || Keys_.back() < idx) {
                    Keys_.push_back(idx);
                }
            }
        }

        if (--RequestsInFlight_ == 0) {
            ReplyAndDie();
        }
    }

    void ReplyAndDie() {
        auto* resp = new TShardEvents::TLabelKeysResponse{Pool_.Build(), std::move(Keys_)};
        Send(Req_->Sender, resp, 0, Req_->Cookie);

        Metrics_->ReadInflight->Dec();
        Metrics_->ReadComplete->Inc();

        PassAway();
    }

private:
    const TVector<TActorId> SubShards_;
    const TActorId Fts_;
    std::shared_ptr<IResourceUsageContext> ShardMeteringContext_;
    std::shared_ptr<TMetrics> Metrics_;
    TSteadyTimer Timer_{};
    TShardEvents::TLabelKeys::TPtr Req_;
    NStringPool::TStringPoolBuilder Pool_;
    std::vector<ui32> Keys_;
    ui32 RequestsInFlight_{0};
};

} // namespace

std::unique_ptr<IActor> CreateLabelKeysActor(
        TVector<TActorId> subShards,
        TActorId fts,
        std::shared_ptr<IResourceUsageContext> shardMeteringContext,
        std::shared_ptr<TMetrics> metrics)
{
    return std::make_unique<TLabelKeys>(std::move(subShards), fts, std::move(shardMeteringContext), std::move(metrics));
}

} // NSolomon::NMemStore::NIndex
