#include "label_values.h"
#include "metrics.h"
#include "shard.h"
#include "subshard.h"

#include <solomon/libs/cpp/steady_timer/steady_timer.h>
#include <solomon/libs/cpp/string_pool/string_pool.h>
#include <solomon/services/memstore/lib/fts/index.h>

#include <library/cpp/actors/core/actor.h>
#include <library/cpp/actors/core/hfunc.h>
#include <library/cpp/containers/absl_flat_hash/flat_hash_map.h>

using namespace NActors;

namespace NSolomon::NMemStore::NIndex {
namespace {

class TLabelValues: public TActor<TLabelValues> {
public:
    TLabelValues(
            std::vector<TActorId> subShards,
            TActorId fts,
            std::shared_ptr<IResourceUsageContext> shardMeteringContext,
            std::shared_ptr<TMetrics> metrics)
        : TActor<TLabelValues>{&TThis::StateFunc}
        , SubShards_{std::move(subShards)}
        , Fts_{fts}
        , ShardMeteringContext_{std::move(shardMeteringContext)}
        , Metrics_(std::move(metrics))
    {
        Y_VERIFY(SubShards_.size() > 0);
        Metrics_->ReadInflight->Inc();
    }

    STATEFN(StateFunc) {
        Timer_.Reset();

        Y_DEFER {
            if (ShardMeteringContext_) {
                ShardMeteringContext_->AddCpuTime(Timer_.Step());
            }
        };

        switch (ev->GetTypeRewrite()) {
            hFunc(TShardEvents::TLabelValues, OnLabelValuesReq);
            hFunc(TSubShardEvents::TFindMetricsResponse, OnFindMetricsResponse)
            hFunc(TFtsIndexEvents::TFindResp, OnFtsResponse)
        }
    }

private:
    void OnLabelValuesReq(TShardEvents::TLabelValues::TPtr& ev) {
        Req_ = ev;

        KeysFilter_.reserve(Req_->Get()->Keys.size());
        for (const auto& key: Req_->Get()->Keys) {
            KeysFilter_.insert(key);
        }

        // TODO: FTS index must handle special event for this type of request
        //       current implementation is completely wrong!
        Send(Fts_, new TFtsIndexEvents::TFindReq{Req_->Get()->Selectors, Req_->Get()->Limit});
    }

    void OnFtsResponse(TFtsIndexEvents::TFindResp::TPtr& ev) {
        auto& metricList = ev->Get()->ResultIds;
        Y_VERIFY(metricList.size() == SubShards_.size(),
                 "Unexpected num of subshards from FTS index: %lu. Expected: %lu",
                 metricList.size(),
                 SubShards_.size());

        auto& selectors = Req_->Get()->Selectors;
        for (size_t subShardId = 0; subShardId < SubShards_.size(); ++subShardId) {
            if (auto& metricId = metricList[subShardId]; !metricId.empty()) {
                auto& subShard = SubShards_[subShardId];
                Send(subShard, new TSubShardEvents::TFindMetrics{SelfId(), selectors, std::move(metricId)});
                RequestsInFlight_++;
            }
        }

        if (RequestsInFlight_ == 0) {
            ReplyAndDie();
        }
    }

    void OnFindMetricsResponse(TSubShardEvents::TFindMetricsResponse::TPtr& ev) {
        const auto& textFilter = Req_->Get()->TextFilter;

        // TODO: limit number of label values
        for (auto& [type, labels]: ev->Get()->Found) {
            for (auto [k, v]: labels) {
                TStringBuf key{k};
                if (KeysFilter_.empty() || KeysFilter_.contains(key)) {
                    TStringBuf value{v};
                    if (textFilter.empty() || value.Contains(textFilter)) {
                        ui32 keyIdx = Pool_.Put(key);
                        ui32 valueIdx = Pool_.Put(value);
                        ValuesPerKey_[keyIdx].insert(valueIdx);
                    }
                }
            }
        }

        MetricCount_ += static_cast<ui32>(ev->Get()->Found.size());

        if (--RequestsInFlight_ == 0) {
            ReplyAndDie();
        }
    }

    void ReplyAndDie() {
        using TLabelState = TShardEvents::TLabelValuesResponse::TLabelState;

        std::vector<TLabelState> labels;
        labels.reserve(ValuesPerKey_.size());

        for (auto& [keyIdx, valuesIdx]: ValuesPerKey_) {
            auto& label = labels.emplace_back();

            label.Key = keyIdx;
            label.Values = std::move(valuesIdx);

            // TODO: fill other fields of TLabelState
        }

        auto* resp = new TShardEvents::TLabelValuesResponse{Pool_.Build(), std::move(labels), MetricCount_};
        Send(Req_->Sender, resp, 0, Req_->Cookie);

        Metrics_->ReadInflight->Dec();
        Metrics_->ReadComplete->Inc();

        PassAway();
    }

private:
    const std::vector<TActorId> SubShards_;
    const TActorId Fts_;
    std::shared_ptr<IResourceUsageContext> ShardMeteringContext_;
    std::shared_ptr<TMetrics> Metrics_;
    TSteadyTimer Timer_{};
    TShardEvents::TLabelValues::TPtr Req_;
    NStringPool::TStringPoolBuilder Pool_;
    absl::flat_hash_set<TStringBuf, THash<TStringBuf>> KeysFilter_;
    absl::flat_hash_map<ui32, absl::flat_hash_set<ui32>> ValuesPerKey_;
    ui32 MetricCount_{0};
    ui32 RequestsInFlight_{0};
};

} // namespace

std::unique_ptr<IActor> CreateLabelValuesActor(
        std::vector<TActorId> subShards,
        TActorId fts,
        std::shared_ptr<IResourceUsageContext> shardMeteringContext,
        std::shared_ptr<TMetrics> metrics)
{
    return std::make_unique<TLabelValues>(std::move(subShards), fts, std::move(shardMeteringContext), std::move(metrics));
}

} // NSolomon::NMemStore::NIndex
