#pragma once

#include <solomon/libs/cpp/ts_model/point_type.h>

#include <library/cpp/monlib/metrics/metric.h>
#include <library/cpp/monlib/metrics/metric_registry.h>

namespace NSolomon::NMemStore::NIndex {

struct TPointMetrics {
    /**
     * How many points of given type per seconds are written to memstore instance
     */
    NMonitoring::IRate* WritePointsRate{nullptr};
};

struct TTsMemoryMetrics {
    NMonitoring::IIntGauge* SealFramesCount;
    NMonitoring::IIntGauge* SealFramePointsCount;
    NMonitoring::IIntGauge* SealFrameDataSizeBytes;
    NMonitoring::IIntGauge* SealFrameBuffersSizeBytes;
    NMonitoring::IIntGauge* SealFramesSizeBytes;
    NMonitoring::IIntGauge* SealFrameHoldersSizeBytes;

    NMonitoring::IIntGauge* MutableFramesCount;
    NMonitoring::IIntGauge* MutableFramePointsCount;
    NMonitoring::IIntGauge* MutableFrameDataSizeBytes;
    NMonitoring::IIntGauge* MutableFrameBuffersSizeBytes;
    NMonitoring::IIntGauge* MutableFramesSizeBytes;
    NMonitoring::IIntGauge* MutableFrameHoldersSizeBytes;

    NMonitoring::IIntGauge* DequeCapacityBytes;
    NMonitoring::IIntGauge* BuffersCacheSizeBytes;

    explicit TTsMemoryMetrics(const std::shared_ptr<NMonitoring::TMetricRegistry>& metrics) {
        SealFramesCount = metrics->IntGauge({{"sensor", "index.ts.seal_frame.count"}});
        SealFramePointsCount = metrics->IntGauge({{"sensor", "index.ts.seal_frame.points_count"}});
        SealFrameDataSizeBytes = metrics->IntGauge({{"sensor", "index.ts.seal_frame.data_size_bytes"}});
        SealFrameBuffersSizeBytes = metrics->IntGauge({{"sensor", "index.ts.seal_frame.buffers_size_bytes"}});
        SealFramesSizeBytes = metrics->IntGauge({{"sensor", "index.ts.seal_frame.size_bytes"}});
        SealFrameHoldersSizeBytes = metrics->IntGauge({{"sensor", "index.ts.seal_frame.holder_size_bytes"}});

        MutableFramesCount = metrics->IntGauge({{"sensor", "index.ts.mutable_frame.count"}});
        MutableFramePointsCount = metrics->IntGauge({{"sensor", "index.ts.mutable_frame.points_count"}});
        MutableFrameDataSizeBytes = metrics->IntGauge({{"sensor", "index.ts.mutable_frame.data_size_bytes"}});
        MutableFrameBuffersSizeBytes = metrics->IntGauge({{"sensor", "index.ts.mutable_frame.buffers_size_bytes"}});
        MutableFramesSizeBytes = metrics->IntGauge({{"sensor", "index.ts.mutable_frame.size_bytes"}});
        MutableFrameHoldersSizeBytes = metrics->IntGauge({{"sensor", "index.ts.mutable_frame.holder_size_bytes"}});

        DequeCapacityBytes = metrics->IntGauge({{"sensor", "index.ts.deque_capacity_bytes"}});
        BuffersCacheSizeBytes = metrics->IntGauge({{"sensor", "index.ts.cache_size_bytes"}});
    }
};

class TMetrics {
public:
    std::shared_ptr<NMonitoring::TMetricRegistry> Registry;

    NMonitoring::IIntGauge* WriteInflight;
    NMonitoring::IRate* WriteComplete;
    NMonitoring::IRate* WriteErrors;
    NMonitoring::IRate* WriteErrorsParsing;
    NMonitoring::IRate* WriteErrorsTypeMismatch;

    NMonitoring::IRate* StorageCleanups;
    NMonitoring::IRate* StorageReorderings;
    NMonitoring::IRate* StorageMerges;
    NMonitoring::IIntGauge* StorageShards;
    NMonitoring::IIntGauge* StorageSubShards;
    NMonitoring::IIntGauge* StorageFrames;
    NMonitoring::IIntGauge* StorageFramesOpen;
    NMonitoring::IIntGauge* StorageFramesSealed;
    NMonitoring::IIntGauge* StorageMetrics;
    NMonitoring::IIntGauge* StorageMetricsEmpty;

    NMonitoring::IIntGauge* MemoryTimeSeries;
    NMonitoring::IIntGauge* MemoryLabelsInternal;
    NMonitoring::IIntGauge* MemoryLabelsData;
    NMonitoring::IIntGauge* MemorySubshardAux;
    NMonitoring::IHistogram* DataExpansion;
    NMonitoring::IHistogram* IncomingPointsPerTs;
    NMonitoring::IIntGauge* StoredPointsCount;

    TTsMemoryMetrics TsMemoryMetrics;

    NMonitoring::IIntGauge* MemoryFts;
    NMonitoring::IIntGauge* MemoryParsers;
    NMonitoring::IIntGauge* MemoryFtsAdd;

    NMonitoring::IIntGauge* SnapshotInflight;
    NMonitoring::IRate* SnapshotComplete;

    NMonitoring::IIntGauge* ReadInflight;
    NMonitoring::IRate* ReadComplete;

    const TPointMetrics& GetPointMetrics(NSolomon::NTsModel::EPointType type) const {
        const i32 key = static_cast<i32>(type);
        auto it = PointMetrics_.find(key);
        return it == PointMetrics_.end() ? PointMetrics_.at(static_cast<i32>(NSolomon::NTsModel::EPointType::Unknown)) : it->second;
    }

    const TPointMetrics& GetTotalPointMetrics() const {
        return PointMetrics_.at(TotalPointMetricsKey_);
    }

    explicit TMetrics(const std::shared_ptr<NMonitoring::TMetricRegistry>& metrics)
            : Registry(metrics)
            , TsMemoryMetrics(metrics)
    {

        WriteInflight = metrics->IntGauge({{"sensor", "index.write.inflight"}});
        WriteComplete = metrics->Rate({{"sensor", "index.write.complete"}});
        WriteErrors = metrics->Rate({{"sensor", "index.write.errors"}, {"type", "TOTAL"}});
        WriteErrorsParsing = metrics->Rate({{"sensor", "index.write.errors"}, {"type", "PARSING_ERROR"}});
        WriteErrorsTypeMismatch = metrics->Rate({{"sensor", "index.write.errors"}, {"type", "TYPE_MISMATCH"}});

        StorageCleanups = metrics->Rate({{"sensor", "index.storage.cleanups"}});
        StorageReorderings = metrics->Rate({{"sensor", "index.storage.reorderings"}});
        StorageMerges = metrics->Rate({{"sensor", "index.storage.merges"}});
        StorageShards = metrics->IntGauge({{"sensor", "index.storage.shards"}});
        StorageSubShards = metrics->IntGauge({{"sensor", "index.storage.subshards"}});
        StorageFrames = metrics->IntGauge({{"sensor", "index.storage.frames"}, {"type", "TOTAL"}});
        StorageFramesOpen = metrics->IntGauge({{"sensor", "index.storage.frames"}, {"type", "OPEN"}});
        StorageFramesSealed = metrics->IntGauge({{"sensor", "index.storage.frames"}, {"type", "SEALED"}});
        StorageMetrics = metrics->IntGauge({{"sensor", "index.storage.metrics"}});
        StorageMetricsEmpty = metrics->IntGauge({{"sensor", "index.storage.metrics->empty"}});

        MemoryTimeSeries = metrics->IntGauge({{"sensor", "index.subshard.time_series.size_bytes"}});
        MemoryLabelsInternal = metrics->IntGauge({{"sensor", "index.subshard.labels_internal.size_bytes"}});
        MemoryLabelsData = metrics->IntGauge({{"sensor", "index.subshard.labels_data.size_bytes"}});
        MemorySubshardAux = metrics->IntGauge({{"sensor", "index.subshard.aux_data.size_bytes"}});
        DataExpansion = metrics->HistogramRate(
                {{"sensor", "index.parsers.expansion"}},
                NMonitoring::ExponentialHistogram(8, 2));
        IncomingPointsPerTs = metrics->HistogramRate(
                {{"sensor", "index.subshard.incoming_points_per_ts"}},
                NMonitoring::ExponentialHistogram(5, 2));
        StoredPointsCount = metrics->IntGauge({{"sensor", "index.point.count_in_memory"}});

        CreatePointMetrics(TotalPointMetricsKey_, "total");
        CreatePointMetrics(static_cast<i32>(NSolomon::NTsModel::EPointType::Unknown), "Unknown");
        CreatePointMetrics(static_cast<i32>(NSolomon::NTsModel::EPointType::DGauge), "DGauge");
        CreatePointMetrics(static_cast<i32>(NSolomon::NTsModel::EPointType::IGauge), "IGauge");
        CreatePointMetrics(static_cast<i32>(NSolomon::NTsModel::EPointType::Counter), "Counter");
        CreatePointMetrics(static_cast<i32>(NSolomon::NTsModel::EPointType::Rate), "Rate");
        CreatePointMetrics(static_cast<i32>(NSolomon::NTsModel::EPointType::Hist), "Hist");
        CreatePointMetrics(static_cast<i32>(NSolomon::NTsModel::EPointType::HistRate), "HistRate");
        CreatePointMetrics(static_cast<i32>(NSolomon::NTsModel::EPointType::LogHist), "LogHist");
        CreatePointMetrics(static_cast<i32>(NSolomon::NTsModel::EPointType::DSummary), "DSummary");

        MemoryFts = metrics->IntGauge({{"sensor", "index.fts.size_bytes"}});
        MemoryParsers = metrics->IntGauge({{"sensor", "index.parsers.size_bytes"}});
        MemoryFtsAdd = metrics->IntGauge({{"sensor", "index.fts.add.size_bytes"}});

        SnapshotInflight = metrics->IntGauge({{"sensor", "index.snapshot.inflight"}});
        SnapshotComplete = metrics->Rate({{"sensor", "index.snapshot.complete"}});

        ReadInflight = metrics->IntGauge({{"sensor", "index.read.inflight"}});
        ReadComplete = metrics->Rate({{"sensor", "index.read.complete"}});
    }

private:
    void CreatePointMetrics(i32 type, const char* TypeName) {
        TPointMetrics& pointMetrics = PointMetrics_[type];
        pointMetrics.WritePointsRate = Registry->Rate({{"sensor", "index.point.write_rate"}, {"type", TypeName}});
    }

private:
    std::unordered_map<i32, TPointMetrics> PointMetrics_;
    static constexpr i32 TotalPointMetricsKey_ = 0x1123214F; // magic key for total
};

} // namespace NSolomon::NMemStore::NIndex
