#include "read.h"
#include "metrics.h"
#include "shard.h"
#include "subshard.h"

#include <solomon/libs/cpp/steady_timer/steady_timer.h>
#include <solomon/libs/cpp/stockpile_codec/metric_archive.h>
#include <solomon/libs/cpp/string_pool/string_pool.h>
#include <solomon/libs/cpp/ts_math/error.h>
#include <solomon/libs/cpp/ts_math/operation_pipeline.h>

#include <library/cpp/actors/core/actor.h>
#include <library/cpp/actors/core/hfunc.h>

using namespace NActors;

namespace NSolomon::NMemStore::NIndex {
namespace {

TVector<NSolomon::NLabels::TLabels::TPair> IdToLabels(const NStringPool::TStringPool& strings, const std::vector<ui32>& ids) {
    Y_VERIFY(ids.size() % 2 == 0, "Id array should contain even number of elements (key/value pairs)");
    TVector<NSolomon::NLabels::TLabels::TPair> res;
    res.reserve(ids.size() / 2);
    for (size_t i = 0; i < ids.size(); i += 2) {
        res.emplace_back(strings[ids[i]].data(), strings[ids[i + 1]].data());
    }
    return res;
}

// TODO: split this actor into two: TReadOneActor and TReadManyActor
class TRead: public TActor<TRead> {
public:
    TRead(
            TNumId numId,
            TVector<TActorId> subShards,
            TActorId fts,
            std::shared_ptr<IResourceUsageContext> shardMeteringContext,
            std::shared_ptr<TMetrics> metrics)
        : TActor<TRead>{&TThis::StateFunc}
        , NumId_{numId}
        , SubShards_(std::move(subShards))
        , Fts_(fts)
        , ShardMeteringContext_{std::move(shardMeteringContext)}
        , Metrics_(std::move(metrics))
    {
        Metrics_->ReadInflight->Inc();
    }

    STATEFN(StateFunc) {
        Timer_.Reset();

        Y_DEFER {
            if (ShardMeteringContext_) {
                ShardMeteringContext_->AddCpuTime(Timer_.Step());
            }
        };

        switch (ev->GetTypeRewrite()) {
            hFunc(TShardEvents::TReadOne, OnReadOne)
            hFunc(TShardEvents::TReadMany, OnReadMany)
            hFunc(TFtsIndexEvents::TReadResp, OnFtsResp);
            hFunc(TSubShardEvents::TReadMetricsResponse, OnReadMetricsResponse)
        }
    }

public:
    void OnReadOne(TShardEvents::TReadOne::TPtr& ev) {
        ReadOne_ = ev;

        auto subShardId = ReadOne_->Get()->Labels.Hash() % SubShards_.size();
        auto* event = new TSubShardEvents::TReadMetricsByLabels(
                SelfId(),
                ReadOne_->Get()->From,
                ReadOne_->Get()->To,
                {ReadOne_->Get()->Labels});
        Send(SubShards_[subShardId], event);
        RequestsInFlight_ = 1;
    }

    void OnReadMany(TShardEvents::TReadMany::TPtr& ev) {
        ReadMany_ = ev;

        if (auto& resolved = ReadMany_->Get()->Resolved) {
            auto commonLabels = IdToLabels(resolved->Pool, resolved->CommonLabels);

            // shardId -> TVector<Labels>
            TVector<TVector<NSolomon::NLabels::TLabels>> res(SubShards_.size());
            for (const auto& labelsIds: resolved->ResolvedKeys) {
                auto labels = IdToLabels(resolved->Pool, labelsIds);
                labels.insert(labels.end(), commonLabels.begin(), commonLabels.end());

                auto storedLabels = NLabels::TLabels::OwnedStorage(labels.begin(), labels.end());
                res[storedLabels.Hash() % SubShards_.size()].push_back(std::move(storedLabels));
            }

            for (size_t subShard = 0; subShard < SubShards_.size(); ++subShard) {
                if (!res[subShard].empty()) {
                    auto event = new TSubShardEvents::TReadMetricsByLabels(
                            SelfId(),
                            ReadMany_->Get()->From,
                            ReadMany_->Get()->To,
                            std::move(res[subShard]));
                    Send(SubShards_[subShard], event);
                    RequestsInFlight_++;
                }
            }
            if (RequestsInFlight_ == 0) {
                ReplyAndDie();
            }
        } else if (auto& lookup = ReadMany_->Get()->Lookup) {
            Send(Fts_, new TFtsIndexEvents::TReadReq{
                std::move(lookup->Selectors),
                lookup->Limit,
                ReadMany_->Get()->FramesOnWindow});
        } else {
            Y_FAIL("invalid read request, nether Resolved nor Lookup are present");
        }
    }

    void OnFtsResp(TFtsIndexEvents::TReadResp::TPtr& ev) {
        auto req = ev->Get();
        Y_VERIFY(SubShards_.size() == req->MetricIdxs.size(),
                 "Unexpected size of FtsIndex response. Size: %lu, expected %lu",
                 req->MetricIdxs.size(), SubShards_.size());
        Y_VERIFY(ReadMany_, "got response from FTS index, but there is no ReadMany request");

        for (size_t shardId = 0; shardId < SubShards_.size(); ++ shardId) {
            if (!req->MetricIdxs[shardId].empty()) {
                auto event = new TSubShardEvents::TReadMetricsByMetricId(
                        SelfId(),
                        std::move(req->MetricIdxs[shardId]),
                        ReadMany_->Get()->From,
                        ReadMany_->Get()->To);
                Send(SubShards_[shardId], event);
                RequestsInFlight_++;
            }
        }
        if (RequestsInFlight_ == 0) {
            ReplyAndDie();
        }
    }

    void OnReadMetricsResponse(TSubShardEvents::TReadMetricsResponse::TPtr& ev) {
        for (auto& [labels, ts]: ev->Get()->Found) {
            auto& source = Source_.emplace_back();
            source.Labels.ConstructInPlace();
            source.Labels->reserve(labels.Size());
            for (auto& label: labels) {
                ui32 key = Pool_.Put(label.first);
                ui32 value = Pool_.Put(label.second);
                source.Labels->emplace_back(key, value);
            }
            source.Data.Reset(new TTimeSeriesFrames{std::move(ts)});
        }
        if (--RequestsInFlight_ == 0) {
            ReplyAndDie();
        }
    }

    void ReplyAndDie() {
        try {
            if (ReadMany_) {
                auto res = ReadMany_->Get()->Pipeline.Apply(std::move(Source_));
                BuildReadManyResponse(res);
            } else {
                auto res = ReadOne_->Get()->Pipeline.Apply(std::move(Source_));
                SendReadOneResponse(res);
            }
        } catch (NTsMath::TException& err) {
            SendErrorResponse(err.Code(), TString{err.AsStrBuf()});
        }

        Metrics_->ReadInflight->Dec();
        Metrics_->ReadComplete->Inc();

        PassAway();
    }

    void SendReadOneResponse(TVector<NTsMath::TTimeSeriesEncoded>& res) {
        Y_VERIFY(res.size() == 1, "too may metrics for ReadOne response: %lu", res.size());
        auto* event = new TShardEvents::TReadOneResponse{
            NumId_,
            ReadOne_->Get()->Format,
            std::move(res[0]),
            Pool_.Build()};
        Send(ReadOne_->Sender, event, 0, ReadOne_->Cookie);
    }

    void BuildReadManyResponse(TVector<NTsMath::TTimeSeriesEncoded>& res) {
        auto* event = new TShardEvents::TReadManyResponse{
            NumId_,
            ReadMany_->Get()->Format,
            std::move(res),
            Pool_.Build()};
        Send(ReadMany_->Sender, event, 0, ReadMany_->Cookie);
    }

    void SendErrorResponse(grpc::StatusCode status, TString message) {
        if (ReadMany_) {
            Send(ReadMany_->Sender, new TShardEvents::TReadError{status, std::move(message)}, 0, ReadMany_->Cookie);
        } else {
            Send(ReadOne_->Sender, new TShardEvents::TReadError{status, std::move(message)}, 0, ReadOne_->Cookie);
        }
    }

private:
    const TNumId NumId_;
    const std::vector<TActorId> SubShards_;
    const TActorId Fts_;
    std::shared_ptr<IResourceUsageContext> ShardMeteringContext_;
    std::shared_ptr<TMetrics> Metrics_;
    TSteadyTimer Timer_{};
    TShardEvents::TReadOne::TPtr ReadOne_;
    TShardEvents::TReadMany::TPtr ReadMany_;
    ui32 RequestsInFlight_{0};
    TVector<NTsMath::TTimeSeries> Source_;
    NStringPool::TStringPoolBuilder Pool_;
};

} // namespace

std::unique_ptr<IActor> CreateReadOneActor(
        TNumId numId,
        TVector<NActors::TActorId> subShards,
        TActorId fts,
        std::shared_ptr<IResourceUsageContext> shardMeteringContext,
        std::shared_ptr<TMetrics> metrics)
{
    return std::make_unique<TRead>(numId, std::move(subShards), fts, std::move(shardMeteringContext), std::move(metrics));
}

std::unique_ptr<IActor> CreateReadManyActor(
        TNumId numId,
        TVector<NActors::TActorId> subShards,
        TActorId fts,
        std::shared_ptr<IResourceUsageContext> shardMeteringContext,
        std::shared_ptr<TMetrics> metrics)
{
    return std::make_unique<TRead>(numId, std::move(subShards), fts, std::move(shardMeteringContext), std::move(metrics));
}

} // NSolomon::NMemStore::NIndex
