#include "shard.h"
#include "add_point.h"
#include "find.h"
#include "label_keys.h"
#include "label_values.h"
#include "metrics.h"
#include "read.h"
#include "shard_manager.h"
#include "snapshot.h"
#include "subshard.h"
#include "unique_labels.h"

#include <solomon/services/memstore/lib/wal/wal_events.h>

#include <solomon/libs/cpp/actors/poison/poisoner.h>
#include <solomon/libs/cpp/backoff/jitter.h>
#include <solomon/libs/cpp/logging/logging.h>
#include <solomon/libs/cpp/steady_timer/steady_timer.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/event.h>
#include <library/cpp/actors/core/hfunc.h>
#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/string_utils/quote/quote.h>

#include <util/generic/deque.h>
#include <util/generic/size_literals.h>

#include <utility>

using namespace NActors;
using namespace NSolomon::NMemStore::NIndex;

namespace NSolomon::NMemStore::NIndex {
namespace {

constexpr TDuration INDEX_POSTPONE_PERIOD = TDuration::MilliSeconds(10);
constexpr TDuration MAX_INDEX_POSTPONE_PERIOD = TDuration::Seconds(1);
constexpr ui32 MAX_POSTPONED_COUNT = 1;

struct TSubShardInfo {
    const TActorId ActorId;
    ui64 OccupiedMemory{0};

    explicit TSubShardInfo(TActorId actorId)
        : ActorId(std::move(actorId))
    {
    }
};

constexpr TDuration MEMORY_METERING_PERIOD = TDuration::MilliSeconds(1000);
constexpr TDuration TIME_SERIES_MEMORY_METERING_PERIOD = TDuration::Seconds(15);

struct TLocalEvents: private TPrivateEvents {
    enum {
        ReportMemoryMetering = SpaceBegin,
        ReportTimeSeriesMemoryStat
    };

    struct TReportMemoryMetering: public NActors::TEventLocal<TReportMemoryMetering, ReportMemoryMetering> {
    };

    struct TReportTimeSeriesMemoryStat: public NActors::TEventLocal<TReportTimeSeriesMemoryStat, ReportTimeSeriesMemoryStat> {
    };
};

class TGatherSubShardMetrics: public TActor<TGatherSubShardMetrics> {
public:
    TGatherSubShardMetrics(
            yandex::monitoring::selfmon::Page page,
            const TVector<TSubShardInfo>& subshards)
        : TActor<TGatherSubShardMetrics>(&TGatherSubShardMetrics::StateFunc)
        , Page_{std::move(page)}
        , SubShards_{subshards}
    {
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(NSelfMon::TEvPageDataReq, OnSelfMon)
            hFunc(TShardEvents::TMetricsTableResponse, OnSubshardTable)
        }
    }

    void OnSelfMon(NSelfMon::TEvPageDataReq::TPtr& ev) {
        Sender_ = ev->Sender;
        TryFromString(ev->Get()->Param("limit"), Limit_);

        TErrorOr<TSelectors, TInvalidSelectorsFormat> selectors = TSelectors{};
        auto selectorsStr = TString{ev->Get()->Param("selectors")};
        if (selectorsStr) {
            CGIUnescape(selectorsStr);
            if (!selectorsStr.StartsWith('{')) {
                selectorsStr = "{" + selectorsStr + "}";
            }
            try {
                selectors = ParseSelectors(selectorsStr);
            } catch (TInvalidSelectorsFormat& err) {
                selectors = err;
            }
        }

        auto* grid = Page_.mutable_grid();

        if (auto* r = grid->add_rows()) {
            auto* form = r->add_columns()->mutable_component()->mutable_form();
            form->set_method(yandex::monitoring::selfmon::FormMethod::Get);

            {
                auto* input = form->add_items()->mutable_input();
                input->set_type(yandex::monitoring::selfmon::InputType::Hidden);
                input->set_name("id");
                input->set_value(TString(ev->Get()->Param("id")));
            }

            {
                auto* item = form->add_items();
                item->set_label("Filter");
                item->set_help("e.g. host=solomon-fetcher-sas-00, sensor=memory_total");
                auto* input = item->mutable_input();
                input->set_type(yandex::monitoring::selfmon::InputType::Text);
                input->set_name("selectors");
                input->set_value(selectorsStr);
            }

            auto* submit = form->add_submit();
            submit->set_title("Filter");
        }

        if (auto* r = grid->add_rows()) {
            auto* h = r->add_columns()->mutable_component()->mutable_heading();
            if (selectors.Success()) {
                h->set_content("Metrics");
            } else {
                h->set_content(FormatExc(selectors.Error()));
            }
            h->set_level(3);
        }

        if (selectors.Success()) {
            auto* t = grid->add_rows()->add_columns()->mutable_component()->mutable_table();
            t->set_numbered(true);

            auto* subshardColumn = t->add_columns();
            subshardColumn->set_title("Subshard");
            SubshardValues_ = subshardColumn->mutable_reference();

            auto* idColumn = t->add_columns();
            idColumn->set_title("Id");
            IdValues_ = idColumn->mutable_reference();

            auto* typeColumn = t->add_columns();
            typeColumn->set_title("Type");
            TypeValues_ = typeColumn->mutable_string();

            auto* labelsColumn = t->add_columns();
            labelsColumn->set_title("Labels");
            LabelsValues_ = labelsColumn->mutable_string();

            Query_ = ev->Get()->Query;
            for (const auto& subshard: SubShards_) {
                Send(subshard.ActorId, MakeHolder<TShardEvents::TMetricsTableRequest>(selectors.Value(), Query_, Limit_));
            }
            Inflight_ = SubShards_.size();
        } else {
            ReplyAndDie();
        }
    }

    void OnSubshardTable(TShardEvents::TMetricsTableResponse::TPtr& ev) {
        auto table = std::move(ev->Get()->Table);
        auto& idValues = *table.mutable_columns()->Mutable(0)->mutable_reference();
        auto& typeValues = *table.mutable_columns()->Mutable(1)->mutable_string();
        auto& labelsValues = *table.mutable_columns()->Mutable(2)->mutable_string();
        auto subshardInd = ToString(ev->Get()->SubshardInd);

        for (size_t i = 0; i < idValues.valuesSize(); ++i) {
            if (auto* ref = SubshardValues_->add_values()) {
                ref->set_title(subshardInd);
                ref->set_page("/shards");
                ref->set_args(Query_ + "&subId=" + subshardInd);
            }

            auto* ref = IdValues_->add_values();
            ref->set_title(std::move(*idValues.mutable_values(i)->mutable_title()));
            ref->set_page(std::move(*idValues.mutable_values(i)->mutable_page()));
            ref->set_args(*idValues.mutable_values(i)->mutable_args() + "&subId=" + subshardInd);

            TypeValues_->add_values(std::move(*typeValues.mutable_values(i)));
            LabelsValues_->add_values(std::move(*labelsValues.mutable_values(i)));

            if (SubshardValues_->valuesSize() >= Limit_) {
                ReplyAndDie();
                return;
            }
        }

        if (--Inflight_ == 0) {
            ReplyAndDie();
        }
    }

    void ReplyAndDie() {
        Send(Sender_, MakeHolder<NSelfMon::TEvPageDataResp>(std::move(Page_)));
        PassAway();
    }

private:
    TActorId Sender_;
    TString Query_;
    yandex::monitoring::selfmon::Page Page_;
    const TVector<TSubShardInfo>& SubShards_;
    size_t Limit_ = 50;
    size_t Inflight_ = 0;
    yandex::monitoring::selfmon::ReferenceArray* SubshardValues_ = nullptr;
    yandex::monitoring::selfmon::ReferenceArray* IdValues_ = nullptr;
    yandex::monitoring::selfmon::StringArray* TypeValues_ = nullptr;
    yandex::monitoring::selfmon::StringArray* LabelsValues_ = nullptr;
};

class TShard: public TActorBootstrapped<TShard> {
#define LOG_P "{shard " << SelfId() << " (" << NumId_ << ")} "

    struct TFrameInfo {
        TInstant CreationTime;
        TLogId LatestLogId;
        bool Sealed = false;
        size_t WritesInProgress = 0;
        ui32 ReadsInProgress = 0;
        TInstant MinPoint = TInstant::Max();
        TInstant MaxPoint = TInstant::Zero();

        TFrameInfo(TInstant creationTime, TLogId logId)
            : CreationTime{creationTime}
            , LatestLogId{logId}
        {
        }
    };

public:
    TShard(
            TActorId shardManager,
            TNumId numId,
            TShardManagerConfig config,
            TActorId indexLimiterId,
            bool isWalInitialized,
            std::shared_ptr<IIndexWriteLimiter> indexLimiter,
            std::shared_ptr<IResourceUsageContext> shardMeteringContext,
            std::shared_ptr<TMetrics> metrics)
        : ShardManager_{shardManager}
        , NumId_{numId}
        , Config_{std::move(config)}
        , IndexLimiterId_{indexLimiterId}
        , IsWalInitialized_{isWalInitialized}
        , IndexGlobalLimiter_{std::move(indexLimiter)}
        , ShardMeteringContext_{std::move(shardMeteringContext)}
        , Metrics_{std::move(metrics)}
    {
        Metrics_->StorageShards->Inc();
        WakeUpCycle_ = std::min(TDuration::Minutes(1), Config_.ChunkLength / 5);
        if (ShardMeteringContext_ && Metrics_) {
            ShardMemoryMetric_ = Metrics_->Registry->IntGauge({
                {"sensor", "index.shard.memory.size_bytes"},
                {"shard", ToString(NumId_)}});
        }
    }

public:
    void Bootstrap() {
        if (ShardMeteringContext_) {
            MON_INFO(Index, LOG_P << "start shard " << NumId_ << " with metering context");
        } else {
            MON_INFO(Index, LOG_P << "start shard " << NumId_ << " without metering context");
        }

        FtsIndex_ = Register(
                FtsIndex(Config_.NumSubshards, Metrics_->MemoryFts, Metrics_->MemoryFtsAdd, ShardMeteringContext_).release(),
                TMailboxType::Simple, Config_.FtsPool);

        for (ui8 i = 0; i < Config_.NumSubshards; ++i) {
            auto subShard = CreateSubShard(NumId_, i, FtsIndex_, IndexLimiterId_, Metrics_, ShardMeteringContext_);
            SubShards_.emplace_back(Register(subShard.release()));
            Send(IndexLimiterId_, new TIndexLimiterEvents::TSubscribe{SubShards_.back().ActorId});
        }

        Schedule(WakeUpCycle_, new TEvents::TEvWakeup);
        Schedule(Jitter_(MEMORY_METERING_PERIOD), new TLocalEvents::TReportMemoryMetering);
        Schedule(Jitter_(TIME_SERIES_MEMORY_METERING_PERIOD), new TLocalEvents::TReportTimeSeriesMemoryStat);
        Become(&TThis::StateFunc);
        Send(IndexLimiterId_, new TIndexLimiterEvents::TSubscribe{SelfId()});
    }

    STATEFN(StateFunc) {
        Timer_.Reset();

        Y_DEFER {
            if (ShardMeteringContext_) {
                // TODO: pass the ctx to all underlying actors as well
                ShardMeteringContext_->AddCpuTime(Timer_.Step());
            }
        };

        switch (ev->GetTypeRewrite()) {
            hFunc(TShardEvents::TIndex, OnIndex)
            sFunc(TShardEvents::TIndexPostpone, OnIndexPostpone)
            hFunc(TShardEvents::TFind, OnFind)
            hFunc(TShardEvents::TLabelKeys, OnLabelKeys)
            hFunc(TShardEvents::TLabelValues, OnLabelValues)
            hFunc(TShardEvents::TUniqueLabels, OnUniqueLabels)
            hFunc(TShardEvents::TReadOne, OnReadOne)
            hFunc(TShardEvents::TReadMany, OnReadMany)
            hFunc(TShardEvents::TReadDone, OnReadDone)
            hFunc(TShardEvents::TAddPointDone, OnAddPointDone)
            hFunc(TShardEvents::TSnapshotDone, OnSnapshotDone)
            hFunc(NSelfMon::TEvPageDataReq, OnSelfMon)
            hFunc(TEvents::TEvWakeup, OnWakeup)
            hFunc(TIndexLimiterEvents::TReportLimiterState, OnGetIndexLimiterState)
            sFunc(NWal::TWalEvents::TWalInitialized, OnWalInitialized)
            sFunc(TLocalEvents::TReportMemoryMetering, OnReportMemoryMetering)
            sFunc(TLocalEvents::TReportTimeSeriesMemoryStat, OnReportTsMemoryStat)
            hFunc(TShardEvents::TMemoryMeteringResponse, OnMemoryMeteringResponse)
            hFunc(TEvents::TEvPoison, OnPoison)
        }
    }

public:
    void RejectIndexQuery(TShardEvents::TIndex::TPtr& ev, TShardEvents::TIndexDone::EStatus status) {
        auto req = ev->Get();
        ui64 totalSize = 0;
        for (auto& request: req->Requests.Requests) {
            totalSize += request.SizeBytes();
            Send(
                    request.ReplyTo,
                    new TShardEvents::TIndexDone{request.SizeBytes(), status},
                    0,
                    request.Cookie);
        }
        Send(ev->Sender, MakeHolder<TShardEvents::TIndexDone>(totalSize, status));
        IndexGlobalLimiter_->AddIndexMessageCount(-1);
    }

    void OnIndex(TShardEvents::TIndex::TPtr& ev) {
        if (Closed_) {
            MON_WARN(Index, LOG_P << "point dropped: shard is closed");
            RejectIndexQuery(ev, TShardEvents::TIndexDone::ShardClosed);
            return;
        }

        if (IndexLimiterData_.GetState().IsMemoryExhausted()) {
            MON_WARN(Index, LOG_P << "point dropped: memory exhausted");
            RejectIndexQuery(ev, TShardEvents::TIndexDone::MemoryExhausted);
            return;
        }

        const bool isLocked = IndexLimiterData_.GetState().IsParsersSizeOverLimit();
        if (isLocked) {
            if (DeferredAddPointActors_.size() > MAX_POSTPONED_COUNT) {
                MON_WARN(Index, LOG_P << "point dropped: postponed requests count is over the limit");
                RejectIndexQuery(ev, TShardEvents::TIndexDone::MemoryExhausted);
                return;
            }
        }

        auto req = ev->Get();
        auto frameIdx = GetFrameForInsertion(req->LogId);
        auto& frame = GetFrame(frameIdx);
        frame.WritesInProgress++;
        RequestsInProgress_++;

        auto addPoint = CreateAddPointActor(
                SelfId(),
                GetSubShards(),
                frameIdx,
                std::move(req->Requests.Requests),
                Metrics_,
                IndexLimiterId_,
                ev->Sender,
                ShardMeteringContext_);


        if (isLocked) {
            DeferredAddPointActors_.push_back(std::move(addPoint));
            Schedule(PostponeCycle_, new TShardEvents::TIndexPostpone{});
            return;
        }
        PostponeCycle_ = INDEX_POSTPONE_PERIOD;

        if (!DeferredAddPointActors_.empty()) {
            DeferredAddPointActors_.push_back(std::move(addPoint));
            addPoint = std::move(DeferredAddPointActors_.front());
            DeferredAddPointActors_.pop_front();
        }
        Register(addPoint.release(), TMailboxType::Simple, Config_.RequestPool);
    }

    void OnIndexPostpone() {
        if (DeferredAddPointActors_.empty()) {
            return;
        }
        if (IndexLimiterData_.GetState().IsParsersSizeOverLimit()
                || IndexLimiterData_.GetState().IsMemoryExhausted())
        {
            PostponeCycle_ = Min(PostponeCycle_ * 2, MAX_INDEX_POSTPONE_PERIOD);
            Schedule(PostponeCycle_, new TShardEvents::TIndexPostpone{});
            return;
        }
        PostponeCycle_ = INDEX_POSTPONE_PERIOD;
        std::unique_ptr<IActor> addPoint = std::move(DeferredAddPointActors_.front());
        DeferredAddPointActors_.pop_front();
        if (!DeferredAddPointActors_.empty()) {
            Schedule(PostponeCycle_, new TShardEvents::TIndexPostpone{});
        }
        Register(addPoint.release(), TMailboxType::Simple, Config_.RequestPool);
    }

    void OnFind(TShardEvents::TFind::TPtr& ev) {
        if (Frames_.empty()) {
            Send(ev->Sender, new TShardEvents::TFindResponse{}, 0, ev->Cookie);
            return;
        }

        auto processorId = Register(
                CreateFindActor(GetSubShards(), FtsIndex_, ShardMeteringContext_, Metrics_).release(),
                TMailboxType::Simple, Config_.RequestPool);
        TActivationContext::Send(ev->Forward(processorId));
    }

    void OnLabelKeys(TShardEvents::TLabelKeys::TPtr& ev) {
        if (Frames_.empty()) {
            Send(ev->Sender, new TShardEvents::TLabelKeysResponse, 0, ev->Cookie);
            return;
        }

        auto processorId = Register(
                CreateLabelKeysActor(GetSubShards(), FtsIndex_, ShardMeteringContext_, Metrics_).release(),
                TMailboxType::Simple, Config_.RequestPool);
        TActivationContext::Send(ev->Forward(processorId));
    }

    void OnLabelValues(TShardEvents::TLabelValues::TPtr& ev) {
        if (Frames_.empty()) {
            Send(ev->Sender, new TShardEvents::TLabelValuesResponse, 0, ev->Cookie);
            return;
        }

        auto processorId = Register(
                CreateLabelValuesActor(GetSubShards(), FtsIndex_, ShardMeteringContext_, Metrics_).release(),
                TMailboxType::Simple, Config_.RequestPool);
        TActivationContext::Send(ev->Forward(processorId));
    }

    void OnUniqueLabels(TShardEvents::TUniqueLabels::TPtr& ev) {
        if (Frames_.empty()) {
            Send(ev->Sender, new TShardEvents::TUniqueLabelsResponse, 0, ev->Cookie);
            return;
        }

        auto processorId = Register(
                CreateUniqueLabelsActor(GetSubShards(), FtsIndex_, ShardMeteringContext_, Metrics_).release(),
                TMailboxType::Simple, Config_.RequestPool);
        TActivationContext::Send(ev->Forward(processorId));
    }

    void OnReadOne(TShardEvents::TReadOne::TPtr& ev) {
        if (Frames_.empty()) {
            auto* event = new TShardEvents::TReadOneResponse{NumId_, ev->Get()->Format};
            Send(ev->Sender, event, 0, ev->Cookie);
            return;
        }

        auto& req = *ev->Get();
        if (!req.To) {
            req.To = TActivationContext::Now();
        }
        if (!req.From) {
            req.From = req.To - Config_.IndexCapacity;
        }

        auto processorId = Register(
                CreateReadOneActor(NumId_, GetSubShards(), FtsIndex_, ShardMeteringContext_, Metrics_).release(),
                TMailboxType::Simple, Config_.RequestPool);
        TActivationContext::Send(ev->Forward(processorId));
    }

    TVector<TFrameIdx> GetFramesOnWindow(TInstant beginWindow, TInstant endWindow) {
        TVector<TFrameIdx> res;
        for (size_t id = 0; id < Frames_.size(); ++ id) {
            auto& frame = Frames_[id];
            if (beginWindow < frame.MaxPoint && endWindow > frame.MinPoint) {
                res.push_back(FirstFrame_ + id);
            }
        }
        return res;
    }

    void OnReadMany(TShardEvents::TReadMany::TPtr& ev) {
        if (Frames_.empty()) {
            Send(ev->Sender, new TShardEvents::TReadManyResponse{NumId_, ev->Get()->Format}, 0, ev->Cookie);
            return;
        }

        auto& req = *ev->Get();
        if (!req.To) {
            req.To = TActivationContext::Now();
        }
        if (!req.From) {
            req.From = req.To - Config_.IndexCapacity;
        }

        auto framesOnWindow = GetFramesOnWindow(req.From, req.To);
        if (framesOnWindow.empty()) {
            Send(ev->Sender, new TShardEvents::TReadManyResponse{NumId_, ev->Get()->Format}, 0, ev->Cookie);
            return;
        }

        // TODO: send frameIdxs through constructor parameter
        req.FramesOnWindow = std::move(framesOnWindow);

        auto processorId = Register(
                CreateReadManyActor(NumId_, GetSubShards(), FtsIndex_, ShardMeteringContext_, Metrics_).release(),
                TMailboxType::Simple, Config_.RequestPool);
        TActivationContext::Send(ev->Forward(processorId));
    }

    void OnReadDone(TShardEvents::TReadDone::TPtr& ev) {
        auto frameIdx = ev->Get()->FrameIdx;
        Y_VERIFY(GetFrame(frameIdx).ReadsInProgress > 0, "extra dec");
        GetFrame(frameIdx).ReadsInProgress--;
    }

    void OnGetIndexLimiterState(TIndexLimiterEvents::TReportLimiterState::TPtr& ev) {
        IndexLimiterData_.SetState(std::move(ev->Get()->State));
    }

    void OnPoison(const TEvents::TEvPoison::TPtr& ev) {
        // TODO: switch state instead
        Closed_ = true;
        PoisonEvent_ = ev;
        Send(IndexLimiterId_, new TIndexLimiterEvents::TUnsubscribe{SelfId()});
        MaybePassAway();
    }

    void OnAddPointDone(TShardEvents::TAddPointDone::TPtr& ev) {
        MON_DEBUG(Index, LOG_P << "done write request in frame #" << ev->Get()->Frame);

        auto& frame = GetFrame(ev->Get()->Frame);
        frame.WritesInProgress--;
        frame.MinPoint = Min(frame.MinPoint, ev->Get()->Range.MinTs);
        frame.MaxPoint = Max(frame.MaxPoint, ev->Get()->Range.MaxTs);
        RequestsInProgress_--;
        IndexGlobalLimiter_->AddIndexMessageCount(-1);

        SnapshotFrames();
        MaybePassAway();
    }

    void OnSnapshotDone(TShardEvents::TSnapshotDone::TPtr& ev) {
        MON_DEBUG(Index, LOG_P << "done snapshotting frame #" << ev->Get()->Frame);

        Send(ShardManager_, new NWal::TWalEvents::TSnapshot{
            NumId_, ev->Get()->LogId, std::move(ev->Get()->Meta), std::move(ev->Get()->Data)});

        GetFrame(ev->Get()->Frame).Sealed = true;
        SnapshotInProgress_ = false;
        SnapshotFrames();
    }

    void OnWakeup(TEvents::TEvWakeup::TPtr&) {
        Schedule(WakeUpCycle_, new TEvents::TEvWakeup);
        SnapshotFrames();
        CleanupFrames();
    }

    void OnSelfMon(NSelfMon::TEvPageDataReq::TPtr& ev) {
        if (auto subIdStr = ev->Get()->Param("subId")) {
            size_t subId = 0;
            if (TryFromString(subIdStr, subId)) {
                TActivationContext::Send(ev->Forward(GetSubShards()[subId]));
                return;
            }
        }

        yandex::monitoring::selfmon::Page page;
        page.set_title(TStringBuilder{} << "Shard " << NumId_);

        auto* grid = page.mutable_grid();
        if (auto* r = grid->add_rows()) {
            auto* obj = r->add_columns()->mutable_component()->mutable_object();
            if (auto* f = obj->add_fields()) {
                f->set_name("Id");
                f->mutable_value()->set_uint32(NumId_);
            }
            if (auto* f = obj->add_fields()) {
                f->set_name("WakeUpCycle");
                f->mutable_value()->set_duration(WakeUpCycle_.GetValue());
            }
            if (auto* f = obj->add_fields()) {
                f->set_name("FirstFrame");
                f->mutable_value()->set_uint64(FirstFrame_);
            }
            if (auto* f = obj->add_fields()) {
                f->set_name("FrameToSnapshot");
                f->mutable_value()->set_uint64(FrameToSnapshot_);
            }
            if (auto* f = obj->add_fields()) {
                f->set_name("SnapshotInProgress");
                f->mutable_value()->set_boolean(SnapshotInProgress_);
            }
            if (auto* f = obj->add_fields()) {
                f->set_name("Closed");
                f->mutable_value()->set_boolean(Closed_);
            }
            if (auto* f = obj->add_fields()) {
                f->set_name("Closed");
                f->mutable_value()->set_boolean(Closed_);
            }
            if (auto* f = obj->add_fields()) {
                f->set_name("RequestsInProgress");
                f->mutable_value()->set_uint64(RequestsInProgress_);
            }
        }
        if (auto* r = grid->add_rows()) {
            auto* h = r->add_columns()->mutable_component()->mutable_heading();
            h->set_content("Frames");
            h->set_level(3);
        }
        if (auto* r = grid->add_rows()) {
            auto* t = r->add_columns()->mutable_component()->mutable_table();
            t->set_numbered(true);

            auto* creationColumn = t->add_columns();
            creationColumn->set_title("Creation Time");
            auto* creationValues = creationColumn->mutable_time();

            auto* logIdColumn = t->add_columns();
            logIdColumn->set_title("LogId");
            auto* logIdValues = logIdColumn->mutable_string();

            auto* sealedColumn = t->add_columns();
            sealedColumn->set_title("Sealed");
            auto* sealedValues = sealedColumn->mutable_boolean();

            auto* writesInFlightColumn = t->add_columns();
            writesInFlightColumn->set_title("Writes InFlight");
            auto* writesInFlightValues = writesInFlightColumn->mutable_uint64();

            auto* readsInFlightColumn = t->add_columns();
            readsInFlightColumn->set_title("Reads InFlight");
            auto* readsInFlightValues = readsInFlightColumn->mutable_uint32();

            auto* minTimeColumn = t->add_columns();
            minTimeColumn->set_title("Min Time");
            auto* minTimeValues = minTimeColumn->mutable_time();

            auto* maxTimeColumn = t->add_columns();
            maxTimeColumn->set_title("Max Time");
            auto* maxTimeValues = maxTimeColumn->mutable_time();

            for (const auto& frame: Frames_) {
                creationValues->add_values(frame.CreationTime.GetValue());
                logIdValues->add_values(TStringBuilder{} << frame.LatestLogId.TabletId << '/' << frame.LatestLogId.Txn);
                sealedValues->add_values(frame.Sealed);
                writesInFlightValues->add_values(frame.WritesInProgress);
                readsInFlightValues->add_values(frame.ReadsInProgress);
                minTimeValues->add_values(frame.MinPoint.GetValue());
                maxTimeValues->add_values(frame.MaxPoint.GetValue());
            }
        }

        auto actor = Register(new TGatherSubShardMetrics{std::move(page), SubShards_});
        TActivationContext::Send(ev->Forward(actor));
    }

    TFrameInfo& GetFrame(TFrameIdx idx) {
        Y_VERIFY(FirstFrame_ <= idx && idx < FirstFrame_ + Frames_.size(),
                 "invalid frame index #%lu. Must be between #%lu and #%lu. ShardId: %u",
                 idx, FirstFrame_, FirstFrame_ + Frames_.size() - 1, NumId_);
        return Frames_[idx - FirstFrame_];
    }

    TFrameIdx GetFrameForInsertion(TLogId logId) {
        if (Frames_.empty() ||
            Frames_.back().Sealed ||
            Frames_.back().CreationTime + Config_.ChunkLength <= TActivationContext::Now() ||
            Frames_.back().LatestLogId.TabletId != logId.TabletId)
        {
            auto frameIdx = FirstFrame_ + Frames_.size();
            auto creationTime = TActivationContext::Now();

            if (Frames_.empty() && creationTime > TInstant::FromValue(Config_.ChunkLength.GetValue())) {
                // Jitter initial chunk length
                creationTime += TDuration::Seconds(RandomNumber(Config_.ChunkLength.Seconds()));
                creationTime -= Config_.ChunkLength * 2 / 3;
            }

            Frames_.push_back(TFrameInfo{creationTime, logId});

            Metrics_->StorageFramesOpen->Inc();
            Metrics_->StorageFrames->Inc();

            for (auto subShard: SubShards_) {
                Send(subShard.ActorId, MakeHolder<TSubShardEvents::TAddFrame>(frameIdx));
            }
            Send(FtsIndex_, MakeHolder<TFtsIndexEvents::TAddFrame>(frameIdx));

            MON_INFO(Index, LOG_P << "created frame #" << frameIdx);
            return frameIdx;
        } else {
            Y_VERIFY(Frames_.back().LatestLogId.Txn < logId.Txn, "inconsistent log id");
            Frames_.back().LatestLogId = logId;
            return FirstFrame_ + Frames_.size() - 1;
        }
    }

    void CleanupFrames() {
        for (;;) {
            if (Frames_.empty()) {
                MON_DEBUG(Index, LOG_P << "no frames to drop");
                return;
            }

            const auto& firstFrame = Frames_.front();
            if (!firstFrame.Sealed) {
                MON_DEBUG(Index, LOG_P << "cannot drop first frame, because it is not yet sealed");
                return;
            }

            if (auto reads = firstFrame.ReadsInProgress) {
                MON_DEBUG(Index, LOG_P << "cannot drop first frame, because there " << reads << " active reads in it");
                return;
            }

            auto expireAt = firstFrame.CreationTime + Config_.IndexCapacity + Config_.ChunkLength;
            if (expireAt > TActivationContext::Now()) {
                const auto limiterState = IndexLimiterData_.GetState();
                const bool forceCleanup = limiterState.IsMemoryExhausted() || limiterState.IsTimeSeriesSizeOverLimit();
                if (IsWalInitialized_ && !forceCleanup) {
                    MON_DEBUG(Index, LOG_P << "cannot drop first frame, because it will be expired only at " << expireAt);
                    return;
                }
            }

            MON_INFO(Index, LOG_P << "drop frame #" << FirstFrame_);

            Metrics_->StorageFramesSealed->Dec();
            Metrics_->StorageFrames->Dec();

            for (auto subShard: SubShards_) {
                Send(subShard.ActorId, MakeHolder<TSubShardEvents::TDropFrame>(FirstFrame_));
            }
            Send(FtsIndex_, MakeHolder<TFtsIndexEvents::TDropFrame>(FirstFrame_));

            Frames_.pop_front();
            FirstFrame_++;
        }
    }

    void SnapshotFrames() {
        if (SnapshotInProgress_) {
            return;
        }

        if (FrameToSnapshot_ >= FirstFrame_ + Frames_.size()) {
            return;
        }

        auto& frame = GetFrame(FrameToSnapshot_);

        if (frame.WritesInProgress > 0) {
            return;
        }

        bool isLastFrame = FrameToSnapshot_ + 1 == FirstFrame_ + Frames_.size();
        if (isLastFrame && frame.CreationTime + Config_.ChunkLength > TActivationContext::Now()) {
            return;
        }

        MON_INFO(Index, LOG_P << "snapshotting frame #" << FrameToSnapshot_);

        Metrics_->StorageFramesOpen->Dec();
        Metrics_->StorageFramesSealed->Inc();

        SnapshotInProgress_ = true;

        auto snapshot = CreateSnapshotActor(
                SelfId(),
                NumId_,
                FrameToSnapshot_,
                frame.LatestLogId,
                SubShards_.size(),
                Metrics_,
                ShardMeteringContext_);
        auto snapshotId = Register(snapshot.release());

        for (auto subShard: SubShards_) {
            Send(subShard.ActorId, MakeHolder<TSubShardEvents::TSealAndSnapshotFrame>(snapshotId, FrameToSnapshot_));
        }

        FrameToSnapshot_++;
    }

    void OnWalInitialized() {
        IsWalInitialized_ = true;
    }

    void OnReportMemoryMetering() {
        ui64 memorySizeBytes{0};
        for (const auto& subShard: SubShards_) {
            memorySizeBytes += subShard.OccupiedMemory;
            Send(subShard.ActorId, new TShardEvents::TMemoryMeteringRequest);
        }

        if (ShardMeteringContext_) {
            ShardMeteringContext_->SetMemoryBytes(memorySizeBytes);
        }
        if (ShardMemoryMetric_) {
            ShardMemoryMetric_->Set(memorySizeBytes);
        }

        Schedule(Jitter_(MEMORY_METERING_PERIOD), new TLocalEvents::TReportMemoryMetering);
    }

    void OnReportTsMemoryStat() {
        for (const auto& subShard: SubShards_) {
            Send(subShard.ActorId, new TShardEvents::TTsMemoryStatReport);
        }
        Schedule(Jitter_(TIME_SERIES_MEMORY_METERING_PERIOD), new TLocalEvents::TReportTimeSeriesMemoryStat);
    }

    void OnMemoryMeteringResponse(TShardEvents::TMemoryMeteringResponse::TPtr& ev) {
        const ui8 shardId = ev->Get()->SubShardId;
        SubShards_[shardId].OccupiedMemory = ev->Get()->MemorySizeBytes;
    }

    void MaybePassAway() {
        // Note: we have to wait for all writes and all fts lookups.
        // AddPoint actor sends messages to subshards and waits for responses. If subshards die before they're able
        // to respond, AddPoint actor will be left hanging forever. We don't wait for other types of requests
        // because they don't interact with subshards directly.
        if (Closed_ && RequestsInProgress_ == 0) {
            // TODO: dec metrics
            MON_DEBUG(Index, LOG_P << "shard is dying");

            if (PoisonEvent_) {
                std::set<TActorId> toPoison;
                for (auto subShard: SubShards_) {
                    toPoison.insert(subShard.ActorId);
                }
                toPoison.insert(FtsIndex_);
                PoisonAll(PoisonEvent_, toPoison);
            } else {
                for (auto subShard: SubShards_) {
                    Send(subShard.ActorId, new TEvents::TEvPoison);
                }
                Send(FtsIndex_, new TEvents::TEvPoison);
            }

            Metrics_->StorageShards->Dec();
            PassAway();
        }
    }

private:
    TVector<TActorId> GetSubShards() const {
        TVector<TActorId> subShards;
        subShards.reserve(SubShards_.size());
        for (const auto& subShard: SubShards_) {
            subShards.push_back(subShard.ActorId);
        }
        return subShards;
    }

private:
    TActorId ShardManager_;
    TNumId NumId_;
    TShardManagerConfig Config_;
    TDeque<std::unique_ptr<IActor>> DeferredAddPointActors_;
    TActorId IndexLimiterId_;
    TIndexLimiterData IndexLimiterData_;
    bool IsWalInitialized_;
    std::shared_ptr<IIndexWriteLimiter> IndexGlobalLimiter_;
    std::shared_ptr<IResourceUsageContext> ShardMeteringContext_;
    TDuration WakeUpCycle_;
    TDuration PostponeCycle_{INDEX_POSTPONE_PERIOD};
    // TODO: optimization -- compute an actual user time
    TSteadyTimer Timer_{};

    TVector<TSubShardInfo> SubShards_;
    TFullJitter Jitter_;
    NMonitoring::TIntGauge* ShardMemoryMetric_{0};
    TDeque<TFrameInfo> Frames_;
    TFrameIdx FirstFrame_ = 0;
    TFrameIdx FrameToSnapshot_ = 0;

    bool SnapshotInProgress_ = false;

    TActorId FtsIndex_;

    bool Closed_ = false;
    TEvents::TEvPoison::TPtr PoisonEvent_;
    size_t RequestsInProgress_ = 0;
    std::shared_ptr<TMetrics> Metrics_;
};

} // namespace

std::unique_ptr<IActor> CreateShard(
        TActorId shardManager,
        TNumId numId,
        TShardManagerConfig config,
        TActorId indexLimiterId,
        bool isWalInitialized,
        std::shared_ptr<IIndexWriteLimiter> indexLimiter,
        std::shared_ptr<IResourceUsageContext> shardMeteringContext,
        std::shared_ptr<TMetrics> metrics)
{
    return std::make_unique<TShard>(
            shardManager,
            numId,
            std::move(config),
            indexLimiterId,
            isWalInitialized,
            std::move(indexLimiter),
            std::move(shardMeteringContext),
            std::move(metrics));
}

} // namespace NSolomon::NMemStore::NIndex
