#pragma once

#include "index_limiter.h"
#include "write_request.h"
#include "shard_manager_config.h"
#include "metrics.h"

#include <solomon/services/memstore/lib/event_slots.h>
#include <solomon/services/memstore/lib/fts/index.h>
#include <solomon/services/memstore/lib/metric/metric.h>
#include <solomon/services/memstore/lib/time_series/time_series.h>
#include <solomon/services/memstore/lib/time_series/ts_memory_stat.h>

#include <solomon/libs/cpp/actors/events/events.h>
#include <solomon/libs/cpp/actors/fwd.h>
#include <solomon/libs/cpp/error_or/error_or.h>
#include <solomon/libs/cpp/selfmon/selfmon.h>
#include <solomon/libs/cpp/shard_metrics/resource_context.h>
#include <solomon/libs/cpp/stockpile_codec/format.h>
#include <solomon/libs/cpp/string_pool/string_pool.h>
#include <solomon/libs/cpp/ts_math/operation_pipeline.h>

#include <library/cpp/actors/core/event_local.h>
#include <library/cpp/containers/absl_flat_hash/flat_hash_set.h>

namespace NSolomon::NMemStore::NIndex {

class TShardEvents: private TEventSlot<EEventSpace::MemStore, ES_SHARD> {
    enum {
        Index = SpaceBegin,
        IndexPostpone,
        IndexDone,
        Find,
        FindResponse,
        LabelKeys,
        LabelKeysResponse,
        LabelValues,
        LabelValuesResponse,
        UniqueLabels,
        UniqueLabelsResponse,
        ReadOne,
        ReadOneResponse,
        ReadMany,
        ReadManyResponse,
        ReadError,

        // TODO: these are non public events, probably they must be moved to a better place
        AddPointDone,
        SnapshotDone,
        ReadDone,

        MetricsTableRequest,
        MetricsTableResponse,
        MemoryMeteringRequest,
        MemoryMeteringResponse,
        TsMemoryStatReport,
        End,
    };
    static_assert(End < SpaceEnd, "too many event types");

public:
    /**
     * Request to index new data.
     */
    struct TIndex: public NActors::TEventLocal<TIndex, Index> {
        /**
         * Identifier of the WAL record in which this data is persisted.
         */
        TLogId LogId;

        /**
         * Requests for indexing.
         */
        TShardAddPointRequests Requests;

        explicit TIndex(TLogId logId, TShardAddPointRequests requests = {})
            : LogId(logId)
            , Requests(std::move(requests))
        {
        }
    };

    struct TIndexPostpone: public NActors::TEventLocal<TIndex, IndexPostpone> {

    };

    /**
     * Data has been indexed. A separate message of this type is generated for every `TAddPointRequest`.
     */
    struct TIndexDone: public NActors::TEventLocal<TIndexDone, IndexDone> {
        enum EStatus {
            OK,
            MemoryExhausted,
            ShardClosed
        };

        ui64 WrittenBytes;
        EStatus Status;

        explicit TIndexDone(ui64 written = 0, EStatus status = OK) noexcept
                : WrittenBytes(written)
                , Status(status)
        {
        }
    };

    /**
     * A 'find' request.
     */
    struct TFind: public NActors::TEventLocal<TFind, Find> {
        TNumId ShardId;
        TSelectors Selectors;
        ui32 Limit;

        TFind(TNumId shardId, TSelectors selectors, ui32 limit = 0) noexcept
            : ShardId{shardId}
            , Selectors{std::move(selectors)}
            , Limit{limit}
        {
        }
    };

    /**
     * A response for the 'find' request.
     */
    struct TFindResponse: public NActors::TEventLocal<TFindResponse, FindResponse> {
        NStringPool::TStringPool Strings;
        std::vector<TMetric> Metrics;
        ui32 TotalCount{0};

        TFindResponse() noexcept = default;

        TFindResponse(NStringPool::TStringPool strings, std::vector<TMetric> metrics, ui32 totalCount) noexcept
            : Strings{std::move(strings)}
            , Metrics{std::move(metrics)}
            , TotalCount{totalCount}
        {
        }
    };

    /**
     * A 'label keys' request.
     */
    struct TLabelKeys: public NActors::TEventLocal<TLabelKeys, LabelKeys> {
        TSelectors Selectors;

        explicit TLabelKeys(TSelectors selectors) noexcept
            : Selectors{std::move(selectors)}
        {
        }
    };

    /**
     * A response for the 'label keys' request.
     */
    struct TLabelKeysResponse: public NActors::TEventLocal<TLabelKeysResponse, LabelKeysResponse> {
        NStringPool::TStringPool Strings;
        std::vector<ui32> Keys;

        TLabelKeysResponse() noexcept = default;

        explicit TLabelKeysResponse(NStringPool::TStringPool strings, std::vector<ui32> keys) noexcept
            : Strings{std::move(strings)}
            , Keys{std::move(keys)}
        {
        }
    };

    /**
     * A 'label values' request.
     */
    struct TLabelValues: public NActors::TEventLocal<TLabelValues, LabelValues> {
        TSelectors Selectors;
        std::vector<TString> Keys;
        TString TextFilter;
        ui32 Limit;

        TLabelValues(TSelectors selectors, std::vector<TString> keys, TString textFilter, ui32 limit = 0) noexcept
            : Selectors{std::move(selectors)}
            , Keys{std::move(keys)}
            , TextFilter{std::move(textFilter)}
            , Limit{limit}
        {
        }
    };

    /**
     * A response for the 'label values' request.
     */
    struct TLabelValuesResponse: public NActors::TEventLocal<TLabelValuesResponse, LabelValuesResponse> {
        struct TLabelState {
            absl::flat_hash_set<ui32> Values;
            ui32 Key{0};
            ui32 MetricCount{0};
            bool Truncated{false};
        };

        NStringPool::TStringPool Strings;
        std::vector<TLabelState> Labels;
        ui32 MetricCount{0};

        TLabelValuesResponse() noexcept = default;

        TLabelValuesResponse(NStringPool::TStringPool strings, std::vector<TLabelState> labels, ui32 metricCount) noexcept
            : Strings{std::move(strings)}
            , Labels{std::move(labels)}
            , MetricCount{metricCount}
        {
        }
    };

    /**
     * A 'unique labels' request.
     */
    struct TUniqueLabels: public NActors::TEventLocal<TUniqueLabels, UniqueLabels> {
        TSelectors Selectors;
        std::vector<TString> Keys;

        TUniqueLabels(TSelectors selectors, std::vector<TString> keys) noexcept
            : Selectors{std::move(selectors)}
            , Keys{std::move(keys)}
        {
        }
    };

    /**
     * A response for the 'unique labels' request.
     */
    struct TUniqueLabelsResponse: public NActors::TEventLocal<TUniqueLabelsResponse, UniqueLabelsResponse> {
        NStringPool::TStringPool Strings;
        absl::flat_hash_set<TLabels, THash<TLabels>> Labels;

        TUniqueLabelsResponse() noexcept = default;

        TUniqueLabelsResponse(NStringPool::TStringPool strings, absl::flat_hash_set<TLabels, THash<TLabels>> labels) noexcept
            : Strings{std::move(strings)}
            , Labels{std::move(labels)}
        {
        }
    };

    /**
     * A 'read one' request.
     */
    struct TReadOne: public NActors::TEventLocal<TReadOne, ReadOne> {
        NLabels::TLabels Labels;
        NStockpile::EFormat Format;
        NTsMath::TOperationPipeline Pipeline;
        TInstant From;
        TInstant To;

        TReadOne(
                NLabels::TLabels labels,
                NStockpile::EFormat format,
                NTsMath::TOperationPipeline pipeline,
                TInstant from,
                TInstant to) noexcept
            : Labels{std::move(labels)}
            , Format{format}
            , Pipeline{std::move(pipeline)}
            , From{from}
            , To{to}
        {
        }
    };

    /**
     * A response for the 'read one' request.
     */
    struct TReadOneResponse: public NActors::TEventLocal<TReadOneResponse, ReadOneResponse> {
        TNumId ShardId;
        NStockpile::EFormat Format;
        NTsMath::TTimeSeriesEncoded Metric;
        NStringPool::TStringPool Strings;

        /**
         * An empty response.
         */
        explicit TReadOneResponse(TNumId shardId, NStockpile::EFormat format) noexcept
            : ShardId{shardId}
            , Format{format}
        {
        }

        explicit TReadOneResponse(
                TNumId shardId,
                NStockpile::EFormat format,
                NTsMath::TTimeSeriesEncoded metric,
                NStringPool::TStringPool strings) noexcept
            : ShardId{shardId}
            , Format{format}
            , Metric{std::move(metric)}
            , Strings{std::move(strings)}
        {
        }
    };

    /**
     * A 'read many' request.
     */
    // TODO: split into two TReadManyLookup and TReadManyResolved events
    struct TReadMany: public NActors::TEventLocal<TReadMany, ReadMany> {
        struct TResolved {
            NStringPool::TStringPool Pool;
            std::vector<ui32> CommonLabels;
            std::vector<std::vector<ui32>> ResolvedKeys;
        };

        struct TLookup {
            TSelectors Selectors;
            ui32 Limit{0};
        };

        NStockpile::EFormat Format;
        NTsMath::TOperationPipeline Pipeline;
        TInstant From;
        TInstant To;
        std::vector<TFrameIdx> FramesOnWindow; // filled only inside TShardActor
        std::unique_ptr<TResolved> Resolved;
        std::unique_ptr<TLookup> Lookup;
    };

    /**
     * A response for the 'read many' request.
     */
    struct TReadManyResponse: public NActors::TEventLocal<TReadManyResponse, ReadManyResponse> {
        TNumId ShardId;
        NStockpile::EFormat Format;
        std::vector<NTsMath::TTimeSeriesEncoded> Metrics;
        NStringPool::TStringPool Strings;

        /**
         * An empty response.
         */
        explicit TReadManyResponse(TNumId shardId, NStockpile::EFormat format) noexcept
            : ShardId{shardId}
            , Format{format}
        {
        }

        explicit TReadManyResponse(
                TNumId shardId,
                NStockpile::EFormat format,
                std::vector<NTsMath::TTimeSeriesEncoded> metrics,
                NStringPool::TStringPool strings) noexcept
            : ShardId{shardId}
            , Format{format}
            , Metrics{std::move(metrics)}
            , Strings{std::move(strings)}
        {
        }
    };

    /**
     * An error response for the 'read one' or 'read many' request.
     */
    struct TReadError: public NActors::TEventLocal<TReadError, ReadError> {
        grpc::StatusCode Status;
        TString Message;

        explicit TReadError(grpc::StatusCode status, TString message) noexcept
            : Status{status}
            , Message{std::move(message)}
        {
        }
    };

    struct TAddPointDone: public NActors::TEventLocal<TAddPointDone, AddPointDone> {
        TFrameIdx Frame;
        TPointsRange Range;

        TAddPointDone(TFrameIdx frame, TPointsRange range)
            : Frame{frame}
            , Range{range}
        {
        }
    };

    struct TSnapshotDone: public NActors::TEventLocal<TSnapshotDone, SnapshotDone> {
        TFrameIdx Frame;
        TLogId LogId;
        TString Meta;
        TString Data;

        TSnapshotDone(TLogId logId, TNumId numId)
            : Frame(numId)
            , LogId(logId)
        {
        }

        TSnapshotDone(TFrameIdx frame, TLogId logId, TString meta, TString data)
            : Frame{frame}
            , LogId{logId}
            , Meta(std::move(meta))
            , Data(std::move(data))
        {
        }
    };

    struct TReadDone: public NActors::TEventLocal<TReadDone, ReadDone> {
        TFrameIdx FrameIdx;

        explicit TReadDone(TFrameIdx frameIdx)
            : FrameIdx{frameIdx}
        {
        }
    };

    struct TMetricsTableRequest: NActors::TEventLocal<TMetricsTableRequest, MetricsTableRequest> {
        TSelectors Selectors;
        TString Query;
        size_t Limit;

        TMetricsTableRequest(TSelectors selectors, TString query, size_t limit)
            : Selectors{std::move(selectors)}
            , Query{std::move(query)}
            , Limit{limit}
        {
        }
    };

    struct TMetricsTableResponse: NActors::TEventLocal<TMetricsTableResponse, MetricsTableResponse> {
        ui8 SubshardInd;
        yandex::monitoring::selfmon::Table Table;

        TMetricsTableResponse(ui8 subshardId, yandex::monitoring::selfmon::Table table)
            : SubshardInd{subshardId}
            , Table{std::move(table)}
        {
        }
    };

    struct TMemoryMeteringRequest: NActors::TEventLocal<TMemoryMeteringRequest, MemoryMeteringRequest> {
    };

    struct TMemoryMeteringResponse: NActors::TEventLocal<TMemoryMeteringResponse, MemoryMeteringResponse> {
        ui8 SubShardId;
        ui64 MemorySizeBytes;

        TMemoryMeteringResponse(ui8 subShardIndex, ui64 memoryOccupied)
            : SubShardId(subShardIndex)
            , MemorySizeBytes(memoryOccupied)
        {
        }
    };

    struct TTsMemoryStatReport: NActors::TEventLocal<TTsMemoryStatReport, TsMemoryStatReport>{
    };
};

/**
 * In-memory storage for a single shard.
 *
 * @param shardManager      shard manager actor.
 * @param numId             id of this shard.
 * @param config            other settings for index.
 */
std::unique_ptr<NActors::IActor> CreateShard(
        NActors::TActorId shardManager,
        TNumId numId,
        TShardManagerConfig config,
        NActors::TActorId indexLimiterId,
        bool isWalInitialized,
        std::shared_ptr<NSolomon::NMemStore::NIndex::IIndexWriteLimiter> indexLimiter,
        std::shared_ptr<IResourceUsageContext> shardMeteringContext,
        std::shared_ptr<TMetrics> metrics);

} // namespace NSolomon::NMemStore::NIndex
