#include "shard_manager.h"
#include "index_limiter.h"
#include "shard.h"

#include <solomon/services/memstore/lib/index/metrics.h>
#include <solomon/services/memstore/lib/wal/wal_events.h>

#include <solomon/libs/cpp/actors/poison/poisoner.h>
#include <solomon/libs/cpp/load_info_service/collector.h>
#include <solomon/libs/cpp/load_info_service/host_info.h>
#include <solomon/libs/cpp/load_info_service/load_info_service.h>
#include <solomon/libs/cpp/local_shard_provider/local_shard_provider.h>
#include <solomon/libs/cpp/logging/logging.h>
#include <solomon/libs/cpp/selfmon/selfmon.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/hfunc.h>

#include <util/generic/deque.h>

using yandex::monitoring::load_info::LoadInfoResponse;
using namespace NActors;
using namespace NSolomon::NDb::NModel;
using namespace NSolomon::NLoadInfo;
using namespace NSolomon::NMemStore::NIndex;
using namespace NSolomon;

namespace NSolomon::NMemStore::NIndex {

namespace {

class TShardManagerSelfMon: public TActor<TShardManagerSelfMon> {
public:
    explicit TShardManagerSelfMon(TActorId shardManager) noexcept
        : TActor<TShardManagerSelfMon>(&TThis::StateFunc)
        , ShardManager_{shardManager}
    {
    }

    STFUNC(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            HFunc(NSelfMon::TEvPageDataReq, OnRequest)
            hFunc(TEvents::TEvPoison, OnPoison)
        }
    }

private:
    void OnRequest(const NSelfMon::TEvPageDataReq::TPtr& ev, const TActorContext& ctx) {
        ctx.Send(ev->Forward(ShardManager_));
    }

    void OnPoison(const TEvents::TEvPoison::TPtr& ev) {
        Send(ev->Sender, new TEvents::TEvPoisonTaken);
        PassAway();
    }

private:
    TActorId ShardManager_;
};

class TShardManager: public TActorBootstrapped<TShardManager> {
#define LOG_P "{shard manager " << SelfId() << "} "

    struct TShardData {
        TShardKey ShardKey;
        TActorId ShardActor;
        TInstant LastUpdate;
    };

public:
    TShardManager(
            TShardManagerConfig config,
            NActors::TActorId indexLimiterId,
            std::shared_ptr<IIndexWriteLimiter> indexLimiter,
            const std::shared_ptr<NMonitoring::TMetricRegistry>& metrics)
        : Config_(std::move(config))
        , Metrics_{std::make_shared<TMetrics>(metrics)}
        , IndexLimiter_(std::move(indexLimiter))
        , IndexLimiterId_(indexLimiterId)
    {
    }

    void Bootstrap() {
        MON_INFO(Index, "Starting shard manager. Each shard has " << (int)Config_.NumSubshards << " subshards.");
        Generation_ = TActivationContext::Now().MicroSeconds() + 1;
        NSelfMon::RegisterPage(
                *TActorContext::ActorSystem(),
                "/shards", "Shards",
                std::make_unique<TShardManagerSelfMon>(SelfId()));

        if (Config_.LocalShardProvider) {
            Send(Config_.LocalShardProvider, new TLocalShardProviderEvents::TSubscribe{});
        }

        if (Config_.MeteringMetricsRepo) {
            // TODO(ivanzhukov):
//            Schedule(MemoryMeteringCollectionInterval_, new TEvCollectMemoryMetering);
        }

        Become(&TThis::StateFunc);
        Send(IndexLimiterId_, new TIndexLimiterEvents::TSubscribe{SelfId()});
    }

    STFUNC(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            HFunc(TShardEvents::TIndex, OnIndex)
            hFunc(TShardManagerEvents::TFindShard, OnFindShard)
            HFunc(TShardManagerEvents::TIndexBatch, OnIndexBatch)
            hFunc(TShardManagerEvents::TListShards, OnListShards)
            hFunc(TShardManagerEvents::TEnableSnapshots, OnEnableSnapshots)
            HFunc(NWal::TWalEvents::TSnapshot, OnSnapshot)
            HFunc(NSelfMon::TEvPageDataReq, OnSelfMon)
            hFunc(TLocalShardProviderEvents::TLocalShards, OnShardsConfigs);
            hFunc(TLoadInfoEvents::TGetLoadInfo, OnGetLoadInfo);
            hFunc(TIndexLimiterEvents::TReportLimiterState, OnGetIndexLimiterState)
            hFunc(NWal::TWalEvents::TWalInitialized, OnWalInitialized)
            hFunc(TEvents::TEvPoison, OnPoison)
        }
    }

private:
    void RemoveShard(TNumId numId) {
        auto it = Shards_.find(numId);
        if (it == Shards_.end()) {
            return;
        }

        MON_INFO(Index, LOG_P << "shard removed: " << it->second.ShardKey << '(' << it->first << ')');

        Send(it->second.ShardActor, new TEvents::TEvPoison());
        Shards_.erase(it);

        ++Generation_;
    }

    void AddShard(const NDb::NModel::TShardConfig& config) {
        auto& shardData = Shards_[config.NumId];

        shardData.ShardKey = {
            config.ProjectId,
            config.ClusterName,
            config.ServiceName,
        };
        MON_INFO(Index, LOG_P << "shard added: " << shardData.ShardKey << '(' << config.NumId << ')');

        std::shared_ptr<TShardMeteringContext> shardMeteringCtx;
        if (Config_.MeteringMetricsRepo) {
            shardMeteringCtx = Config_.MeteringMetricsRepo->GetContext(config.ProjectId, config.Id);
            shardMeteringCtx->ShardNumId = config.NumId;
        }

        auto shard = CreateShard(
                SelfId(),
                config.NumId,
                Config_,
                IndexLimiterId_,
                IsWalInitialized_,
                IndexLimiter_,
                std::move(shardMeteringCtx),
                Metrics_);
        shardData.ShardActor = Register(shard.release(), TMailboxType::HTSwap, Config_.RequestPool);
        ++Generation_;
    }

    void OnShardsConfigs(const TLocalShardProviderEvents::TLocalShards::TPtr& evPtr) {
        MON_INFO(Index, "got local shard configs");

        for (const auto& numId: evPtr->Get()->Removed) {
            RemoveShard(numId);
        }

        for (const auto& config: evPtr->Get()->Added) {
            AddShard(config);
        }

        // for now, MemStore doesn't use any shard settings
//        for (const auto& config: evPtr->Get()->Updated) {
//        }
    }

    void OnGetLoadInfo(TLoadInfoEvents::TGetLoadInfo::TPtr& ev) {
        auto responsePtr = std::make_unique<LoadInfoResponse>();

        if (Config_.MeteringMetricsRepo) {
            TLoadInfoCollector collector{responsePtr.get()};
            Config_.MeteringMetricsRepo->Visit(collector);
        }

        GatherHostInfo(responsePtr.get());

        Send(
                ev->Get()->ReplyTo,
                new TLoadInfoEvents::TGetLoadInfoResult{std::move(responsePtr)});
    }

    bool ReplyDoneForMovedShard(TActorId replyTo, const TShardAddPointRequests& requests) {
        MON_WARN(Index, LOG_P << "skipping data for shard " << requests.NumId << ". Probably it's been assigned to another host");

        // TODO: add a metric

        ui64 totalSize = 0;
        for (auto& request: requests.Requests) {
            totalSize += request.SizeBytes();
            Send(request.ReplyTo, new TShardEvents::TIndexDone{request.Meta.size() + request.Data.size()}, 0, request.Cookie);
        }

        Send(replyTo, MakeHolder<TShardEvents::TIndexDone>(totalSize));
        IndexLimiter_->AddIndexMessageCount(-1);
        return true;
    }

    void OnIndex(const TShardEvents::TIndex::TPtr& ev, const TActorContext& ctx) {
        // TODO: a place for optimization: do this check before sending an event from wal to index
        if (Shards_.contains(ev->Get()->Requests.NumId)) {
            RelayIndexEventToShard(ev.Get(), ctx);
        } else {
            ReplyDoneForMovedShard(ev->Sender, ev->Get()->Requests);
        }
    }

    void OnFindShard(TShardManagerEvents::TFindShard::TPtr& ev) {
        TActorId shardActor;
        if (auto it = Shards_.find(ev->Get()->NumId); it != Shards_.end()) {
            shardActor = it->second.ShardActor;
        }
        Send(ev->Sender, new TShardManagerEvents::TFindShardResponse{shardActor}, 0, ev->Cookie);
    }

    void OnIndexBatch(const TShardManagerEvents::TIndexBatch::TPtr& ev, const TActorContext& ctx) {
        auto* req = ev->Get();
        for (auto& [numId, requests]: req->Requests) {
            if (!Shards_.contains(numId)) {
                ReplyDoneForMovedShard(ev->Sender, requests);
                continue;
            }

            IEventHandle handle{
                {},
                ev->Sender,
                new TShardEvents::TIndex{req->LogId, std::move(requests)}
            };
            RelayIndexEventToShard(reinterpret_cast<TShardEvents::TIndex::THandle*>(&handle), ctx);
        }
    }

    void RelayIndexEventToShard(TShardEvents::TIndex::THandle* ev, const TActorContext& ctx) {
        TNumId numId = ev->Get()->Requests.NumId;

        TShardData& shardData = Shards_[numId];
        shardData.LastUpdate = TActivationContext::Now();

        if (!shardData.ShardActor) {
            MON_ERROR(Index, LOG_P << "shard actor for id " << numId << " is not initialized");
            return;
        }

        ctx.Send(ev->Forward(shardData.ShardActor));
    }

    void OnListShards(const TShardManagerEvents::TListShards::TPtr& ev) {
        ui64 generation = ev->Get()->Generation;
        std::vector<std::pair<TNumId, TShardKey>> shards;

        if (generation == 0 || generation != Generation_) {
            shards.reserve(Shards_.size());
            for (auto& [numId, shardData]: Shards_) {
                shards.emplace_back(numId, shardData.ShardKey);
            }
        }

        Send(ev->Sender, new TShardManagerEvents::TListShardsResponse{Generation_, std::move(shards)}, 0, ev->Cookie);
    }

    void OnEnableSnapshots(TShardManagerEvents::TEnableSnapshots::TPtr& ev) {
        SnapshotCollector_ = ev->Get()->Collector;
    }

    void OnSnapshot(NWal::TWalEvents::TSnapshot::TPtr& ev, const TActorContext& ctx) {
        ctx.Send(ev->Forward(SnapshotCollector_));
    }

    void OnSelfMon(NSelfMon::TEvPageDataReq::TPtr& ev, const TActorContext& ctx) {
        if (auto idStr = ev->Get()->Param("id")) {
            TNumId id;
            if (TryFromString(idStr, id)) {
                if (auto it = Shards_.find(id); it != Shards_.end()) {
                    ctx.Send(ev->Forward(it->second.ShardActor));
                    return;
                } else {
                    yandex::monitoring::selfmon::Page page;
                    page.mutable_component()->mutable_value()->set_string(TString{"Unable to find shard "} + idStr);
                    Send(ev->Sender, MakeHolder<NSelfMon::TEvPageDataResp>(std::move(page)));
                    return;
                }
            }
        }

        yandex::monitoring::selfmon::Page page;
        auto* grid = page.mutable_grid();
        if (auto* r = grid->add_rows()) {
            auto* obj = r->add_columns()->mutable_component()->mutable_object();
            if (auto* f = obj->add_fields()) {
                f->set_name("generation");
                f->mutable_value()->set_uint64(Generation_);
            }
            if (auto* f = obj->add_fields()) {
                f->set_name("shards");
                f->mutable_value()->set_uint64(Shards_.size());
            }
        }
        if (auto* r = grid->add_rows()) {
            auto* table = r->add_columns()->mutable_component()->mutable_table();
            table->set_numbered(true);

            auto* idColumn = table->add_columns();
            idColumn->set_title("Id");
            auto* idValues = idColumn->mutable_reference();

            auto* projectColumn = table->add_columns();
            projectColumn->set_title("Project");
            auto* projectValues = projectColumn->mutable_string();

            auto* clusterColumn = table->add_columns();
            clusterColumn->set_title("Cluster");
            auto* clusterValues = clusterColumn->mutable_string();

            auto* serviceColumn = table->add_columns();
            serviceColumn->set_title("Service");
            auto* serviceValues = serviceColumn->mutable_string();

            auto* lastUpdateColumn = table->add_columns();
            lastUpdateColumn->set_title("Last Updated (ago)");
            auto* lastUpdateValues = lastUpdateColumn->mutable_duration();

            auto now = TActivationContext::Now();
            for (const auto& [id, shard]: Shards_) {
                auto idStr = ToString(id);
                if (auto* ref = idValues->add_values()) {
                    ref->set_title(idStr);
                    ref->set_page("/shards");
                    ref->set_args("id=" + idStr);
                }

                projectValues->add_values(shard.ShardKey.ProjectId);
                clusterValues->add_values(shard.ShardKey.ClusterName);
                serviceValues->add_values(shard.ShardKey.ServiceName);
                lastUpdateValues->add_values((now - shard.LastUpdate).GetValue());
            }
        }

        Send(ev->Sender, MakeHolder<NSelfMon::TEvPageDataResp>(std::move(page)));
    }

    void OnGetIndexLimiterState(TIndexLimiterEvents::TReportLimiterState::TPtr& ev) {
        IndexLimiterData_.SetState(std::move(ev->Get()->State));
    }

    void OnWalInitialized(const NWal::TWalEvents::TWalInitialized::TPtr& ev) {
        IsWalInitialized_ = true;
        /**
         * first we say it to all shards
         */
        for (const auto& [numId, shard]: Shards_) {
            if (shard.ShardActor) {
                Send(shard.ShardActor, new NWal::TWalEvents::TWalInitialized{});
            }
        }
        /**
         * now we can say to write handlers,
         * it is guaranteed that any message from them will be processed by shard in correct state
         */
        for (const auto& fwd: ev->Get()->ForwardTo) {
            Send(fwd, new NWal::TWalEvents::TWalInitialized{});
        }
    }

    void OnPoison(TEvents::TEvPoison::TPtr& ev) {
        MON_INFO(Index, LOG_P << "index is dying");

        Send(IndexLimiterId_, new TIndexLimiterEvents::TUnsubscribe{SelfId()});

        std::set<TActorId> toPoison;
        for (const auto& [_, shard]: Shards_) {
            toPoison.insert(shard.ShardActor);
        }

        PoisonAll(ev, std::move(toPoison));
        PassAway();
    }

private:
    TShardManagerConfig Config_;
    TActorId SnapshotCollector_;
    ui64 Generation_ = 1;
    THashMap<TNumId, TShardData> Shards_;
    std::shared_ptr<TMetrics> Metrics_;
    std::shared_ptr<IIndexWriteLimiter> IndexLimiter_;
    NActors::TActorId IndexLimiterId_;
    TIndexLimiterData IndexLimiterData_;
    bool IsWalInitialized_{false};
};

} // namespace

std::unique_ptr<IActor> CreateShardManager(
        TShardManagerConfig config,
        NActors::TActorId indexLimiterId,
        std::shared_ptr<IIndexWriteLimiter> indexLimiter,
        std::shared_ptr<NMonitoring::TMetricRegistry> metrics)  // NOLINT(performance-unnecessary-value-param): false positive
 {
    return std::make_unique<TShardManager>(std::move(config), indexLimiterId, std::move(indexLimiter), std::move(metrics));
}

} // namespace NSolomon::NMemStore::NIndex
