#pragma once

#include "shard_manager_config.h"
#include "write_request.h"

#include <solomon/services/memstore/lib/event_slots.h>
#include <solomon/services/memstore/lib/index/index_limiter.h>

#include <solomon/libs/cpp/actors/events/events.h>
#include <solomon/libs/cpp/actors/fwd.h>
#include <solomon/libs/cpp/error_or/error_or.h>

#include <library/cpp/actors/core/actorid.h>
#include <library/cpp/actors/core/event_local.h>

namespace NSolomon::NMemStore::NIndex {

class TShardManagerEvents: private TEventSlot<EEventSpace::MemStore, ES_SHARD_MANAGER> {
    enum {
        IndexBatch = SpaceBegin,
        EnableSnapshots,
        ListShards,
        ListShardsResponse,
        ShardIsEmpty,
        FindShard,
        FindShardResponse,
        End,
    };
    static_assert(End < SpaceEnd, "too many event types");

public:
    /**
     * Request to index new data.
     */
    struct TIndexBatch: public NActors::TEventLocal<TIndexBatch, IndexBatch> {
        /**
         * Identifier of the WAL record in which this data is persisted.
         */
        TLogId LogId;

        /**
         * Requests for indexing.
         */
        THashMap<TNumId, TShardAddPointRequests> Requests;

        explicit TIndexBatch(TLogId logId, THashMap<TNumId, TShardAddPointRequests> requests = {})
            : LogId(logId)
            , Requests(std::move(requests))
        {
        }
    };

    /**
     * Start writing snapshots.
     */
    struct TEnableSnapshots: public NActors::TEventLocal<TEnableSnapshots, EnableSnapshots> {
        /**
         * Where to send snapshots. `NWal::TWalEvents::TSnapshot` events will be sent here.
         */
        NActors::TActorId Collector;

        explicit TEnableSnapshots(NActors::TActorId collector)
            : Collector(collector)
        {
        }
    };

    /**
     * A 'list shards' request.
     */
    struct TListShards: public NActors::TEventLocal<TListShards, ListShards> {
        ui64 Generation;

        explicit TListShards(ui64 generation) noexcept
            : Generation(generation)
        {
        }
    };

    /**
     * A response for the 'list shards' request.
     */
    struct TListShardsResponse: public NActors::TEventLocal<TListShardsResponse, ListShardsResponse> {
        ui64 Generation;
        std::vector<std::pair<TNumId, TShardKey>> Shards;

        explicit TListShardsResponse(ui64 generation, std::vector<std::pair<TNumId, TShardKey>> shards) noexcept
            : Generation{generation}
            , Shards(std::move(shards))
        {
        }
    };

    /**
     * Asking ShardManager to tell us if this numId is processed by this exact node
     */
    struct TFindShard: public NActors::TEventLocal<TFindShard, FindShard> {
        TNumId NumId;

        explicit TFindShard(TNumId numId) noexcept
            : NumId{numId}
        {
        }
    };

    struct TFindShardResponse: public NActors::TEventLocal<TFindShardResponse, FindShardResponse> {
        NActors::TActorId ShardActor;

        explicit TFindShardResponse(NActors::TActorId shardActor)
            : ShardActor{shardActor}
        {
        }
    };
};

/**
 * Shard manager for in-memory time series storage.
 *
 * @param config   settings of shard manager.s
 */
std::unique_ptr<NActors::IActor> CreateShardManager(
        TShardManagerConfig config,
        NActors::TActorId indexLimiterId,
        std::shared_ptr<NSolomon::NMemStore::NIndex::IIndexWriteLimiter> indexLimiter,
        std::shared_ptr<NMonitoring::TMetricRegistry> metrics);

} // namespace NSolomon::NMemStore::NIndex
