#include "snapshot.h"
#include "metrics.h"
#include "shard.h"
#include "subshard.h"

#include <solomon/libs/cpp/slog/unresolved_meta/builder.h>
#include <solomon/libs/cpp/slog/snapshot_data/builder.h>
#include <solomon/libs/cpp/steady_timer/steady_timer.h>

#include <library/cpp/actors/core/actor.h>
#include <library/cpp/actors/core/hfunc.h>

using namespace NActors;

namespace NSolomon::NMemStore::NIndex {
namespace {

class TSnapshotActor: public TActor<TSnapshotActor> {
public:
    TSnapshotActor(
            TActorId shard,
            TNumId numId,
            TFrameIdx frameIdx,
            TLogId logId,
            size_t numSubShards,
            std::shared_ptr<IResourceUsageContext> shardMeteringContext,
            std::shared_ptr<TMetrics> metrics)
        : TActor<TSnapshotActor>{&TThis::StateFunc}
        , Shard_{shard}
        , NumId_{numId}
        , FrameIdx_{frameIdx}
        , LogId_{logId}
        , NumSubShards_{numSubShards}
        , ShardMeteringContext_{std::move(shardMeteringContext)}
        , Metrics_(std::move(metrics))
    {
        MetaBuilder_ = NSlog::NUnresolvedMeta::CreateUnresolvedMetaBuilder(
                NumId_,
                NMonitoring::ECompression::LZ4,
                &MetaStream_);

        DataBuilder_ = NSlog::NSnapshotData::CreateSnapshotLogDataBuilder(
                NumId_,
                NMonitoring::ECompression::LZ4,
                &DataStream_);

        Y_VERIFY(numSubShards > 0);
        Metrics_->SnapshotInflight->Inc();
    }

    STATEFN(StateFunc) {
        Timer_.Reset();

        Y_DEFER {
            if (ShardMeteringContext_) {
                ShardMeteringContext_->AddCpuTime(Timer_.Step());
            }
        };

        switch (ev->GetTypeRewrite()) {
            hFunc(TSubShardEvents::TSealAndSnapshotFrameDone, OnSealAndSnapshotFrameDone)
        }
    }

public:
    void OnSealAndSnapshotFrameDone(TSubShardEvents::TSealAndSnapshotFrameDone::TPtr& ev) {
        auto& snapshots = ev->Get()->Snapshots;

        for (auto& snapshot: snapshots) {
            if (snapshot.Data.NumPoints() == 0) {
                continue;
            }

            auto size = DataBuilder_->OnTimeSeries(
                snapshot.Data.Type(),
                snapshot.Data.Columns().ToColumnSet(snapshot.Data.Type()).Mask(),
                snapshot.Data.Data(),
                snapshot.Data.NumPoints());
            MetaBuilder_->OnMetric(
                snapshot.Data.Type(),
                std::move(snapshot.Labels),
                snapshot.Data.NumPoints(),
                size);
        }

        if (--NumSubShards_ == 0) {
            MetaBuilder_->Close();
            MetaStream_.Flush();
            DataBuilder_->Close();
            DataStream_.Flush();

            Send(Shard_, MakeHolder<TShardEvents::TSnapshotDone>(
                FrameIdx_,
                LogId_,
                std::move(Meta_),
                std::move(Data_)));

            Metrics_->SnapshotComplete->Inc();
            Metrics_->SnapshotInflight->Dec();

            PassAway();
        }
    }

private:
    TActorId Shard_;
    TNumId NumId_;
    TFrameIdx FrameIdx_;
    TLogId LogId_;
    size_t NumSubShards_;
    std::shared_ptr<IResourceUsageContext> ShardMeteringContext_;
    std::shared_ptr<TMetrics> Metrics_;
    TSteadyTimer Timer_{};

    TString Meta_;
    TStringOutput MetaStream_{Meta_};
    NSlog::NUnresolvedMeta::IUnresolvedMetaBuilderPtr MetaBuilder_;

    TString Data_;
    TStringOutput DataStream_{Data_};
    NSlog::NSnapshotData::ISnapshotLogDataBuilderPtr DataBuilder_;
};

} // namespace

std::unique_ptr<IActor> CreateSnapshotActor(
        TActorId shard,
        TNumId numId,
        TFrameIdx frameIdx,
        TLogId logId,
        size_t numSubShards,
        std::shared_ptr<TMetrics> metrics,
        std::shared_ptr<IResourceUsageContext> shardMeteringContext)
{
    return std::make_unique<TSnapshotActor>(shard, numId, frameIdx, logId, numSubShards, std::move(shardMeteringContext), std::move(metrics));
}

} // NSolomon::NMemStore::NIndex
