#include "subshard.h"
#include "shard.h"
#include "metrics.h"
#include "shard_manager.h"

#include <solomon/services/memstore/lib/labels/labels_pool.h>
#include <solomon/services/memstore/lib/labels/match.h>

#include <solomon/libs/cpp/actors/events/common.h>
#include <solomon/libs/cpp/logging/logging.h>
#include <solomon/libs/cpp/selfmon/selfmon.h>
#include <solomon/libs/cpp/shard_metrics/resource_context.h>
#include <solomon/libs/cpp/steady_timer/steady_timer.h>
#include <solomon/libs/cpp/ts_model/visit.h>
#include <solomon/libs/cpp/ts_model/points.h>

#include <library/cpp/containers/absl_flat_hash/flat_hash_map.h>
#include <library/cpp/string_utils/quote/quote.h>

using namespace NActors;

namespace NSolomon::NMemStore::NIndex {

using NLabels::TLabels;

namespace {
constexpr TMetricId InvalidMetricId = Max<TMetricId>();
struct TMetric {
    TMetricId Id;
    NLabels::TLabels Label;
    TTimeSeriesPtr Series;
};

class TSubShard: public TActor<TSubShard> {
#define LOG_P "{subshard " << SelfId() << " (" << NumId_ << ":" << static_cast<ui32>(SubShardIndex_) << ")} "
public:
    TSubShard(
            TNumId numId,
            ui8 subShardIndex,
            TActorId ftsIndex,
            TActorId indexLimiterId,
            std::shared_ptr<IResourceUsageContext> shardMeteringContext,
            std::shared_ptr<TMetrics> metrics)
        : TActor<TSubShard>(&TThis::StateFunc)
        , NumId_{numId}
        , SubShardIndex_{subShardIndex}
        , FtsIndex_(ftsIndex)
        , ShardMeteringContext_{std::move(shardMeteringContext)}
        , PtrMetrics_(std::move(metrics))
        , IndexLimiterId_(indexLimiterId)
    {
        PtrMetrics_->StorageSubShards->Inc();
    }

    STATEFN(StateFunc) {
        Timer_.Reset();

        Y_DEFER {
            if (ShardMeteringContext_) {
                ShardMeteringContext_->AddCpuTime(Timer_.Step());
            }
        };

        switch (ev->GetTypeRewrite()) {
            hFunc(TSubShardEvents::TWriteToFrame, OnWriteToFrame)
            hFunc(TSubShardEvents::TSealAndSnapshotFrame, OnSealAndSnapshotFrame)
            hFunc(TSubShardEvents::TAddFrame, OnAddFrame)
            hFunc(TSubShardEvents::TDropFrame, OnDropFrame)
            hFunc(TSubShardEvents::TFindMetrics, OnFindMetrics)
            hFunc(TSubShardEvents::TReadMetricsByLabels, OnReadMetricsByLabels)
            hFunc(TSubShardEvents::TReadMetricsByMetricId, OnReadMetricsByMetricId)
            hFunc(NSelfMon::TEvPageDataReq, OnSelfMon)
            hFunc(TShardEvents::TMetricsTableRequest, OnTableRequest)
            hFunc(TIndexLimiterEvents::TReportLimiterState, OnGetIndexLimiterState)
            hFunc(TShardEvents::TMemoryMeteringRequest, OnMemoryMeteringRequest)
            sFunc(TShardEvents::TTsMemoryStatReport, OnTsMemoryStatReport);
            hFunc(TEvents::TEvPoison, OnPoison)
        }
    }

    void OnWriteToFrame(TSubShardEvents::TWriteToFrame::TPtr& ev) {
        size_t addedMetrics = 0;
        THolder<TFtsIndexEvents::TAdd> addEvent;
        TPointsRange range;

        for (auto& parser: ev->Get()->Parser) {
            PtrMetrics_->IncomingPointsPerTs->Record(parser.NumPoints());
            PtrMetrics_->GetTotalPointMetrics().WritePointsRate->Add(parser.NumPoints());
            PtrMetrics_->GetPointMetrics(parser.GetMetricType()).WritePointsRate->Add(parser.NumPoints());

            auto AddMetricToFtsEvent = [&](TMetricId metricId) {
                if (!addEvent) {
                    addEvent = MakeHolder<TFtsIndexEvents::TAdd>(ev->Get()->Frame);
                }
                addEvent->Metrics.emplace_back(
                    NSolomon::NLabels::TLabels::OwnedStrings(parser.GetLabels().begin(), parser.GetLabels().end()),
                    NMemStore::TMetricId{SubShardIndex_, metricId}
                );
                addedMetrics++;
            };
            const auto& labels = parser.GetLabels();
            NLabels::TLabels ownedLabels = TLabels::CopyLabelsTo(labels, *LabelsIntern_, *LabelsData_);

            const auto& [it, inserted] = Index_.try_emplace(ownedLabels, 0);
            if (inserted) {
                TTimeSeriesPtr ts = TTimeSeries::Create(parser.GetMetricType(), FramesBegin_, FramesSeal_, FramesEnd_);
                auto metricPosition = NewMetric(ownedLabels, std::move(ts));
                it->second = metricPosition;
            } else {
                for (auto label = ownedLabels.Iter(); label.HasValue(); label.Advance()) {
                    LabelsIntern_->Remove(label.Key());
                    LabelsIntern_->Remove(label.Value());
                }
                ownedLabels.Free(*LabelsData_);
            }
            auto& [metricId, _, ts] = Metrics_[it->second];

            try {
                if (ts->LastFrameIsEmpty()) {
                    AddMetricToFtsEvent(metricId);
                }
                if (ts->NeedsToConvert(parser.GetMetricType())) {
                    ts = ts->ConvertTo(parser.GetMetricType());
                }
                auto tsRange = ts->AddPoint(ev->Get()->Frame, parser);
                range.Update(tsRange);
            } catch (NSLog::TParsingError& e) {
                MON_WARN(Index, LOG_P << "point dropped: " << e.what());
                PtrMetrics_->WriteErrors->Inc();
                PtrMetrics_->WriteErrorsParsing->Inc();
            } catch (TTypeError& e) {
                MON_WARN(Index, LOG_P << "point dropped: " << e.what());
                PtrMetrics_->WriteErrors->Inc();
                PtrMetrics_->WriteErrorsTypeMismatch->Inc();
            } catch (...) {
                MON_WARN(Index, LOG_P << "point dropped: " << CurrentExceptionMessage());
            }
        }

        // TODO: check contention on these operations
        PtrMetrics_->StorageMetrics->Add(addedMetrics);
        ReportSizeBytes(false);

        if (addEvent) {
            addEvent->ReplyTo = ev->Sender;
            addEvent->ReplyCookie = ev->Cookie;
            addEvent->Range = range;
            addEvent->AllocatedSize = addEvent->SizeBytes();
            PtrMetrics_->MemoryFtsAdd->Add(addEvent->AllocatedSize);
            Send(FtsIndex_, std::move(addEvent));
        } else {
            Send(ev->Sender, MakeHolder<TSubShardEvents::TWriteToFrameDone>(range), 0, ev->Cookie);
        }
    }

    void OnSealAndSnapshotFrame(TSubShardEvents::TSealAndSnapshotFrame::TPtr& handle) {
        const auto& ev = handle->Get();
        auto frameIdx = ev->Frame;

        Y_VERIFY(frameIdx == FramesSeal_, "wrong frame to seal. frame id: %lu, expected: %lu", frameIdx, FramesSeal_);

        auto snapshot = MakeHolder<TSubShardEvents::TSealAndSnapshotFrameDone>();
        snapshot->Snapshots.reserve(Metrics_.size());
        for (auto& [metricId, labels, ts]: Metrics_) {
            if (metricId == InvalidMetricId) {
                continue;
            }
            ts->SealFrame(frameIdx);
            auto frame = ts->ReadFrame(frameIdx);
            if (frame.NumPoints() == 0) {
                continue;
            }

            NMonitoring::TLabels labelsHeavy;
            for (auto it = labels.Iter(); it.HasValue(); it.Advance()) {
                labelsHeavy.Add(LabelsIntern_->Find(it.Key()), LabelsIntern_->Find(it.Value()));
            }

            snapshot->Snapshots.push_back(
                TSubShardEvents::TSealAndSnapshotFrameDone::TSnapshot{std::move(labelsHeavy), std::move(frame)});
        }
        FramesSeal_++;
        ReportSizeBytes(false);
        Send(ev->ReplyTo, std::move(snapshot), 0, handle->Cookie);
    }

    void OnAddFrame(TSubShardEvents::TAddFrame::TPtr& ev) {
        auto frameIdx = ev->Get()->Frame;

        Y_VERIFY(frameIdx == FramesEnd_,
                 "frame mismatch on add. frame id to add: %lu, expected frame id: %lu", frameIdx, FramesEnd_);

        for (auto& metric: Metrics_) {
            if (metric.Id != InvalidMetricId) {
                metric.Series->AddFrame(frameIdx);
            }
        }

        FramesEnd_++;
        ReportSizeBytes(false);
    }

    void OnDropFrame(TSubShardEvents::TDropFrame::TPtr& handle) {
        const auto& ev = handle->Get();
        auto frameIdx = ev->Frame;
        Y_VERIFY(frameIdx == FramesBegin_, "wrong frame to drop. frame id: %lu, expected: %lu", frameIdx, FramesBegin_);
        FramesBegin_++;

        TVector<size_t> emptyMetrics;
        for (size_t pos = 0; pos < Metrics_.size(); ++pos) {
            auto& [metricId, _, ts] = Metrics_[pos];
            if (metricId == InvalidMetricId) {
                continue;
            }
            ts->DropFrame(frameIdx);
            if (ts->Empty()) {
                emptyMetrics.push_back(pos);
            }
        }
        FreeMetrics(emptyMetrics);
        CompactMetrics();
        ReportSizeBytes(false);
    }

    void OnFindMetrics(TSubShardEvents::TFindMetrics::TPtr& ev) {
        try {
            Y_VERIFY(ev->Get() != nullptr);
            auto& selectors = ev->Get()->Selectors;
            const auto& metricList = ev->Get()->MetricList;

            auto response = MakeHolder<TSubShardEvents::TFindMetricsResponse>(LabelsIntern_);

            for (auto idx: metricList) {
                size_t pos;
                if (auto it = MetricPosition_.find(idx); it == MetricPosition_.end()) {
                    MON_DEBUG(Index, LOG_P << "Metric #" << idx << " is deleted. Skipped.");
                    continue;
                } else {
                    pos = it->second;
                }
                auto&[id, labels, ts] = Metrics_[pos];
                Y_VERIFY(id != InvalidMetricId, "Cannot read deleted metric.");
                Y_VERIFY(NLabels::Matches(labels, *LabelsIntern_, selectors), "label doesn't match");
                response->Found.push_back({
                        ts->MetricType(),
                        labels.ToInternerAgnostic(*LabelsIntern_)});
            }
            Send(ev->Get()->ReplyTo, std::move(response), 0, ev->Cookie);
        } catch (const std::exception& e) {
            MON_ERROR(Index, LOG_P << "exception: " << e.what());
        } catch (...) {
            MON_ERROR(Index, LOG_P << "uncatched exception");
        }
    }

    void OnReadMetricsByLabels(TSubShardEvents::TReadMetricsByLabels::TPtr& ev) {
        auto response = MakeHolder<TSubShardEvents::TReadMetricsResponse>(LabelsIntern_);

        for (auto& labelsSrc: ev->Get()->Labels) {
            auto labels = NLabels::TLabelsOwned::Lookup(labelsSrc, *LabelsIntern_);
            if (labels.Empty()) {
                continue;
            }
            if (auto it = Index_.find(labels.GetRef()); it != Index_.end()) {
                auto [labels, pos] = *it;
                response->Found.push_back({
                    labels.ToInternerAgnostic(*LabelsIntern_),
                    Metrics_[pos].Series->ReadBetween(ev->Get()->WindowBegin, ev->Get()->WindowEnd)});
            }
        }

        Send(ev->Get()->ReplyTo, std::move(response), 0, ev->Cookie);
    }

    void OnReadMetricsByMetricId(TSubShardEvents::TReadMetricsByMetricId::TPtr& ev) {
        auto& idxs = ev->Get()->MetricIdxs;
        auto response = MakeHolder<TSubShardEvents::TReadMetricsResponse>(LabelsIntern_);

        for (auto idx: idxs) {
            size_t pos;
            if (auto it = MetricPosition_.find(idx); it == MetricPosition_.end()) {
                MON_DEBUG(Index, LOG_P << "Metric #" << idx << " too old. Skipped");
                continue;
            } else {
                pos = it->second;
            }
            auto& [id, labels, ts] = Metrics_[pos];
            Y_VERIFY(id != InvalidMetricId, "Cannot read deleted metric");
            response->Found.emplace_back(
                     labels.ToInternerAgnostic(*LabelsIntern_),
                     ts->ReadBetween(ev->Get()->WindowBegin, ev->Get()->WindowEnd));
        }
        Send(ev->Get()->ReplyTo, std::move(response), 0, ev->Cookie);
    }

    TString LabelsToString(const NLabels::TLabels& labels) {
        auto it = labels.Iter();
        TStringBuilder builder;
        builder << "{";
        while (it.HasValue()) {
            builder << LabelsIntern_->Find(it.Key()) << "=" << '"' << LabelsIntern_->Find(it.Value()) << '"';
            it.Advance();
            if (it.HasValue()) {
                builder << ", ";
            }
        }
        builder << "}";
        return builder;
    }

    yandex::monitoring::selfmon::Page MetricPage(ui32 metricId) {
        auto& [_, labels, ts] = Metrics_[MetricPosition_[metricId]];

        yandex::monitoring::selfmon::Page page;
        page.set_title(TStringBuilder{} << "Shard " << NumId_
                                        << ", SubShard " << static_cast<ui32>(SubShardIndex_)
                                        << ", Metric " << metricId);
        auto* grid = page.mutable_grid();
        if (auto* r = grid->add_rows()) {
            auto* val = r->add_columns()->mutable_component()->mutable_value();
            val->set_string(LabelsToString(labels));

            auto* t = grid->add_rows()->add_columns()->mutable_component()->mutable_table();
            t->set_numbered(true);

            auto* idColumn = t->add_columns();
            idColumn->set_title("Data");
            auto* idValues = idColumn->mutable_string();

            for (TFrameIdx i = ts->FrameIdxBegin(); i < ts->FrameIdxEnd(); ++i) {
                auto iterable = ts->ReadFrame(i);
                NSolomon::NTsModel::Visit(iterable.Type(), [idValues, &iterable](auto traits) {
                    auto genericIt = iterable.Iterator();
                    auto& it = traits.DowncastIterator(*genericIt);
                    auto point = traits.MakePoint();
                    while (it.NextPoint(&point)) {
                        idValues->add_values(ToString(point));
                    }
                });
            }
        }
        return page;
    }

    yandex::monitoring::selfmon::Table MetricsTable(const TSelectors& selectors, const TString& query, size_t limit) {
        yandex::monitoring::selfmon::Table table;
        table.set_numbered(true);

        auto* idColumn = table.add_columns();
        idColumn->set_title("Id");
        auto* idValues = idColumn->mutable_reference();

        auto* typeColumn = table.add_columns();
        typeColumn->set_title("Type");
        auto* typeValues = typeColumn->mutable_string();

        auto* labelsColumn = table.add_columns();
        labelsColumn->set_title("Labels");
        auto* labelsValues = labelsColumn->mutable_string();

        size_t i = 0;
        for (const auto& [id, pos]: MetricPosition_) {
            if (i++ == limit) {
                break;
            }

            auto& [_, labels, ts] = Metrics_[pos];
            if (NLabels::Matches(labels, *LabelsIntern_, selectors)) {
                auto idStr = ToString(id);
                if (auto* ref = idValues->add_values()) {
                    ref->set_title(idStr);
                    ref->set_page("/shards");
                    ref->set_args(query + "&metricId=" + idStr);
                }

                typeValues->add_values(ToString(ts->MetricType()));
                labelsValues->add_values(LabelsToString(labels));
            }
        }
        return table;
    }

    void OnTableRequest(TShardEvents::TMetricsTableRequest::TPtr& ev) {
        auto table = MetricsTable(std::move(ev->Get()->Selectors), std::move(ev->Get()->Query), ev->Get()->Limit);
        Send(ev->Sender, MakeHolder<TShardEvents::TMetricsTableResponse>(SubShardIndex_, std::move(table)));
    }

    void OnSelfMon(NSelfMon::TEvPageDataReq::TPtr& ev) {
        if (auto metricIdStr = ev->Get()->Param("metricId")) {
            ui32 metricId = 0;
            if (TryFromString(metricIdStr, metricId)) {
                Send(ev->Sender, MakeHolder<NSelfMon::TEvPageDataResp>(MetricPage(metricId)));
                return;
            }
        }

        size_t limit = 50;
        TryFromString(ev->Get()->Param("limit"), limit);

        TErrorOr<TSelectors, TInvalidSelectorsFormat> selectors = TSelectors{};
        auto selectorsStr = TString{ev->Get()->Param("selectors")};
        if (selectorsStr) {
            CGIUnescape(selectorsStr);
            if (!selectorsStr.StartsWith('{')) {
                selectorsStr = "{" + selectorsStr + "}";
            }
            try {
                selectors = ParseSelectors(selectorsStr);
            } catch (TInvalidSelectorsFormat& err) {
                selectors = err;
            }
        }

        yandex::monitoring::selfmon::Page page;
        page.set_title(TStringBuilder{} << "Shard " << NumId_ );

        auto* grid = page.mutable_grid();
        if (auto* r = grid->add_rows()) {
            auto* obj = r->add_columns()->mutable_component()->mutable_object();
            if (auto* f = obj->add_fields()) {
                f->set_name("Id");
                f->mutable_value()->set_uint32(NumId_);
            }
            if (auto* f = obj->add_fields()) {
                f->set_name("SubShardIndex");
                f->mutable_value()->set_uint32(SubShardIndex_);
            }
            if (auto* f = obj->add_fields()) {
                f->set_name("FramesBegin");
                f->mutable_value()->set_uint64(FramesBegin_);
            }
            if (auto* f = obj->add_fields()) {
                f->set_name("FramesSeal");
                f->mutable_value()->set_uint64(FramesSeal_);
            }
            if (auto* f = obj->add_fields()) {
                f->set_name("FramesEnd");
                f->mutable_value()->set_uint64(FramesEnd_);
            }

            if (auto* f = obj->add_fields()) {
                f->set_name("MetricCount");
                f->mutable_value()->set_uint64(MetricCount());
            }

            auto* form = r->add_columns()->mutable_component()->mutable_form();
            form->set_method(yandex::monitoring::selfmon::FormMethod::Get);

            {
                auto* input = form->add_items()->mutable_input();
                input->set_type(yandex::monitoring::selfmon::InputType::Hidden);
                input->set_name("subId");
                input->set_value(TString(ev->Get()->Param("subId")));
            }

            {
                auto* input = form->add_items()->mutable_input();
                input->set_type(yandex::monitoring::selfmon::InputType::Hidden);
                input->set_name("id");
                input->set_value(TString(ev->Get()->Param("id")));
            }

            {
                auto* item = form->add_items();
                item->set_label("Filter");
                item->set_help("e.g. host=solomon-fetcher-sas-00, sensor=memory_total");
                auto* input = item->mutable_input();
                input->set_type(yandex::monitoring::selfmon::InputType::Text);
                input->set_name("selectors");
                input->set_value(selectorsStr);
            }

            auto* submit = form->add_submit();
            submit->set_title("Filter");
        }

        if (auto* r = grid->add_rows()) {
            auto* h = r->add_columns()->mutable_component()->mutable_heading();
            if (selectors.Success()) {
                h->set_content("Metrics");
            } else {
                h->set_content(FormatExc(selectors.Error()));
            }
            h->set_level(3);
        }

        if (selectors.Success()) {
            if (auto* r = grid->add_rows()) {
                auto* t = r->add_columns()->mutable_component()->mutable_table();
                *t = MetricsTable(selectors.Value(), TString{ev->Get()->Query}, limit);
            }
        }

        Send(ev->Sender, MakeHolder<NSelfMon::TEvPageDataResp>(std::move(page)));
    }

    /**
     * Create new TMetric with labels and ts
     * @return position of new metric in Metrics_ array
     */
    size_t NewMetric(NLabels::TLabels labels, TTimeSeriesPtr ts) {
        size_t metricPosition;
        if (FreePosition_.empty()) {
            metricPosition = Metrics_.size();
            Metrics_.push_back({LastMetricId_, labels, std::move(ts)});
        } else {
            metricPosition = FreePosition_.back();
            FreePosition_.pop_back();
            Metrics_[metricPosition] = {LastMetricId_, labels, std::move(ts)};
        }
        MetricPosition_[LastMetricId_] = metricPosition;
        LastMetricId_++;
        Y_VERIFY(Metrics_[metricPosition].Series != nullptr, "Cannot initialize time series with nullptr");
        return metricPosition;
    }

    void FreeMetrics(const TVector<size_t>& metricId) {
        for (auto emptyMetricPos: metricId) {
            auto& [id, labels, ts] = Metrics_[emptyMetricPos];
            MetricPosition_.erase(id);
            id = InvalidMetricId;
            ts.Drop();
            for (auto iter = labels.Iter(); iter.HasValue(); iter.Advance()) {
                LabelsIntern_->Remove(iter.Key());
                LabelsIntern_->Remove(iter.Value());
            }
            Index_.erase(labels);
            labels.Free(*LabelsData_);
            FreePosition_.push_back(emptyMetricPos);
        }
        PtrMetrics_->StorageMetrics->Add(-static_cast<i64>(metricId.size()));
    }

    size_t MetricCount() const {
        return Metrics_.size() - FreePosition_.size();
    }

    void OnGetIndexLimiterState(TIndexLimiterEvents::TReportLimiterState::TPtr& ev) {
        IndexLimiterData_.SetState(std::move(ev->Get()->State));
    }

    void OnMemoryMeteringRequest(TShardEvents::TMemoryMeteringRequest::TPtr& ev) {
        const ui64 occupiedMemory = TimeSeriesSize_ + LabelsInternalSize_ + LabelsDataSize_ + AuxDataSize_;
        Send(ev->Sender, new TShardEvents::TMemoryMeteringResponse(SubShardIndex_, occupiedMemory));
    }

    void OnTsMemoryStatReport() {
        ReportTsMemoryStat(false);
    }

    void OnPoison(TEvents::TEvPoison::TPtr& ev) {
        PtrMetrics_->StorageMetrics->Add(-static_cast<i64>(MetricCount()));
        PtrMetrics_->StorageSubShards->Dec();
        Send(IndexLimiterId_, new TIndexLimiterEvents::TUnsubscribe{SelfId()});
        Send(ev->Sender, new TEvents::TEvPoisonTaken);
        PassAway();
        ReportSizeBytes(true);
        ReportTsMemoryStat(true);
        ClearOnPoison();
    }

    void CompactMetrics() {
        for (auto& metric: Metrics_) {
            if (metric.Series) {
                metric.Series->Compact();
            }
        }
    }

    size_t CalcTimeSeriesSize() const {
        size_t size{0};
        for (const auto& metric: Metrics_) {
            if (metric.Series) {
                size += metric.Series->SizeBytes();
            }
        }
        return size;
    }

    size_t CalcStoredPointsCount() const {
        size_t count{0};
        for (const auto& metric: Metrics_) {
            if (metric.Series) {
                count += metric.Series->NumPoints();
            }
        }
        return count;
    }

    TTsMemoryStat CalcTsMemoryStat() const {
        TTsMemoryStat stat;
        for (const auto& metric: Metrics_) {
            if (metric.Series) {
                stat += metric.Series->CalcMemoryStat();
            }
        }
        return stat;
    }

    void ReportTimeSeriesSizeBytes(bool isDying) {
        const size_t tsSize = isDying ? 0 : CalcTimeSeriesSize();
        const i64 delta = static_cast<i64>(tsSize) - static_cast<i64>(TimeSeriesSize_);
        PtrMetrics_->MemoryTimeSeries->Add(delta);
        TimeSeriesSize_ = tsSize;
        IndexLimiterData_.AddTimeSeriesSizeBytes(delta);
    }

    void ReportStoredPointsCount(bool isDying) {
        const size_t pointsCount = isDying ? 0 : CalcStoredPointsCount();
        const i64 delta = static_cast<i64>(pointsCount) - static_cast<i64>(PointsCount_);
        PtrMetrics_->StoredPointsCount->Add(delta);
        PointsCount_ = pointsCount;
    }

    void ReportTsMemoryStat(bool isDying) {
        TTsMemoryStat stat = isDying ? TTsMemoryStat{} : CalcTsMemoryStat();
        const TTsMemoryStat delta = stat - TsMemoryStat_;

        auto& metrics = PtrMetrics_->TsMemoryMetrics;
        metrics.SealFramesCount->Add(delta.SealedFramesStat.FramesCount);
        metrics.SealFramePointsCount->Add(delta.SealedFramesStat.PointsCount);
        metrics.SealFrameDataSizeBytes->Add(delta.SealedFramesStat.DataSizeBytes);
        metrics.SealFrameBuffersSizeBytes->Add(delta.SealedFramesStat.BuffersSizeBytes);
        metrics.SealFramesSizeBytes->Add(delta.SealedFramesStat.FramesSizeBytes);
        metrics.SealFrameHoldersSizeBytes->Add(delta.SealedFramesStat.FrameHoldersSizeBytes);

        metrics.MutableFramesCount->Add(delta.MutableFramesStat.FramesCount);
        metrics.MutableFramePointsCount->Add(delta.MutableFramesStat.PointsCount);
        metrics.MutableFrameDataSizeBytes->Add(delta.MutableFramesStat.DataSizeBytes);
        metrics.MutableFrameBuffersSizeBytes->Add(delta.MutableFramesStat.BuffersSizeBytes);
        metrics.MutableFramesSizeBytes->Add(delta.MutableFramesStat.FramesSizeBytes);
        metrics.MutableFrameHoldersSizeBytes->Add(delta.MutableFramesStat.FrameHoldersSizeBytes);

        metrics.DequeCapacityBytes->Add(delta.DequeCapacityBytes);
        metrics.BuffersCacheSizeBytes->Add(delta.BuffersCacheSizeBytes);

        TsMemoryStat_ = stat;
    }

    void ReportLabelsSizeBytes(bool isDying) {
        const size_t labelsInternalSize = isDying ? 0 : LabelsIntern_->AllocatedBytes();
        const i64 labelsInternDelta = static_cast<i64>(labelsInternalSize) - static_cast<i64>(LabelsInternalSize_);
        PtrMetrics_->MemoryLabelsInternal->Add(labelsInternDelta);
        LabelsInternalSize_ = labelsInternalSize;

        const size_t labelsDataSize = isDying ? 0 : LabelsData_->SizeBytes();
        const i64 labelsDataDelta = static_cast<i64>(labelsDataSize) - static_cast<i64>(LabelsDataSize_);
        PtrMetrics_->MemoryLabelsData->Add(labelsDataDelta);
        LabelsDataSize_ = labelsDataSize;

        const size_t auxDataSize = isDying ? 0 : CalcAuxDataSizeBytes();
        const i64 auxDataDelta = static_cast<i64>(auxDataSize) - static_cast<i64>(AuxDataSize_);
        PtrMetrics_->MemorySubshardAux->Add(auxDataDelta);
        AuxDataSize_ = auxDataSize;

        const i64 labelsDelta = labelsInternDelta + labelsDataDelta + auxDataDelta;
        IndexLimiterData_.AddLabelsSizeBytes(labelsDelta);
    }

    size_t CalcAuxDataSizeBytes() const {
        return sizeof(*this)
                + Index_.capacity() * (sizeof(NLabels::TLabels) + sizeof(size_t))
                + MetricPosition_.capacity() * (sizeof(TMetricId) + sizeof(size_t))
                + FreePosition_.capacity() * sizeof(size_t);
    }

    void ReportSizeBytes(bool isDying) {
        ReportTimeSeriesSizeBytes(isDying);
        ReportStoredPointsCount(isDying);
        ReportLabelsSizeBytes(isDying);
        const bool needReply = !isDying;
        Send(IndexLimiterId_, new TIndexLimiterEvents::TAddParameterValues(IndexLimiterData_.DetachDelta()), needReply);
    }

    void ClearOnPoison() {
        // to free memory as soon as possible, because we don't know when actor lib will call destructor
        auto freePosition = std::move(FreePosition_);
        auto metrics = std::move(Metrics_);
        auto metricsPosition = std::move(MetricPosition_);
        auto index = std::move(Index_);
        LabelsData_.reset();
        LabelsIntern_.Drop();
    }

private:
    TNumId NumId_;
    ui8 SubShardIndex_;

    TFrameIdx FramesBegin_ = 0;
    TFrameIdx FramesSeal_ = 0;
    TFrameIdx FramesEnd_ = 0;

    TAtomicSharedPtr<NIntern::THashMapStringPool> LabelsIntern_ = MakeAtomicShared<NIntern::THashMapStringPool>();
    std::unique_ptr<NLabels::TLabelsPool> LabelsData_ = std::make_unique<NLabels::TLabelsPool>(1024);
    absl::flat_hash_map<NLabels::TLabels, size_t, THash<NLabels::TLabels>> Index_;
    absl::flat_hash_map<TMetricId, size_t> MetricPosition_;
    TVector<TMetric> Metrics_;
    size_t TimeSeriesSize_{0};
    size_t LabelsInternalSize_{0};
    size_t LabelsDataSize_{0};
    size_t AuxDataSize_{0};
    size_t PointsCount_{0};
    TTsMemoryStat TsMemoryStat_;
    TVector<size_t> FreePosition_;
    TMetricId LastMetricId_{0};
    TActorId FtsIndex_;
    std::shared_ptr<IResourceUsageContext> ShardMeteringContext_;
    std::shared_ptr<TMetrics> PtrMetrics_;
    TSteadyTimer Timer_{};
    TActorId IndexLimiterId_;
    TIndexLimiterData IndexLimiterData_;
};

} // namespace

std::unique_ptr<IActor> CreateSubShard(
        TNumId numId,
        ui8 index,
        TActorId ftsIndex,
        TActorId indexLimiterId,
        std::shared_ptr<TMetrics> metrics,
        std::shared_ptr<IResourceUsageContext> shardMeteringContext)
{
    return std::make_unique<TSubShard>(
            numId,
            index,
            ftsIndex,
            indexLimiterId,
            std::move(shardMeteringContext),
            std::move(metrics));
}

} // NSolomon::NMemStore::NIndex
