#pragma once

#include "index_limiter.h"
#include "metrics.h"

#include <solomon/services/memstore/lib/event_slots.h>
#include <solomon/services/memstore/lib/fts/index.h>
#include <solomon/services/memstore/lib/slog/parser.h>
#include <solomon/services/memstore/lib/time_series/time_series.h>
#include <solomon/services/memstore/lib/types.h>

#include <solomon/libs/cpp/actors/events/events.h>
#include <solomon/libs/cpp/actors/fwd.h>
#include <solomon/libs/cpp/intern/str_pool_hash_map.h>

#include <library/cpp/actors/core/event_local.h>

namespace NSolomon::NMemStore::NIndex {

using TMetricId = ui32;

class TSubShardEvents: private TEventSlot<EEventSpace::MemStore, ES_SUB_SHARD> {
    enum {
        WriteToFrame = SpaceBegin,
        WriteToFrameDone,
        SealAndSnapshotFrame,
        SealAndSnapshotFrameDone,
        AddFrame,
        DropFrame,
        FindMetrics,
        FindMetricsResponse,
        ReadMetricsByLabels,
        ReadMetricsByMetricId,
        ReadMetricsResponse,
        End,
    };
    static_assert(End < SpaceEnd, "too many event types");

public:
    struct TWriteToFrame: public NActors::TEventLocal<TWriteToFrame, WriteToFrame> {
        TFrameIdx Frame;
        TVector<NSLog::TTsParser> Parser;

        TWriteToFrame(TFrameIdx frame, TVector<NSLog::TTsParser> parser)
            : Frame(frame)
            , Parser(std::move(parser))
        {
        }
    };

    struct TWriteToFrameDone: public NActors::TEventLocal<TWriteToFrameDone, WriteToFrameDone> {
        TPointsRange Range;

        explicit TWriteToFrameDone(TPointsRange range)
            : Range(range)
        {
        }
    };

    struct TSealAndSnapshotFrame: public NActors::TEventLocal<TSealAndSnapshotFrame, SealAndSnapshotFrame> {
        NActors::TActorId ReplyTo;
        TFrameIdx Frame;

        TSealAndSnapshotFrame(NActors::TActorId replyTo, TFrameIdx frame)
            : ReplyTo{replyTo}
            , Frame{frame}
        {
        }
    };

    struct TSealAndSnapshotFrameDone: public NActors::TEventLocal<TSealAndSnapshotFrameDone, SealAndSnapshotFrameDone> {
        struct TSnapshot {
            NMonitoring::TLabels Labels;
            TTimeSeriesFrame Data;
        };

        TVector<TSnapshot> Snapshots;

        explicit TSealAndSnapshotFrameDone(TVector<TSnapshot> snapshot = {})
            : Snapshots(std::move(snapshot))
        {
        }
    };

    struct TAddFrame: public NActors::TEventLocal<TAddFrame, AddFrame> {
        TFrameIdx Frame;

        explicit TAddFrame(TFrameIdx frame)
            : Frame{frame}
        {
        }
    };

    struct TDropFrame: public NActors::TEventLocal<TDropFrame, DropFrame> {
        TFrameIdx Frame;

        explicit TDropFrame(TFrameIdx frame)
            : Frame{frame}
        {
        }
    };

    struct TFindMetrics: public NActors::TEventLocal<TFindMetrics, FindMetrics> {
        NActors::TActorId ReplyTo;

        // Should be sorted lexicographically by keys!
        TSelectors Selectors;

        // List of metrics matches to Selectors found with FTS
        TVector<ui32> MetricList;

        TFindMetrics(NActors::TActorId replyTo, TSelectors selectors)
            : ReplyTo{replyTo}
            , Selectors(std::move(selectors))
        {
        }

        TFindMetrics(NActors::TActorId replyTo, TSelectors selectors, TVector<ui32> metricList)
            : ReplyTo{replyTo}
            , Selectors(std::move(selectors))
            , MetricList(std::move(metricList))
        {
        }
    };

    struct TFindMetricsResponse: public NActors::TEventLocal<TFindMetricsResponse, FindMetricsResponse> {
        TAtomicSharedPtr<NIntern::THashMapStringPool> LabelsPool;
        TVector<std::pair<NTsModel::EPointType, NSolomon::NLabels::TLabels>> Found;

        explicit TFindMetricsResponse(TAtomicSharedPtr<NIntern::THashMapStringPool> labelsPool)
            : LabelsPool(std::move(labelsPool))
        {
        }
    };

    struct TReadMetricsByLabels: public NActors::TEventLocal<TReadMetricsByLabels, ReadMetricsByLabels> {
        NActors::TActorId ReplyTo;
        TVector<NSolomon::NLabels::TLabels> Labels;
        TInstant WindowBegin;
        TInstant WindowEnd;

        TReadMetricsByLabels(
                NActors::TActorId replyTo,
                TInstant windowBegin,
                TInstant windowEnd,
                TVector<NSolomon::NLabels::TLabels> labels = {})
            : ReplyTo{replyTo}
            , Labels{std::move(labels)}
            , WindowBegin{windowBegin}
            , WindowEnd{windowEnd}
        {
        }
    };

    struct TReadMetricsByMetricId: public NActors::TEventLocal<TReadMetricsByMetricId, ReadMetricsByMetricId> {
        NActors::TActorId ReplyTo;
        TVector<ui32> MetricIdxs;
        TInstant WindowBegin;
        TInstant WindowEnd;

        TReadMetricsByMetricId(NActors::TActorId replyTo, TVector<ui32> metricIdxs, TInstant windowBegin, TInstant windowEnd)
            : ReplyTo{replyTo}
            , MetricIdxs{std::move(metricIdxs)}
            , WindowBegin{windowBegin}
            , WindowEnd{windowEnd}
        {
        }
    };

    struct TReadMetricsResponse: public NActors::TEventLocal<TReadMetricsResponse, ReadMetricsResponse> {
        TAtomicSharedPtr<NIntern::THashMapStringPool> LabelsPool;
        TVector<std::pair<NSolomon::NLabels::TLabels, TTimeSeriesFrames>> Found;

        TReadMetricsResponse() = default;

        explicit TReadMetricsResponse(TAtomicSharedPtr<NIntern::THashMapStringPool> labelsPool)
            : LabelsPool(std::move(labelsPool))
        {
        }
    };
};

/**
 * In-memory storage for a single sub shard.
 *
 * @param numId             id of the shard.
 * @param index             unique index of this sub shard across other sub shards.
 * @param ftsEngine         full text search engine (shared between sub shards), optional
 * @param shardMeteringContext  object for collecting per-shard metrics
 */
std::unique_ptr<NActors::IActor> CreateSubShard(
        TNumId numId,
        ui8 index,
        NActors::TActorId ftsIndex,
        NActors::TActorId indexLimiterId,
        std::shared_ptr<TMetrics> metrics,
        std::shared_ptr<IResourceUsageContext> shardMeteringContext = nullptr);

} // namespace NSolomon::NMemStore::NIndex
