#include "unique_labels.h"
#include "metrics.h"
#include "shard.h"
#include "subshard.h"

#include <solomon/libs/cpp/steady_timer/steady_timer.h>
#include <solomon/libs/cpp/string_pool/string_pool.h>

#include <library/cpp/actors/core/actor.h>
#include <library/cpp/actors/core/hfunc.h>

using namespace NActors;

namespace NSolomon::NMemStore::NIndex {
namespace {

class TUniqueLabels: public TActor<TUniqueLabels> {
public:
    TUniqueLabels(
            std::vector<TActorId> subShards,
            TActorId fts,
            std::shared_ptr<IResourceUsageContext> shardMeteringContext,
            std::shared_ptr<TMetrics> metrics)
        : TActor<TUniqueLabels>{&TThis::StateFunc}
        , SubShards_{std::move(subShards)}
        , Fts_{fts}
        , ShardMeteringContext_{std::move(shardMeteringContext)}
        , Metrics_(std::move(metrics))
    {
        Y_VERIFY(SubShards_.size() > 0);
        Metrics_->ReadInflight->Inc();
    }

    STATEFN(StateFunc) {
        Timer_.Reset();

        Y_DEFER {
            if (ShardMeteringContext_) {
                ShardMeteringContext_->AddCpuTime(Timer_.Step());
            }
        };

        switch (ev->GetTypeRewrite()) {
            hFunc(TShardEvents::TUniqueLabels, OnUniqueLabelsReq);
            hFunc(TFtsIndexEvents::TFindResp, OnFtsResponse)
            hFunc(TSubShardEvents::TFindMetricsResponse, OnFindMetricsResponse)
        }
    }

public:
    void OnUniqueLabelsReq(TShardEvents::TUniqueLabels::TPtr& ev) {
        Req_ = ev;

        KeysFilter_.reserve(Req_->Get()->Keys.size());
        for (const auto& key: Req_->Get()->Keys) {
            KeysFilter_.insert(key);
        }

        // TODO: FTS index must handle special event for this type of request
        //       current implementation is completely wrong!
        Send(Fts_, new TFtsIndexEvents::TFindReq{Req_->Get()->Selectors, 0});
    }

    void OnFtsResponse(TFtsIndexEvents::TFindResp::TPtr& ev) {
        auto& metricList = ev->Get()->ResultIds;
        Y_VERIFY(metricList.size() == SubShards_.size(),
                 "Unexpected num of subshards from FTS index: %lu. Expected: %lu",
                 metricList.size(),
                 SubShards_.size());

        auto& selectors = Req_->Get()->Selectors;
        for (size_t subShardId = 0; subShardId < SubShards_.size(); ++subShardId) {
            if (auto& metricId = metricList[subShardId]; !metricId.empty()) {
                auto& subShard = SubShards_[subShardId];
                Send(subShard, new TSubShardEvents::TFindMetrics{SelfId(), selectors, std::move(metricId)});
                RequestsInFlight_++;
            }
        }

        if (RequestsInFlight_ == 0) {
            ReplyAndDie();
        }
    }

    void OnFindMetricsResponse(TSubShardEvents::TFindMetricsResponse::TPtr& ev) {
        for (auto& [type, labels]: ev->Get()->Found) {
            TLabels labelsIdx;
            for (auto [k, v]: labels) {
                TStringBuf key{k};
                if (KeysFilter_.empty() || KeysFilter_.contains(key)) {
                    TStringBuf value{v};
                    labelsIdx.emplace_back(Pool_.Put(key), Pool_.Put(value));
                }
            }
            UniqueLabels_.insert(std::move(labelsIdx));
        }

        if (--RequestsInFlight_ == 0) {
            ReplyAndDie();
        }
    }

    void ReplyAndDie() {
        auto* resp = new TShardEvents::TUniqueLabelsResponse{Pool_.Build(), std::move(UniqueLabels_)};
        Send(Req_->Sender, resp, 0, Req_->Cookie);

        Metrics_->ReadInflight->Dec();
        Metrics_->ReadComplete->Inc();

        PassAway();
    }

private:
    const std::vector<TActorId> SubShards_;
    const TActorId Fts_;
    std::shared_ptr<IResourceUsageContext> ShardMeteringContext_;
    std::shared_ptr<TMetrics> Metrics_;
    TSteadyTimer Timer_{};
    TShardEvents::TUniqueLabels::TPtr Req_;
    NStringPool::TStringPoolBuilder Pool_;
    absl::flat_hash_set<TStringBuf, THash<TStringBuf>> KeysFilter_;
    absl::flat_hash_set<TLabels, THash<TLabels>> UniqueLabels_;
    ui32 RequestsInFlight_{0};
};

} // namespace

std::unique_ptr<IActor> CreateUniqueLabelsActor(
        std::vector<TActorId> subShards,
        TActorId fts,
        std::shared_ptr<IResourceUsageContext> shardMeteringContext,
        std::shared_ptr<TMetrics> metrics)
{
    return std::make_unique<TUniqueLabels>(std::move(subShards), fts, std::move(shardMeteringContext), std::move(metrics));
}

} // NSolomon::NMemStore::NIndex
