#include "common.h"

#include <solomon/services/memstore/lib/index/shard.h>
#include <solomon/services/memstore/lib/wal/wal_events.h>

#include <solomon/libs/cpp/actors/events/common.h>
#include <solomon/libs/cpp/shard_metrics/ut/helpers.h>
#include <solomon/libs/cpp/stockpile_codec/metric_archive.h>
#include <solomon/libs/cpp/string_pool/string_pool.h>
#include <solomon/libs/cpp/ts_math/operations/downsampling.h>
#include <solomon/libs/cpp/ts_model/points.h>
#include <solomon/libs/cpp/ts_model/visit.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon;
using namespace NMemStore;
using namespace NMonitoring;
using namespace NIndex;
using namespace NActors;

template <typename TContainer>
std::vector<TString> ToStrings(const NStringPool::TStringPool& pool, const TContainer& ids) {
    std::vector<TString> strings;
    strings.reserve(ids.size());
    std::transform(ids.begin(), ids.end(), std::back_inserter(strings), [&](ui32 id) {
        return TString{pool[id]};
    });
    std::sort(strings.begin(), strings.end());
    return strings;
}

class Shard: public TIndexTestBase {
protected:
    void SetUp() override {
        TIndexTestBase::SetUp();

        ShardManagerId_ = Runtime->AllocateEdgeActor();
        LimiterId_ = Runtime->Register(CreateIndexLimiter(TIndexLimiterConfig::Default(), {}).release());
        ShardId_ = Runtime->Register(BuildShard().release());
        Runtime->WaitForBootstrap();
    }

    virtual std::unique_ptr<IActor> BuildShard() {
        auto indexLimiter = CreateIndexWriteLimiter(TIndexLimiterConfig::Default());
        return CreateShard(ShardManagerId_, 10, {}, LimiterId_, true, indexLimiter, {}, Metrics);
    }

    IEventHandle* CreateIndexHandle(TLogId logId, TString meta, TString data, TActorId replyTo) {
        TNumId numId = 10;
        auto* event = new TShardEvents::TIndex{
            logId,
            TShardAddPointRequests{
                numId,
                TShardKey{"project", "cluster", "service"},
                {{std::move(meta), std::move(data), replyTo, 0}}
            }
        };
        return new IEventHandle{ShardId_, {}, event};
    }

    TEvents::TEvPoisonTaken::TPtr PoisonShard() {
        auto poisoner = Runtime->AllocateEdgeActor();
        Runtime->Send(ShardId_, poisoner, MakeHolder<TEvents::TEvPoison>());
        return Runtime->GrabEdgeEvent<TEvents::TEvPoisonTaken>(poisoner);
    }

protected:
    TActorId ShardManagerId_;
    TActorId ShardId_;
    TActorId LimiterId_;
    TShardManagerConfig Config_{};
};

TEST_F(Shard, Destroy) {
    auto evResp = PoisonShard();
    ASSERT_TRUE(evResp);
}

TEST_F(Shard, DestroyNonEmpty) {
    auto [meta, data] = NSlog::MakeSlog(10)
        .Gauge({{"sensor", "a"}})
            .Add("2000-01-01T00:00:00Z", 5)
            .Add("2000-01-01T00:00:10Z", 10)
            .Add("2000-01-01T00:00:20Z", 15)
            .Done()
        .Gauge({{"sensor", "b"}})
            .Add("2000-01-01T00:00:00Z", 5)
            .Add("2000-01-01T00:00:10Z", 10)
            .Add("2000-01-01T00:00:20Z", 15)
            .Done()
        .Done();

    auto apiActor = Runtime->AllocateEdgeActor();
    Runtime->Send(CreateIndexHandle(TLogId{10, 0}, meta, data, apiActor));
    Runtime->GrabEdgeEvent<TShardEvents::TIndexDone>(apiActor);

    EXPECT_EQ(Metrics->StorageShards->Get(), 1);
    EXPECT_EQ(Metrics->StorageSubShards->Get(), 64);
    EXPECT_EQ(Metrics->StorageMetrics->Get(), 2);

    PoisonShard();

    EXPECT_EQ(Metrics->StorageShards->Get(), 0);
    EXPECT_EQ(Metrics->StorageSubShards->Get(), 0);
    EXPECT_EQ(Metrics->StorageMetrics->Get(), 0);
}

TEST_F(Shard, DestroyWritesInProgress) {
    auto apiActor = Runtime->AllocateEdgeActor();

    {
        auto [meta, data] = NSlog::MakeSlog(10)
            .Gauge({{"sensor", "a"}})
                .Add("2000-01-01T00:00:00Z", 5)
                .Done()
            .Gauge({{"sensor", "b"}})
                .Add("2000-01-01T00:00:00Z", 5)
                .Done()
            .Done();

        Runtime->Send(CreateIndexHandle(TLogId{10, 0}, meta, data, apiActor));
    }

    auto poisoner = Runtime->AllocateEdgeActor();
    Runtime->Send(ShardId_, poisoner, MakeHolder<TEvents::TEvPoison>());

    {
        auto [meta, data] = NSlog::MakeSlog(10)
            .Gauge({{"sensor", "c"}})
                .Add("2000-01-01T00:00:10Z", 5)
                .Done()
            .Gauge({{"sensor", "d"}})
                .Add("2000-01-01T00:00:20Z", 5)
                .Done()
            .Done();

        Runtime->Send(CreateIndexHandle(TLogId{10, 1}, meta, data, apiActor));
    }

    Runtime->GrabEdgeEvent<TShardEvents::TIndexDone>(apiActor);
    Runtime->GrabEdgeEvent<TShardEvents::TIndexDone>(apiActor);

    Runtime->GrabEdgeEvent<TEvents::TEvPoisonTaken>(poisoner);

    EXPECT_EQ(Metrics->StorageShards->Get(), 0);
    EXPECT_EQ(Metrics->StorageSubShards->Get(), 0);
    EXPECT_EQ(Metrics->StorageMetrics->Get(), 0);
}

TEST_F(Shard, AddPoint) {
    {
        auto [meta, data] = NSlog::MakeSlog(10)
            .Gauge({{"sensor", "a"}})
                .Add("2000-01-01T00:00:00Z", 5)
                .Add("2000-01-01T00:00:10Z", 10)
                .Add("2000-01-01T00:00:20Z", 15)
                .Done()
            .Gauge({{"sensor", "b"}})
                .Add("2000-01-01T00:00:00Z", 5)
                .Add("2000-01-01T00:00:10Z", 10)
                .Add("2000-01-01T00:00:20Z", 15)
                .Done()
            .Done();

        auto apiActor = Runtime->AllocateEdgeActor();
        Runtime->Send(CreateIndexHandle(TLogId{10, 0}, meta, data, apiActor));
        Runtime->GrabEdgeEvent<TShardEvents::TIndexDone>(apiActor);
    }

    EXPECT_EQ(Metrics->WriteComplete->Get(), 1u);
    EXPECT_EQ(Metrics->WriteErrors->Get(), 0u);
    EXPECT_EQ(Metrics->WriteErrorsParsing->Get(), 0u);
    EXPECT_EQ(Metrics->WriteErrorsTypeMismatch->Get(), 0u);
    EXPECT_EQ(Metrics->StorageShards->Get(), 1);
    EXPECT_EQ(Metrics->StorageSubShards->Get(), 64);
    EXPECT_EQ(Metrics->StorageFrames->Get(), 1);
    EXPECT_EQ(Metrics->StorageFramesOpen->Get(), 1);
    EXPECT_EQ(Metrics->StorageFramesSealed->Get(), 0);
    EXPECT_EQ(Metrics->StorageMetrics->Get(), 2);
    EXPECT_EQ(Metrics->StorageMetricsEmpty->Get(), 0);

    {
        auto [meta, data] = NSlog::MakeSlog(10)
            .Gauge({{"sensor", "a"}})
                .Add("2000-01-01T00:00:30Z", 20)
                .Done()
            .Gauge({{"sensor", "c"}})
                .Add("2000-01-01T00:00:00Z", 5)
                .Done()
            .Done();

        auto apiActor = Runtime->AllocateEdgeActor();
        Runtime->Send(CreateIndexHandle(TLogId{10, 1}, meta, data, apiActor));
        Runtime->GrabEdgeEvent<TShardEvents::TIndexDone>(apiActor);
    }

    EXPECT_EQ(Metrics->WriteComplete->Get(), 2u);
    EXPECT_EQ(Metrics->WriteErrors->Get(), 0u);
    EXPECT_EQ(Metrics->WriteErrorsParsing->Get(), 0u);
    EXPECT_EQ(Metrics->WriteErrorsTypeMismatch->Get(), 0u);
    EXPECT_EQ(Metrics->StorageShards->Get(), 1);
    EXPECT_EQ(Metrics->StorageSubShards->Get(), 64);
    EXPECT_EQ(Metrics->StorageFrames->Get(), 1);
    EXPECT_EQ(Metrics->StorageFramesOpen->Get(), 1);
    EXPECT_EQ(Metrics->StorageFramesSealed->Get(), 0);
    EXPECT_EQ(Metrics->StorageMetrics->Get(), 3);
    EXPECT_EQ(Metrics->StorageMetricsEmpty->Get(), 0);
}

TEST_F(Shard, AddPointParsingError) {
    auto [meta, data] = NSlog::MakeSlog(10)
        .Gauge({{"sensor", "a"}})
            .Add("2000-01-01T00:00:00Z", 5)
            .Done()
        .Gauge({{"sensor", "b"}})
            .Add("2000-01-01T00:00:00Z", 5)
            .Done()
        .Done();

    meta[0] = 'X';

    auto apiActor = Runtime->AllocateEdgeActor();
    Runtime->Send(CreateIndexHandle(TLogId{10, 0}, meta, data, apiActor));
    Runtime->GrabEdgeEvent<TShardEvents::TIndexDone>(apiActor);

    EXPECT_EQ(Metrics->WriteComplete->Get(), 1u);
    EXPECT_EQ(Metrics->WriteErrors->Get(), 1u);
    EXPECT_EQ(Metrics->WriteErrorsParsing->Get(), 1u);
    EXPECT_EQ(Metrics->WriteErrorsTypeMismatch->Get(), 0u);
    EXPECT_EQ(Metrics->StorageFrames->Get(), 1);
}

TEST_F(Shard, AddPointTypeError) {
    {
        auto [meta, data] = NSlog::MakeSlog(10)
            .Gauge({{"sensor", "a"}})
                .Add("2000-01-01T00:00:00Z", 5)
                .Done()
            .Done();

        auto apiActor = Runtime->AllocateEdgeActor();
        Runtime->Send(CreateIndexHandle(TLogId{10, 0}, meta, data, apiActor));
        Runtime->GrabEdgeEvent<TShardEvents::TIndexDone>(apiActor);
    }

    {
        auto [meta, data] = NSlog::MakeSlog(10)
            .DSummary({{"sensor", "a"}})
                .Add("2000-01-01T00:00:30Z", new NMonitoring::TSummaryDoubleSnapshot{5, 2, 3, 3, 2})
                .Done()
            .Done();

        auto apiActor = Runtime->AllocateEdgeActor();
        Runtime->Send(CreateIndexHandle(TLogId{10, 1}, meta, data, apiActor));
        Runtime->GrabEdgeEvent<TShardEvents::TIndexDone>(apiActor);
    }

    EXPECT_EQ(Metrics->WriteComplete->Get(), 2u);
    EXPECT_EQ(Metrics->WriteErrors->Get(), 1u);
    EXPECT_EQ(Metrics->WriteErrorsParsing->Get(), 0u);
    EXPECT_EQ(Metrics->WriteErrorsTypeMismatch->Get(), 1u);
    EXPECT_EQ(Metrics->StorageShards->Get(), 1);
    EXPECT_EQ(Metrics->StorageSubShards->Get(), 64);
    EXPECT_EQ(Metrics->StorageFrames->Get(), 1);
    EXPECT_EQ(Metrics->StorageFramesOpen->Get(), 1);
    EXPECT_EQ(Metrics->StorageFramesSealed->Get(), 0);
    EXPECT_EQ(Metrics->StorageMetrics->Get(), 1);
    EXPECT_EQ(Metrics->StorageMetricsEmpty->Get(), 0);
}

TEST_F(Shard, AddPointInconsistentWalId) {
    {
        auto [meta, data] = NSlog::MakeSlog(10)
            .Gauge({{"sensor", "a"}})
                .Add("2000-01-01T00:00:00Z", 5)
                .Done()
            .Done();

        auto apiActor = Runtime->AllocateEdgeActor();
        Runtime->Send(CreateIndexHandle(TLogId{10, 0}, meta, data, apiActor));
        Runtime->GrabEdgeEvent<TShardEvents::TIndexDone>(apiActor);
    }

    EXPECT_DEATH(
        ({
            auto [meta, data] = NSlog::MakeSlog(10)
                .Gauge({{"sensor", "a"}})
                    .Add("2000-01-01T00:00:30Z", 5)
                    .Done()
                .Done();

            auto apiActor = Runtime->AllocateEdgeActor();
            Runtime->Send(CreateIndexHandle(TLogId{10, 0}, meta, data, apiActor));
            Runtime->GrabEdgeEvent<TShardEvents::TIndexDone>(apiActor);
        }),
        "inconsistent log id");
}

TEST_F(Shard, AddPointTabletSwitch) {
    {
        auto [meta, data] = NSlog::MakeSlog(10)
            .Gauge({{"sensor", "a"}})
                .Add("2000-01-01T00:00:00Z", 5)
                .Done()
            .Done();

        auto apiActor = Runtime->AllocateEdgeActor();
        Runtime->Send(CreateIndexHandle(TLogId{10, 0}, meta, data, apiActor));
        Runtime->GrabEdgeEvent<TShardEvents::TIndexDone>(apiActor);
    }

    EXPECT_EQ(Metrics->StorageFrames->Get(), 1);
    EXPECT_EQ(Metrics->StorageFramesOpen->Get(), 1);
    EXPECT_EQ(Metrics->StorageFramesSealed->Get(), 0);

    {
        auto [meta, data] = NSlog::MakeSlog(10)
            .Gauge({{"sensor", "a"}})
                .Add("2000-01-01T00:00:00Z", 5)
                .Done()
            .Done();

        auto apiActor = Runtime->AllocateEdgeActor();
        Runtime->Send(CreateIndexHandle(TLogId{11, 0}, meta, data, apiActor));
        Runtime->GrabEdgeEvent<TShardEvents::TIndexDone>(apiActor);
    }

    EXPECT_EQ(Metrics->StorageFrames->Get(), 2);
    EXPECT_EQ(Metrics->StorageFramesOpen->Get(), 1);
    EXPECT_EQ(Metrics->StorageFramesSealed->Get(), 1);
}

TEST_F(Shard, AddPointAndSnapshotNoExternalPulse) {
    {
        auto [meta, data] = NSlog::MakeSlog(10)
            .Gauge({{"sensor", "a"}})
                .Add("2000-01-01T00:00:00Z", 5)
                .Done()
            .Done();

        auto apiActor = Runtime->AllocateEdgeActor();
        Runtime->Send(CreateIndexHandle(TLogId{10, 5}, meta, data, apiActor));
        Runtime->GrabEdgeEvent<TShardEvents::TIndexDone>(apiActor);
    }

    auto tsBegin = Runtime->GetCurrentTime();
    auto ev = Runtime->GrabEdgeEvent<NWal::TWalEvents::TSnapshot>(ShardManagerId_);

    EXPECT_GE(Runtime->GetCurrentTime() - tsBegin, TDuration::Minutes(5));
    EXPECT_EQ(ev->Get()->LogId, TLogId(10, 5));
    EXPECT_EQ(ev->Get()->NumId, 10u);
    EXPECT_THAT(ev->Get()->Meta, testing::Not(testing::IsEmpty()));
    EXPECT_THAT(ev->Get()->Data, testing::Not(testing::IsEmpty()));

    EXPECT_EQ(Metrics->StorageFrames->Get(), 1);
    EXPECT_EQ(Metrics->StorageFramesOpen->Get(), 0);
    EXPECT_EQ(Metrics->StorageFramesSealed->Get(), 1);
}

TEST_F(Shard, AddPointAndSnapshotWithExternalPulse) {
    {
        auto [meta, data] = NSlog::MakeSlog(10)
            .Gauge({{"sensor", "a"}})
                .Add("2000-01-01T00:00:00Z", 5)
                .Done()
            .Done();

        auto apiActor = Runtime->AllocateEdgeActor();
        Runtime->Send(CreateIndexHandle(TLogId{10, 5}, meta, data, apiActor));
        Runtime->GrabEdgeEvent<TShardEvents::TIndexDone>(apiActor);
    }

    EXPECT_EQ(Metrics->StorageFrames->Get(), 1);
    EXPECT_EQ(Metrics->StorageFramesOpen->Get(), 1);
    EXPECT_EQ(Metrics->StorageFramesSealed->Get(), 0);

    Runtime->AdvanceCurrentTime(TDuration::Minutes(2));

    {
        auto [meta, data] = NSlog::MakeSlog(10)
            .Gauge({{"sensor", "a"}})
                .Add("2000-01-01T00:00:10Z", 5)
                .Done()
            .Done();

        auto apiActor = Runtime->AllocateEdgeActor();
        Runtime->Send(CreateIndexHandle(TLogId{10, 6}, meta, data, apiActor));
        Runtime->GrabEdgeEvent<TShardEvents::TIndexDone>(apiActor);
    }

    EXPECT_EQ(Metrics->StorageFrames->Get(), 1);
    EXPECT_EQ(Metrics->StorageFramesOpen->Get(), 1);
    EXPECT_EQ(Metrics->StorageFramesSealed->Get(), 0);

    Runtime->AdvanceCurrentTime(TDuration::Minutes(2));

    {
        auto [meta, data] = NSlog::MakeSlog(10)
            .Gauge({{"sensor", "a"}})
                .Add("2000-01-01T00:00:10Z", 5)
                .Done()
            .Done();

        auto apiActor = Runtime->AllocateEdgeActor();
        Runtime->Send(CreateIndexHandle(TLogId{11, 2}, meta, data, apiActor));
        Runtime->GrabEdgeEvent<TShardEvents::TIndexDone>(apiActor);
    }

    {
        auto ev = Runtime->GrabEdgeEvent<NWal::TWalEvents::TSnapshot>(ShardManagerId_);

        EXPECT_EQ(ev->Get()->LogId, TLogId(10, 6));
        EXPECT_EQ(ev->Get()->NumId, 10u);
        EXPECT_THAT(ev->Get()->Meta, testing::Not(testing::IsEmpty()));
        EXPECT_THAT(ev->Get()->Data, testing::Not(testing::IsEmpty()));

        EXPECT_EQ(Metrics->StorageFrames->Get(), 2);
        EXPECT_EQ(Metrics->StorageFramesOpen->Get(), 1);
        EXPECT_EQ(Metrics->StorageFramesSealed->Get(), 1);
    }

    {
        auto ev = Runtime->GrabEdgeEvent<NWal::TWalEvents::TSnapshot>(ShardManagerId_);

        EXPECT_EQ(ev->Get()->LogId, TLogId(11, 2));
        EXPECT_EQ(ev->Get()->NumId, 10u);
        EXPECT_THAT(ev->Get()->Meta, testing::Not(testing::IsEmpty()));
        EXPECT_THAT(ev->Get()->Data, testing::Not(testing::IsEmpty()));

        EXPECT_EQ(Metrics->StorageFrames->Get(), 2);
        EXPECT_EQ(Metrics->StorageFramesOpen->Get(), 0);
        EXPECT_EQ(Metrics->StorageFramesSealed->Get(), 2);
    }
}

TEST_F(Shard, PointDroppedAfter35Minutes) {
    {
        auto [meta, data] = NSlog::MakeSlog(10)
            .Gauge({{"sensor", "a"}})
                .Add("2000-01-01T00:00:00Z", 5)
                .Done()
            .Done();

        auto apiActor = Runtime->AllocateEdgeActor();
        Runtime->Send(CreateIndexHandle(TLogId{10, 5}, meta, data, apiActor));
        Runtime->GrabEdgeEvent<TShardEvents::TIndexDone>(apiActor);
    }

    EXPECT_EQ(Metrics->StorageFrames->Get(), 1);
    EXPECT_EQ(Metrics->StorageFramesOpen->Get(), 1);
    EXPECT_EQ(Metrics->StorageFramesSealed->Get(), 0);
    EXPECT_EQ(Metrics->StorageMetrics->Get(), 1);
    EXPECT_EQ(Metrics->StorageMetricsEmpty->Get(), 0);

    Runtime->GrabEdgeEvent<NWal::TWalEvents::TSnapshot>(ShardManagerId_);
    Runtime->DispatchEvents({}, TDuration::Minutes(35));

    EXPECT_EQ(Metrics->StorageFrames->Get(), 0);
    EXPECT_EQ(Metrics->StorageFramesOpen->Get(), 0);
    EXPECT_EQ(Metrics->StorageFramesSealed->Get(), 0);
    EXPECT_EQ(Metrics->StorageMetrics->Get(), 0);
    EXPECT_LE(Metrics->StorageMetricsEmpty->Get(), 1);
}

class ShardWithFts: public Shard {
protected:
    std::unique_ptr<IActor> BuildShard() override {
        TShardManagerConfig config(
                TDuration::Minutes(5),
                TDuration::Minutes(30));
        auto indexLimiter = CreateIndexWriteLimiter(TIndexLimiterConfig::Default());
        return CreateShard(ShardManagerId_, 10, config, LimiterId_, true, indexLimiter,{}, Metrics);
    }
};

TEST_F(ShardWithFts, AddPoint) {
    {
        auto [meta, data] = NSlog::MakeSlog(10)
            .Gauge({{"sensor", "a"}})
                .Add("2000-01-01T00:00:00Z", 5)
                .Add("2000-01-01T00:00:10Z", 10)
                .Add("2000-01-01T00:00:20Z", 15)
                .Done()
            .Gauge({{"sensor", "q"}})
                .Add("2000-01-01T00:00:00Z", 5)
                .Add("2000-01-01T00:00:10Z", 10)
                .Add("2000-01-01T00:00:20Z", 15)
                .Done()
            .Done();

        auto apiActor = Runtime->AllocateEdgeActor();
        Runtime->Send(CreateIndexHandle(TLogId{10, 0}, meta, data, apiActor));
        Runtime->GrabEdgeEvent<TShardEvents::TIndexDone>(apiActor);

        // 64 subshards
        EXPECT_EQ(Metrics->StorageSubShards->Get(), 64);
    }
}

TEST_F(ShardWithFts, FindMetrics) {
    auto apiActor = Runtime->AllocateEdgeActor();
    {
        auto [meta, data] = NSlog::MakeSlog(10)
            .Gauge({{"host", "xxa"}, {"sensor", "a"}})
                .Add("2000-01-01T00:00:00Z", 5)
                .Add("2000-01-01T00:00:10Z", 10)
                .Add("2000-01-01T00:00:20Z", 15)
                .Done()
            .Gauge({{"host", "xxa"}, {"sensor", "q"}})
                .Add("2000-01-01T00:00:00Z", 5)
                .Add("2000-01-01T00:00:10Z", 10)
                .Add("2000-01-01T00:00:20Z", 15)
                .Done()
            .Gauge({{"host", "xxb"}, {"sensor", "q"}})
                .Add("2000-01-01T00:00:00Z", 5)
                .Add("2000-01-01T00:00:10Z", 10)
                .Add("2000-01-01T00:00:20Z", 15)
                .Done()
            .Gauge({{"host", "yya"}, {"sensor", "q"}})
                .Add("2000-01-01T00:00:00Z", 5)
                .Add("2000-01-01T00:00:10Z", 10)
                .Add("2000-01-01T00:00:20Z", 15)
                .Done()
            .Done();

        Runtime->Send(CreateIndexHandle(TLogId{10, 5}, meta, data, apiActor));
        Runtime->GrabEdgeEvent<TShardEvents::TIndexDone>(apiActor);
        Runtime->DispatchEvents({}, TDuration::Minutes(2));
    }

    {
        auto* evReq = new TShardEvents::TFind{10, ParseSelectors("{host=xx*}")};
        Runtime->Send(ShardId_, apiActor, THolder(evReq));

        auto evResp = Runtime->GrabEdgeEvent<TShardEvents::TFindResponse>(apiActor);
        auto* resp = evResp->Get();
        EXPECT_EQ(resp->Strings.Size(), 6u);
        EXPECT_EQ(resp->Metrics.size(), 3u);
        EXPECT_EQ(resp->TotalCount, 3u);
    }
}

TEST_F(ShardWithFts, ReadMetrics) {
    auto apiActor = Runtime->AllocateEdgeActor();
    {
        Runtime->UpdateCurrentTime(TInstant::ParseIso8601("2000-01-01T00:03:00Z"));
        auto [meta, data] = NSlog::MakeSlog(10)
            .Gauge({{"host", "xxa"}, {"sensor", "a"}})
                .Add("2000-01-01T00:02:10Z", 5)
                .Add("2000-01-01T00:02:20Z", 10)
                .Add("2000-01-01T00:02:30Z", 15)
            .Done()
            .Gauge({{"host", "xxa"}, {"sensor", "q"}})
                .Add("2000-01-01T00:02:10Z", 5)
                .Add("2000-01-01T00:02:20Z", 10)
                .Add("2000-01-01T00:02:30Z", 15)
            .Done()
            .Gauge({{"host", "xxb"}, {"sensor", "q"}})
                .Add("2000-01-01T00:02:10Z", 5)
                .Add("2000-01-01T00:02:20Z", 10)
                .Add("2000-01-01T00:02:30Z", 15)
            .Done()
            .Gauge({{"host", "yya"}, {"sensor", "q"}})
                .Add("2000-01-01T00:02:10Z", 5)
                .Add("2000-01-01T00:02:20Z", 10)
                .Add("2000-01-01T00:02:30Z", 15)
            .Done()
            .Done();

        Runtime->Send(CreateIndexHandle(TLogId{10, 5}, meta, data, apiActor));
        Runtime->GrabEdgeEvent<TShardEvents::TIndexDone>(apiActor);
        Runtime->DispatchEvents({}, TDuration::Minutes(2));
    }

    {
        auto req = std::make_unique<TShardEvents::TReadMany>();
        req->Lookup = std::make_unique<TShardEvents::TReadMany::TLookup>();
        req->Lookup->Selectors.Add("host", "xx*");
        req->Lookup->Limit = 100;

        Runtime->Send(ShardId_, apiActor, std::move(req));
        auto evResp = Runtime->GrabEdgeEvent<TShardEvents::TReadManyResponse>(apiActor);

        auto* resp = evResp->Get();
        EXPECT_EQ(resp->Metrics.size(), 3u);

        TSet<TVector<TString>> expectedLabels = {
                {"host", "xxa", "sensor", "a"},
                {"host", "xxa", "sensor", "q"},
                {"host", "xxb", "sensor", "q"},
        };
        TSet<TVector<TString>> labels;
        auto& strings = resp->Strings;

        for (const auto& metric: resp->Metrics) {
            TVector<TString> l;
            for (auto& label: *metric.Labels) {
                l.emplace_back(strings[label.Key()]);
                l.emplace_back(strings[label.Value()]);
            }
            labels.insert(l);

            NTs::TDoubleTsDecoder decoder{
                    metric.Columns.ToColumnSet(metric.Type),
                    metric.Data};

            NTsModel::TGaugePoint point;
            double expectedValue = 5.;
            int pointCounter = 0;
            while (decoder.NextPoint(&point)) {
                EXPECT_DOUBLE_EQ(point.Num, expectedValue);
                expectedValue += 5.;
                pointCounter++;
            }
            EXPECT_EQ(pointCounter, 3);
        }
        EXPECT_EQ(labels, expectedLabels);
    }
}

TEST_F(ShardWithFts, ReadMetricsCheckTime) {
    auto apiActor = Runtime->AllocateEdgeActor();
    const ui64 numOfRequests = 1;
    const ui64 numOfHots = 100'000;
    Runtime->UpdateCurrentTime(TInstant::ParseIso8601("2000-01-01T00:03:00Z"));

    {
        auto getMetaData = [&](const TString& suff) {
            auto log = NSlog::MakeSlog(10);
            for (ui64 i = 0 ; i < numOfHots; ++i) {
                log.Gauge({{"host", "xx" + ToString(i) + "_" + suff}})
                    .Add("2000-01-01T00:02:10Z", 5)
                    .Add("2000-01-01T00:02:20Z", 10)
                    .Add("2000-01-01T00:02:30Z", 15)
                    .Done();
            }
            return log.Done();
        };

        for (ui64 requestId = 0; requestId < numOfRequests; ++requestId) {
            auto[meta, data] = getMetaData(ToString(requestId));
            Runtime->Send(CreateIndexHandle(TLogId{10, 5 + requestId}, meta, data, apiActor));
        }

        auto [meta, data] = NSlog::MakeSlog(10)
                .Gauge({{"code", "300"}})
                    .Add("2000-01-01T00:02:10Z", 5)
                    .Add("2000-01-01T00:02:20Z", 10)
                    .Add("2000-01-01T00:02:30Z", 15)
                    .Done()
                .Done();

        Runtime->Send(CreateIndexHandle(TLogId{10, 5 + numOfRequests}, meta, data, apiActor));
        Runtime->GrabEdgeEvent<TShardEvents::TIndexDone>(apiActor);
        Runtime->DispatchEvents({}, TDuration::Minutes(2));
    }

    {
        // Big request
        auto req = std::make_unique<TShardEvents::TReadMany>();
        req->Lookup = std::make_unique<TShardEvents::TReadMany::TLookup>();
        req->Lookup->Selectors.Add("host", "xy*");
        req->Lookup->Limit = 100;

        Runtime->Send(ShardId_, apiActor, std::move(req));

        auto evResp = Runtime->GrabEdgeEvent<TShardEvents::TReadManyResponse>(apiActor);
        auto* resp = evResp->Get();

        EXPECT_EQ(0 * numOfHots * numOfRequests, resp->Metrics.size());
    }

    {
        // Small request
        auto req = std::make_unique<TShardEvents::TReadMany>();
        req->Lookup = std::make_unique<TShardEvents::TReadMany::TLookup>();
        req->Lookup->Selectors.Add("code", "3*");
        req->Lookup->Limit = 100;

        Runtime->Send(ShardId_, apiActor, std::move(req));

        auto evResp = Runtime->GrabEdgeEvent<TShardEvents::TReadManyResponse>(apiActor);
        auto* resp = evResp->Get();

        EXPECT_EQ(resp->Metrics.size(), 1u);
    }
}

TEST_F(ShardWithFts, ReadLabelValues) {
    auto apiActor = Runtime->AllocateEdgeActor();
    {
        Runtime->UpdateCurrentTime(TInstant::ParseIso8601("2000-01-01T00:03:00Z"));
        auto [meta, data] = NSlog::MakeSlog(10)
            .Gauge({{"host", "xxa"}, {"sensor", "a"}})
                .Add("2000-01-01T00:02:10Z", 5)
                .Add("2000-01-01T00:02:20Z", 10)
                .Add("2000-01-01T00:02:30Z", 15)
            .Done()
            .Gauge({{"host", "xxa"}, {"sensor", "q"}})
                .Add("2000-01-01T00:02:10Z", 5)
                .Add("2000-01-01T00:02:20Z", 10)
                .Add("2000-01-01T00:02:30Z", 15)
            .Done()
            .Gauge({{"host", "xxb"}, {"sensor", "q"}})
                .Add("2000-01-01T00:02:10Z", 5)
                .Add("2000-01-01T00:02:20Z", 10)
                .Add("2000-01-01T00:02:30Z", 15)
            .Done()
            .Gauge({{"host", "yya"}, {"sensor", "q"}})
                .Add("2000-01-01T00:02:10Z", 5)
                .Add("2000-01-01T00:02:20Z", 10)
                .Add("2000-01-01T00:02:30Z", 15)
            .Done()
            .Done();

        Runtime->Send(CreateIndexHandle(TLogId{10, 5}, meta, data, apiActor));
        Runtime->GrabEdgeEvent<TShardEvents::TIndexDone>(apiActor);
        Runtime->DispatchEvents({}, TDuration::Minutes(2));
    }

    {
        auto evReq = new TShardEvents::TLabelValues{ParseSelectors("{host=x*}"), {}, {}, 1000};
        Runtime->Send(ShardId_, apiActor, THolder(evReq));

        auto evResp = Runtime->GrabEdgeEvent<TShardEvents::TLabelValuesResponse>(apiActor);
        auto* resp = evResp->Get();

        EXPECT_EQ(resp->Strings.Size(), 6u);
        EXPECT_EQ(resp->Labels.size(), 2u);
        EXPECT_EQ(resp->MetricCount, 3u);

        for (const auto& label: resp->Labels) {
            if (resp->Strings[label.Key] == "sensor") {
                auto values = ToStrings(resp->Strings, label.Values);
                std::vector<TString> expected{"a", "q"};
                EXPECT_EQ(values, expected);
            } else if (resp->Strings[label.Key] == "host") {
                auto values = ToStrings(resp->Strings, label.Values);
                std::vector<TString> expected{"xxa", "xxb"};
                EXPECT_EQ(values, expected);
            } else {
                FAIL() << "Unexpected key: " << resp->Strings[label.Key];
            }
        }
    }
}

TEST_F(ShardWithFts, ReadLabelKeys) {
    auto apiActor = Runtime->AllocateEdgeActor();
    {
        Runtime->UpdateCurrentTime(TInstant::ParseIso8601("2000-01-01T00:03:00Z"));
        auto [meta, data] = NSlog::MakeSlog(10)
            .Gauge({{"host", "xxa"}, {"sensor", "a"}})
                .Add("2000-01-01T00:02:10Z", 5)
                .Add("2000-01-01T00:02:20Z", 10)
                .Add("2000-01-01T00:02:30Z", 15)
            .Done()
            .Gauge({{"host", "xxa"}, {"sensor", "q"}})
                .Add("2000-01-01T00:02:10Z", 5)
                .Add("2000-01-01T00:02:20Z", 10)
                .Add("2000-01-01T00:02:30Z", 15)
            .Done()
            .Gauge({{"host", "xxb"}, {"destination", "q"}})
                .Add("2000-01-01T00:02:10Z", 5)
                .Add("2000-01-01T00:02:20Z", 10)
                .Add("2000-01-01T00:02:30Z", 15)
            .Done()
            .Gauge({{"host", "yya"}, {"sensor", "q"}})
                .Add("2000-01-01T00:02:10Z", 5)
                .Add("2000-01-01T00:02:20Z", 10)
                .Add("2000-01-01T00:02:30Z", 15)
            .Done()
            .Done();

        Runtime->Send(CreateIndexHandle(TLogId{10, 5}, meta, data, apiActor));
        Runtime->GrabEdgeEvent<TShardEvents::TIndexDone>(apiActor);
        Runtime->DispatchEvents({}, TDuration::Minutes(2));
    }

    {
        auto evReq = new TShardEvents::TLabelKeys{ParseSelectors("{host=x*}")};
        Runtime->Send(ShardId_, apiActor, THolder(evReq));

        auto evResp = Runtime->GrabEdgeEvent<TShardEvents::TLabelKeysResponse>(apiActor);
        auto* resp = evResp->Get();

        std::vector<TString> keys = ToStrings(resp->Strings, resp->Keys);
        std::vector<TString> expected{"destination", "host", "sensor"};
        EXPECT_EQ(keys, expected);
    }

    {
        auto evReq = new TShardEvents::TLabelKeys{ParseSelectors("{host=xxa}")};
        Runtime->Send(ShardId_, apiActor, THolder(evReq));

        auto evResp = Runtime->GrabEdgeEvent<TShardEvents::TLabelKeysResponse>(apiActor);
        auto* resp = evResp->Get();

        std::vector<TString> keys = ToStrings(resp->Strings, resp->Keys);
        std::vector<TString> expected{"host", "sensor"};
        EXPECT_EQ(keys, expected);
    }
}

TEST_F(ShardWithFts, ReadDSummaryWithDownsampling) {
    using NMonitoring::TSummaryDoubleSnapshot;

    auto apiActor = Runtime->AllocateEdgeActor();
    {
        Runtime->UpdateCurrentTime(TInstant::ParseIso8601("2020-01-01T00:13:00Z"));
        auto [meta, data] = NSlog::MakeSlog(10)
            .DSummary({{"host", "xxa"}, {"sensor", "a"}})
                .Add("2020-01-01T00:02:10Z", MakeIntrusive<TSummaryDoubleSnapshot>(5., 2., 2., 2., 1u))
                .Add("2020-01-01T00:02:11Z", MakeIntrusive<TSummaryDoubleSnapshot>(7., 2., 2., 2., 1u))
                .Add("2020-01-01T00:12:10Z", MakeIntrusive<TSummaryDoubleSnapshot>(7., 2., 2., 2., 1u))
                .Add("2020-01-01T00:12:11Z", MakeIntrusive<TSummaryDoubleSnapshot>(5., 2., 2., 2., 1u))
            .Done()
            .Done();

        Runtime->Send(CreateIndexHandle(TLogId{10, 5}, meta, data, apiActor));
        Runtime->GrabEdgeEvent<TShardEvents::TIndexDone>(apiActor);
        Runtime->DispatchEvents({}, TDuration::Minutes(2));
    }

    {
        yandex::solomon::math::OperationDownsampling op;
        op.set_grid_millis(3000);

        auto req = std::make_unique<TShardEvents::TReadMany>();
        req->Lookup = std::make_unique<TShardEvents::TReadMany::TLookup>();
        req->Lookup->Selectors.Add("host", "*");
        req->Lookup->Limit = 100;
        req->Pipeline.Add(NTsMath::Downsampling(op));

        Runtime->Send(ShardId_, apiActor, std::move(req));

        auto evRes = Runtime->GrabEdgeEvent<TShardEvents::TReadManyResponse>(apiActor);
        auto* resp = evRes->Get();
        EXPECT_EQ(resp->Metrics.size(), 1u);

        auto& strings = resp->Strings;

        TSet<TVector<TString>> expectedLabels = {
            {"host", "xxa", "sensor", "a"},
        };
        TSet<TVector<TString>> labels;

        for (const auto& metric: resp->Metrics) {
            TVector<TString> l;
            for (auto& label: *metric.Labels) {
                l.emplace_back(strings[label.Key()]);
                l.emplace_back(strings[label.Value()]);
            }
            labels.insert(l);

            NTs::TSummaryDoubleTsDecoder decoder{
                    metric.Columns.ToColumnSet(metric.Type),
                    metric.Data};

            NTsModel::TDSummaryPoint point;
            double expectedSum = 7.;
            int pointCounter = 0;
            while (decoder.NextPoint(&point)) {
                EXPECT_EQ(point.Count, 2u);
                EXPECT_DOUBLE_EQ(point.Sum, expectedSum);
                expectedSum -= 2.;
                pointCounter++;
            }
            EXPECT_EQ(pointCounter, 2);
        }
        EXPECT_EQ(labels, expectedLabels);
    }
}

TEST_F(ShardWithFts, ReadMetricsWithResolvedKeys) {
    auto apiActor = Runtime->AllocateEdgeActor();
    {
        Runtime->UpdateCurrentTime(TInstant::ParseIso8601("2000-01-01T00:03:00Z"));
        auto [meta, data] = NSlog::MakeSlog(10)
            .Gauge({{"host", "xxa"}, {"sensor", "a"}})
                .Add("2000-01-01T00:02:10Z", 5)
                .Add("2000-01-01T00:02:20Z", 10)
                .Add("2000-01-01T00:02:30Z", 15)
            .Done()
            .Gauge({{"host", "xxa"}, {"sensor", "q"}})
                .Add("2000-01-01T00:02:10Z", 5)
                .Add("2000-01-01T00:02:20Z", 10)
                .Add("2000-01-01T00:02:30Z", 15)
            .Done()
            .Gauge({{"host", "xxb"}, {"sensor", "q"}})
                .Add("2000-01-01T00:02:10Z", 5)
                .Add("2000-01-01T00:02:20Z", 10)
                .Add("2000-01-01T00:02:30Z", 15)
            .Done()
            .Gauge({{"host", "yya"}, {"sensor", "q"}})
                .Add("2000-01-01T00:02:10Z", 5)
                .Add("2000-01-01T00:02:20Z", 10)
                .Add("2000-01-01T00:02:30Z", 15)
            .Done()
            .Done();

        Runtime->Send(CreateIndexHandle(TLogId{10, 5}, meta, data, apiActor));
        Runtime->GrabEdgeEvent<TShardEvents::TIndexDone>(apiActor);
        Runtime->DispatchEvents({}, TDuration::Minutes(2));
    }

    {
        auto req = std::make_unique<TShardEvents::TReadMany>();
        {
            req->Resolved = std::make_unique<TShardEvents::TReadMany::TResolved>();

            NStringPool::TStringPoolBuilder strings;
            ui32 hostId = strings.Put("host");
            ui32 xxaId = strings.Put("xxa");
            ui32 metricId = strings.Put("sensor");
            ui32 aId = strings.Put("a");
            ui32 qId = strings.Put("q");

            req->Resolved->Pool = strings.Build();
            req->Resolved->CommonLabels = {hostId, xxaId};
            req->Resolved->ResolvedKeys.push_back({metricId, aId});
            req->Resolved->ResolvedKeys.push_back({metricId, qId});
        }

        Runtime->Send(ShardId_, apiActor, std::move(req));

        auto evRes = Runtime->GrabEdgeEvent<TShardEvents::TReadManyResponse>(apiActor);
        auto* resp = evRes->Get();

        ASSERT_EQ(resp->Metrics.size(), 2u);

        TSet<TVector<TString>> expectedLabels = {
            {"host", "xxa", "sensor", "a"},
            {"host", "xxa", "sensor", "q"},
        };

        auto& strings = resp->Strings;
        TSet<TVector<TString>> labels;

        for (const auto& metric: resp->Metrics) {
            TVector<TString> l;
            for (auto& label: *metric.Labels) {
                l.emplace_back(strings[label.Key()]);
                l.emplace_back(strings[label.Value()]);
            }
            labels.insert(l);

            NTs::TDoubleTsDecoder decoder{
                    metric.Columns.ToColumnSet(metric.Type),
                    metric.Data};

            NTsModel::TGaugePoint point;
            double expectedValue = 5.;
            int pointCounter = 0;
            while (decoder.NextPoint(&point)) {
                EXPECT_DOUBLE_EQ(point.Num, expectedValue) << "pointCounter=" << pointCounter;
                expectedValue += 5.;
                pointCounter++;
            }
            EXPECT_EQ(pointCounter, 3);
        }
        EXPECT_EQ(labels, expectedLabels);
    }
}

TEST_F(ShardWithFts, FindMetricsTotalCount) {
    auto apiActor = Runtime->AllocateEdgeActor();
    {
        auto [meta, data] = NSlog::MakeSlog(10)
            .Gauge({{"host", "xxa"}, {"sensor", "a"}})
                .Add("2000-01-01T00:00:00Z", 5)
                .Add("2000-01-01T00:00:10Z", 10)
                .Add("2000-01-01T00:00:20Z", 15)
            .Done()
            .Gauge({{"host", "xxa"}, {"sensor", "q"}})
                .Add("2000-01-01T00:00:00Z", 5)
                .Add("2000-01-01T00:00:10Z", 10)
                .Add("2000-01-01T00:00:20Z", 15)
            .Done()
            .Gauge({{"host", "xxb"}, {"sensor", "q"}})
                .Add("2000-01-01T00:00:00Z", 5)
                .Add("2000-01-01T00:00:10Z", 10)
                .Add("2000-01-01T00:00:20Z", 15)
            .Done()
            .Gauge({{"host", "yya"}, {"sensor", "q"}})
                .Add("2000-01-01T00:00:00Z", 5)
                .Add("2000-01-01T00:00:10Z", 10)
                .Add("2000-01-01T00:00:20Z", 15)
            .Done()
            .Done();

        Runtime->Send(CreateIndexHandle(TLogId{10, 5}, meta, data, apiActor));
        Runtime->GrabEdgeEvent<TShardEvents::TIndexDone>(apiActor);
        Runtime->DispatchEvents({}, TDuration::Minutes(2));
    }

    {
        auto* evReq = new TShardEvents::TFind{10, ParseSelectors("{host=xx*}"), 2};
        Runtime->Send(ShardId_, apiActor, THolder(evReq));

        auto evResp = Runtime->GrabEdgeEvent<TShardEvents::TFindResponse>(apiActor);
        auto* resp = evResp->Get();
        EXPECT_EQ(resp->Strings.Size(), 5u);
        EXPECT_EQ(resp->Metrics.size(), 2u);
        EXPECT_EQ(resp->TotalCount, 3u);
    }
}

TEST_F(ShardWithFts, AddHistPointToLogHistMetric) {
    auto apiActor = Runtime->AllocateEdgeActor();
    {
        Runtime->UpdateCurrentTime(TInstant::ParseIso8601("2000-01-01T00:03:00Z"));
        TLogHistogramSnapshotPtr logHist1 = MakeIntrusive<TLogHistogramSnapshot>(2., 0u, 0, TVector<double>{5.});
        NMonitoring::TLogHistogramSnapshotPtr logHist2 = MakeIntrusive<TLogHistogramSnapshot>(2., 0u, 0, TVector<double>{10.});

        auto [meta, data] = NSlog::MakeSlog(10)

            .LogHist({{"host", "xxa"}})
                .Add("2000-01-01T00:00:00Z", logHist1)
                .Add("2000-01-01T00:00:01Z", logHist2)
            .Done()
            .Hist({{"host", "xxa"}})
                .Add("2000-01-01T00:00:10Z", MakeSnapshot({{1., 15}}))
                .Add("2000-01-01T00:00:11Z", MakeSnapshot({{1., 20}}))
            .Done()

            .Done();

        Runtime->Send(CreateIndexHandle(TLogId{10, 5}, meta, data, apiActor));
        Runtime->GrabEdgeEvent<TShardEvents::TIndexDone>(apiActor);
        Runtime->DispatchEvents({}, TDuration::Minutes(2));
    }

    {
        auto req = std::make_unique<TShardEvents::TReadMany>();
        req->Lookup = std::make_unique<TShardEvents::TReadMany::TLookup>();
        req->Lookup->Selectors.Add("host", "xx*");
        req->Lookup->Limit = 100;
        Runtime->Send(ShardId_, apiActor, std::move(req));

        auto evResp = Runtime->GrabEdgeEvent<TShardEvents::TReadManyResponse>(apiActor);
        auto* resp = evResp->Get();

        ASSERT_EQ(resp->Metrics.size(), 1u);

        TSet<TVector<TString>> expectedLabels = {
            {"host", "xxa"},
        };

        auto& strings = resp->Strings;
        TSet<TVector<TString>> labels;

        for (const auto& metric: resp->Metrics) {
            TVector<TString> l;
            for (auto& label: *metric.Labels) {
                l.emplace_back(strings[label.Key()]);
                l.emplace_back(strings[label.Value()]);
            }
            labels.insert(l);

            NTs::THistogramTsDecoder decoder{
                    metric.Columns.ToColumnSet(metric.Type),
                    metric.Data};

            NTsModel::THistPoint point;
            ui32 expectedValue = 5;
            int pointCounter = 0;

            while (decoder.NextPoint(&point)) {
                EXPECT_DOUBLE_EQ(point.Count, 1);

                auto itr = FindIf(point.Buckets, [](const auto& bucket) { return std::abs(bucket.UpperBound - 1.) < 1e-3; });
                EXPECT_NE(itr, point.Buckets.end());
                EXPECT_EQ(itr->Value, expectedValue);
                expectedValue += 5;
                pointCounter++;
            }
            EXPECT_EQ(pointCounter, 4);
        }
        EXPECT_EQ(labels, expectedLabels);
    }
}

TEST_F(ShardWithFts, ReadOneEmpty) {
    auto apiActor = Runtime->AllocateEdgeActor();

    {
        auto req = std::make_unique<TShardEvents::TReadOne>(
                NLabels::TLabels::OwnedStorageSorted({{"host", "xxa"}}),
                NStockpile::EFormat::DELETED_SHARDS_39,
                NTsMath::TOperationPipeline{},
                TInstant::Zero(),
                TInstant::Zero());
        Runtime->Send(ShardId_, apiActor, std::move(req));

        auto evResp = Runtime->GrabEdgeEvent<TShardEvents::TReadOneResponse>(apiActor);
        auto* resp = evResp->Get();
        ASSERT_EQ(resp->Metric.Data.Size(), 0u);
    }
}

TEST_F(ShardWithFts, ReadEmptyAfterAllFramesDropped) {
    auto apiActor = Runtime->AllocateEdgeActor();
    {
        auto [meta, data] = NSlog::MakeSlog(10)
            .Gauge({{"sensor", "a"}})
            .Add("2000-01-01T00:00:00Z", 5)
            .Done()
            .Done();

        auto apiActor = Runtime->AllocateEdgeActor();
        Runtime->Send(CreateIndexHandle(TLogId{10, 5}, meta, data, apiActor));
        Runtime->GrabEdgeEvent<TShardEvents::TIndexDone>(apiActor);
    }

    EXPECT_EQ(Metrics->StorageFrames->Get(), 1);
    EXPECT_EQ(Metrics->StorageFramesOpen->Get(), 1);
    EXPECT_EQ(Metrics->StorageFramesSealed->Get(), 0);
    EXPECT_EQ(Metrics->StorageMetrics->Get(), 1);
    EXPECT_EQ(Metrics->StorageMetricsEmpty->Get(), 0);

    Runtime->GrabEdgeEvent<NWal::TWalEvents::TSnapshot>(ShardManagerId_);
    Runtime->DispatchEvents({}, TDuration::Minutes(35));

    EXPECT_EQ(Metrics->StorageFrames->Get(), 0);
    EXPECT_EQ(Metrics->StorageFramesOpen->Get(), 0);
    EXPECT_EQ(Metrics->StorageFramesSealed->Get(), 0);
    EXPECT_EQ(Metrics->StorageMetrics->Get(), 0);
    EXPECT_LE(Metrics->StorageMetricsEmpty->Get(), 1);

    // read many
    {
        auto req = std::make_unique<TShardEvents::TReadMany>();
        req->Lookup = std::make_unique<TShardEvents::TReadMany::TLookup>();
        req->Lookup->Selectors.Add("host", "xx*");
        req->Lookup->Limit = 100;

        Runtime->Send(ShardId_, apiActor, std::move(req));

        auto evResp = Runtime->GrabEdgeEvent<TShardEvents::TReadManyResponse>(apiActor);
        ASSERT_TRUE(evResp);

        auto* resp = evResp->Get();
        ASSERT_TRUE(resp->Metrics.empty());
    }

    // read one
    {
        auto req = std::make_unique<TShardEvents::TReadOne>(
                NLabels::TLabels::OwnedStorageSorted({{"host", "xxa"}}),
                NStockpile::EFormat::DELETED_SHARDS_39,
                NTsMath::TOperationPipeline{},
                TInstant::Zero(),
                TInstant::Zero());
        Runtime->Send(ShardId_, apiActor, std::move(req));

        auto evResp = Runtime->GrabEdgeEvent<TShardEvents::TReadOneResponse>(apiActor);
        auto* resp = evResp->Get();
        ASSERT_EQ(resp->Metric.Data.Size(), 0u);
    }

    // read label values
    {
        auto* evReq = new TShardEvents::TLabelValues{ParseSelectors("{host=x*}"), {}, {}, 1000};
        Runtime->Send(ShardId_, apiActor, THolder(evReq));

        auto evResp = Runtime->GrabEdgeEvent<TShardEvents::TLabelValuesResponse>(apiActor);
        auto* resp = evResp->Get();

        EXPECT_EQ(resp->Strings.Size(), 0u);
        EXPECT_EQ(resp->Labels.size(), 0u);
        EXPECT_EQ(resp->MetricCount, 0u);
    }

    // read label keys
    {
        auto evReq = new TShardEvents::TLabelKeys{ParseSelectors("{host=x*}")};
        Runtime->Send(ShardId_, apiActor, THolder(evReq));

        auto evResp = Runtime->GrabEdgeEvent<TShardEvents::TLabelKeysResponse>(apiActor);
        auto* resp = evResp->Get();

        ASSERT_TRUE(resp->Keys.empty());
        ASSERT_EQ(resp->Strings.Size(), 0u);
    }

    // find metrics
    {
        auto* evReq = new TShardEvents::TFind{10, ParseSelectors("{host=xx*}")};
        Runtime->Send(ShardId_, apiActor, THolder(evReq));

        auto evResp = Runtime->GrabEdgeEvent<TShardEvents::TFindResponse>(apiActor);
        auto* resp = evResp->Get();
        EXPECT_EQ(resp->Strings.Size(), 0u);
        EXPECT_EQ(resp->Metrics.size(), 0u);
        EXPECT_EQ(resp->TotalCount, 0u);
    }
}

TEST_F(ShardWithFts, ManyFrameDrops) {
    auto apiActor = Runtime->AllocateEdgeActor();

    auto GetMetaData = [](TInstant time) {
        return NSlog::MakeSlog(10)
            .Gauge({{"host",   "xxa"},
                    {"sensor", "a"}})
            .Add(time, 5)
            .Done()
            .Gauge({{"host",   "xxa"},
                    {"sensor", "q"}})
            .Add(time, 5)
            .Done()
            .Gauge({{"host",   "xxb"},
                    {"sensor", "q"}})
            .Add(time, 5)
            .Done()
            .Gauge({{"host",   "yya"},
                    {"sensor", "q"}})
            .Add(time, 5)
            .Done()
            .Done();
    };

    for (ui64 frameId = 0; frameId < 10; frameId++) {
        Runtime->UpdateCurrentTime(TInstant::ParseIso8601("2000-01-01T00:03:00Z") + Config_.ChunkLength * frameId);
        auto [meta, data] = GetMetaData(TInstant::ParseIso8601("2000-01-01T00:02:10Z") + Config_.ChunkLength * frameId);

        Runtime->Send(CreateIndexHandle(TLogId{10, frameId + 5}, meta, data, apiActor));
        Runtime->GrabEdgeEvent<TShardEvents::TIndexDone>(apiActor);
        Runtime->DispatchEvents({}, TDuration::Minutes(2));

        auto req = std::make_unique<TShardEvents::TReadMany>();
        req->Lookup = std::make_unique<TShardEvents::TReadMany::TLookup>();
        req->Lookup->Selectors.Add("host", "xx*");
        req->Lookup->Limit = 100;
        Runtime->Send(ShardId_, apiActor, std::move(req));

        auto evResp = Runtime->GrabEdgeEvent<TShardEvents::TReadManyResponse>(apiActor);
        auto* resp = evResp->Get();
        EXPECT_EQ(resp->Metrics.size(), 3u);
    }
}

TEST_F(ShardWithFts, ReadOne) {
    auto apiActor = Runtime->AllocateEdgeActor();

    {
        auto [meta, data] = NSlog::MakeSlog(10)
            .Gauge({{"host", "xxa"}, {"sensor", "a"}})
                .Add("2000-01-01T00:00:00Z", 5)
                .Add("2000-01-01T00:00:10Z", 10)
                .Add("2000-01-01T00:00:20Z", 15)
            .Done()
            .Gauge({{"host", "xxa"}, {"sensor", "q"}})
                .Add("2000-01-01T00:00:00Z", 5)
                .Add("2000-01-01T00:00:10Z", 10)
                .Add("2000-01-01T00:00:20Z", 15)
            .Done()
            .Gauge({{"host", "xxb"}, {"sensor", "q"}})
                .Add("2000-01-01T00:00:00Z", 5)
                .Add("2000-01-01T00:00:10Z", 10)
                .Add("2000-01-01T00:00:20Z", 15)
            .Done()
            .Gauge({{"host", "yya"}, {"sensor", "q"}})
                .Add("2000-01-01T00:00:00Z", 5)
                .Add("2000-01-01T00:00:10Z", 10)
                .Add("2000-01-01T00:00:20Z", 15)
            .Done()
            .Done();

        Runtime->Send(CreateIndexHandle(TLogId{10, 5}, meta, data, apiActor));
        Runtime->GrabEdgeEvent<TShardEvents::TIndexDone>(apiActor);
        Runtime->DispatchEvents({}, TDuration::Minutes(2));
    }

    {
        auto req = std::make_unique<TShardEvents::TReadOne>(
                NLabels::TLabels::OwnedStorageSorted({{"host", "xxa"}, {"sensor", "q"}}),
                NStockpile::EFormat::DELETED_SHARDS_39,
                NTsMath::TOperationPipeline{},
                TInstant::Zero(),
                TInstant::Zero());
        Runtime->Send(ShardId_, apiActor, std::move(req));

        auto evResp = Runtime->GrabEdgeEvent<TShardEvents::TReadOneResponse>(apiActor);
        auto* resp = evResp->Get();
        ASSERT_NE(resp->Metric.Data.Size(), 0u);
    }
}

// TODO: test unique labels handler
