#include <library/cpp/testing/gtest/gtest.h>

#include <solomon/services/memstore/lib/index/shard.h>
#include <solomon/services/memstore/lib/index/shard_manager.h>

#include <solomon/libs/cpp/clients/slicer/slicelet_listener.h>
#include <solomon/libs/cpp/conf_db/puller/puller.h>
#include <solomon/libs/cpp/local_shard_provider/local_shard_provider.h>

#include "common.h"

using namespace NSolomon;
using namespace NMemStore;
using namespace NIndex;
using namespace NActors;
using namespace testing;

using NMonitoring::EMetricType;

class ShardManager: public TIndexTestBase {
protected:
    void SetUp() override {
        TIndexTestBase::SetUp();

        ListenerId_ = Runtime->AllocateEdgeActor();
        ApiListenerId_ = Runtime->AllocateEdgeActor();
        LimiterId_ = Runtime->Register(CreateIndexLimiter(TIndexLimiterConfig::Default(), {}).release());

        auto indexLimiter = CreateIndexWriteLimiter(TIndexLimiterConfig::Default());
        ShardManagerId_ = Runtime->Register(CreateShardManager({}, LimiterId_, indexLimiter, Registry).release());
        Runtime->WaitForBootstrap();
    }

    IEventHandle* CreateIndexHandle(TLogId logId, TNumId numId, TString meta, TString data) {
        auto* event = new TShardEvents::TIndex{
            logId,
            TShardAddPointRequests{
                numId,
                TShardKey{"project", "cluster", "service"},
                {{std::move(meta), std::move(data), ApiListenerId_, 0}}
            }
        };
        return new IEventHandle{ShardManagerId_, ListenerId_, event};
    }

    void SendLocalShardsUpdate(TVector<TNumId>&& removed, TVector<TNumId>&& add) {
        TVector<NDb::NModel::TShardConfig> added;

        for (auto numId: add) {
            added.emplace_back(NDb::NModel::TShardConfig{
                    .Id = TStringBuilder() << "p" << numId << "_c" << numId << "_s" << numId,
                    .NumId = numId,
                    .ClusterName = TStringBuilder() << "c" << numId,
                    .ProjectId = TStringBuilder() << "p" << numId,
                    .ServiceName = TStringBuilder() << "s" << numId,
            });
        }

        Runtime->Send(ShardManagerId_, THolder(new TLocalShardProviderEvents::TLocalShards{
            std::move(removed),
            {}, // updated
            std::move(added),
        }));
    }

protected:
    TActorId ListenerId_;
    TActorId ApiListenerId_;
    TActorId LimiterId_;
    TActorId ShardManagerId_;
};

TEST_F(ShardManager, ListShards) {
    ui64 generation;

    {
        Runtime->Send(ShardManagerId_, ListenerId_, MakeHolder<TShardManagerEvents::TListShards>(0));
        auto ev = Runtime->GrabEdgeEvent<TShardManagerEvents::TListShardsResponse>(ListenerId_);
        EXPECT_GT(ev->Get()->Generation, 0u);
        EXPECT_THAT(ev->Get()->Shards, testing::IsEmpty());
        EXPECT_EQ(Metrics->StorageShards->Get(), 0);

        generation = ev->Get()->Generation;
    }

    {
        TNumId numId = 10;
        auto [meta, data] = NSlog::MakeSlog(numId).Gauge({{"sensor", "rps"}}).Add("2000-01-01T00:00:00Z", 10).Done().Done();
        Runtime->Send(CreateIndexHandle(TLogId{0, 0}, numId, meta, data));
        Runtime->GrabEdgeEvent<TShardEvents::TIndexDone>(ApiListenerId_);
    }

    {
        Runtime->Send(ShardManagerId_, ListenerId_, MakeHolder<TShardManagerEvents::TListShards>(generation));
        auto ev = Runtime->GrabEdgeEvent<TShardManagerEvents::TListShardsResponse>(ListenerId_);
        EXPECT_EQ(ev->Get()->Generation, generation);
        EXPECT_EQ(ev->Get()->Shards.size(), 0u);
        EXPECT_EQ(Metrics->StorageShards->Get(), 0);
    }

    {
        SendLocalShardsUpdate({}, {1u});
    }

    {
        Runtime->Send(ShardManagerId_, ListenerId_, MakeHolder<TShardManagerEvents::TListShards>(generation));
        auto ev = Runtime->GrabEdgeEvent<TShardManagerEvents::TListShardsResponse>(ListenerId_);
        EXPECT_GT(ev->Get()->Generation, generation);
        EXPECT_EQ(ev->Get()->Shards.size(), 1u);
        EXPECT_EQ(ev->Get()->Shards[0].first, 1u);
        EXPECT_EQ(Metrics->StorageShards->Get(), 1u);

        generation = ev->Get()->Generation;
    }

    {
        SendLocalShardsUpdate({1u}, {2u});
        Runtime->DispatchEvents({}, TDuration::MilliSeconds(1)); // dispatch TPoison sent to ShardActors
    }

    {
        Runtime->Send(ShardManagerId_, ListenerId_, MakeHolder<TShardManagerEvents::TListShards>(generation));
        auto ev = Runtime->GrabEdgeEvent<TShardManagerEvents::TListShardsResponse>(ListenerId_);
        EXPECT_GT(ev->Get()->Generation, generation);
        EXPECT_EQ(ev->Get()->Shards.size(), 1u);
        EXPECT_EQ(ev->Get()->Shards[0].first, 2u);
        EXPECT_EQ(Metrics->StorageShards->Get(), 1u);

        generation = ev->Get()->Generation;
    }

    {
        SendLocalShardsUpdate({}, {3u});
    }

    {
        Runtime->Send(ShardManagerId_, ListenerId_, MakeHolder<TShardManagerEvents::TListShards>(generation));
        auto ev = Runtime->GrabEdgeEvent<TShardManagerEvents::TListShardsResponse>(ListenerId_);
        EXPECT_GT(ev->Get()->Generation, generation);
        EXPECT_EQ(ev->Get()->Shards.size(), 2u);

        TVector<TNumId> numIds;
        for (auto [numId, _]: ev->Get()->Shards) {
            numIds.emplace_back(numId);
        }

        EXPECT_THAT(numIds, UnorderedElementsAre(2u, 3u));
        EXPECT_EQ(Metrics->StorageShards->Get(), 2u);

        generation = ev->Get()->Generation;
    }

    {
        SendLocalShardsUpdate({2u, 3u}, {});
        Runtime->DispatchEvents({}, TDuration::MilliSeconds(1)); // dispatch TPoison sent to ShardActors
    }

    {
        Runtime->Send(ShardManagerId_, ListenerId_, MakeHolder<TShardManagerEvents::TListShards>(generation));
        auto ev = Runtime->GrabEdgeEvent<TShardManagerEvents::TListShardsResponse>(ListenerId_);
        EXPECT_GT(ev->Get()->Generation, generation);
        EXPECT_EQ(ev->Get()->Shards.size(), 0u);
        EXPECT_EQ(Metrics->StorageShards->Get(), 0u);

        generation = ev->Get()->Generation;
    }

    {
        Runtime->Send(ShardManagerId_, ListenerId_, MakeHolder<TShardManagerEvents::TListShards>(generation));
        auto ev = Runtime->GrabEdgeEvent<TShardManagerEvents::TListShardsResponse>(ListenerId_);
        EXPECT_EQ(ev->Get()->Generation, generation);
        EXPECT_EQ(ev->Get()->Shards.size(), 0u);
        EXPECT_EQ(Metrics->StorageShards->Get(), 0u);
    }
}

TEST_F(ShardManager, FindShard) {
    // no shards at all
    {
        Runtime->Send(ShardManagerId_, ListenerId_, MakeHolder<TShardManagerEvents::TFindShard>(15));
        auto ev = Runtime->GrabEdgeEvent<TShardManagerEvents::TFindShardResponse>(ListenerId_);
        ASSERT_FALSE(ev->Get()->ShardActor);
    }

    SendLocalShardsUpdate({}, {15u});

    // found shard #15
    {
        Runtime->Send(ShardManagerId_, ListenerId_, MakeHolder<TShardManagerEvents::TFindShard>(15));
        auto ev = Runtime->GrabEdgeEvent<TShardManagerEvents::TFindShardResponse>(ListenerId_);
        ASSERT_TRUE(ev->Get()->ShardActor);
    }

    // there is no shard #10
    {
        Runtime->Send(ShardManagerId_, ListenerId_, MakeHolder<TShardManagerEvents::TFindShard>(10));
        auto ev = Runtime->GrabEdgeEvent<TShardManagerEvents::TFindShardResponse>(ListenerId_);
        ASSERT_FALSE(ev->Get()->ShardActor);
    }
}
