#include "common.h"

#include <solomon/services/memstore/lib/index/shard.h>
#include <solomon/services/memstore/lib/index/snapshot.h>
#include <solomon/services/memstore/lib/index/subshard.h>
#include <solomon/services/memstore/lib/time_series/time_series.h>

#include <solomon/libs/cpp/slog/unresolved_meta/iterator.h>
#include <solomon/libs/cpp/slog/snapshot_data/iterator.h>
#include <solomon/libs/cpp/ts_codec/bit_buffer.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <util/string/hex.h>

using namespace NSolomon;
using namespace NMemStore;
using namespace NMemStore::NIndex;

class Snapshot: public TIndexTestBase {
};

NMonitoring::TLabels MonLabels(std::initializer_list<NMonitoring::TLabel> labels) {
    NMonitoring::TLabels result(labels);
    result.SortByName();
    return result;
}

NSolomon::NLabels::TLabels Labels(std::initializer_list<NSolomon::NLabels::TLabels::TPair> labels) {
    return NSolomon::NLabels::TLabels::OwnedStorage(labels.begin(), labels.end());
}

TEST_F(Snapshot, SnapshotCreated) {
    auto shard = Runtime->AllocateEdgeActor();

    auto snapshotActor = Runtime->Register(CreateSnapshotActor(
        shard,
        10,
        5,
        NSolomon::NMemStore::TLogId{1, 2},
        3,
        Metrics).release());

    EXPECT_EQ(Metrics->SnapshotInflight->Get(), 1);
    EXPECT_EQ(Metrics->SnapshotComplete->Get(), 0u);

    Runtime->Send(snapshotActor, THolder(new TSubShardEvents::TSealAndSnapshotFrameDone{
        {
            {
                MonLabels({{"sensor", "s1"}}),
                TTimeSeriesFrame{
                    NTs::TBitBuffer::FromString("00010001"), 5, NSolomon::NTsModel::EPointType::DGauge,
                    ~NSolomon::NTsModel::TPointColumns{}, {TInstant::Zero(), TInstant::Max()}
                }
            },
            {
                MonLabels({{"sensor", "s2"}}),
                TTimeSeriesFrame{
                    NTs::TBitBuffer::FromString("00100010"), 0, NSolomon::NTsModel::EPointType::DGauge,
                    ~NSolomon::NTsModel::TPointColumns{}, {TInstant::Zero(), TInstant::Max()}
                }
            },
            {
                MonLabels({{"sensor", "s3"}}),
                TTimeSeriesFrame{
                    NTs::TBitBuffer::FromString("00110011"), 2, NSolomon::NTsModel::EPointType::Hist,
                    ~NSolomon::NTsModel::TPointColumns{}, {TInstant::Zero(), TInstant::Max()}
                }
            },
        }
    }));
    Runtime->Send(snapshotActor, THolder(new TSubShardEvents::TSealAndSnapshotFrameDone{
        {
        }
    }));
    Runtime->Send(snapshotActor, THolder(new TSubShardEvents::TSealAndSnapshotFrameDone{
        {
            {
                MonLabels({{"sensor", "s4"}}),
                TTimeSeriesFrame{
                    NTs::TBitBuffer::FromString("01000100"), 0, NSolomon::NTsModel::EPointType::DGauge,
                    ~NSolomon::NTsModel::TPointColumns{}, {TInstant::Zero(), TInstant::Max()}
                }
            },
            {
                MonLabels({{"sensor", "s5"}}),
                TTimeSeriesFrame{
                    NTs::TBitBuffer::FromString("01010101"), 2, NSolomon::NTsModel::EPointType::DSummary,
                    ~NSolomon::NTsModel::TPointColumns{}, {TInstant::Zero(), TInstant::Max()}
                }
            },
        }
    }));

    auto ev = Runtime->GrabEdgeEvent<TShardEvents::TSnapshotDone>(shard);

    EXPECT_EQ(ev->Get()->Frame, 5u);
    EXPECT_EQ(ev->Get()->LogId, NSolomon::NMemStore::TLogId(1, 2));

    {
        TStringInput in(ev->Get()->Meta);
        auto meta = NSolomon::NSlog::NUnresolvedMeta::CreateUnresolvedMetaIterator(&in);

        EXPECT_EQ(meta->TotalMetricCount(), 3u);
        EXPECT_EQ(meta->TotalPointCount(), 9u);

        {
            ASSERT_TRUE(meta->HasNext());
            auto metric = meta->Next();
            EXPECT_EQ(metric.Type, NSolomon::NTsModel::EPointType::DGauge);
            EXPECT_EQ(metric.Labels, Labels({{"sensor", "s1"}}));
            EXPECT_EQ(metric.PointCounts, 5u);
            EXPECT_EQ(metric.DataSize, 12u);
        }
        {
            ASSERT_TRUE(meta->HasNext());
            auto metric = meta->Next();
            EXPECT_EQ(metric.Type, NSolomon::NTsModel::EPointType::Hist);
            EXPECT_EQ(metric.Labels, Labels({{"sensor", "s3"}}));
            EXPECT_EQ(metric.PointCounts, 2u);
            EXPECT_EQ(metric.DataSize, 12u);
        }
        {
            ASSERT_TRUE(meta->HasNext());
            auto metric = meta->Next();
            EXPECT_EQ(metric.Type, NSolomon::NTsModel::EPointType::DSummary);
            EXPECT_EQ(metric.Labels, Labels({{"sensor", "s5"}}));
            EXPECT_EQ(metric.PointCounts, 2u);
            EXPECT_EQ(metric.DataSize, 12u);
        }

        ASSERT_FALSE(meta->HasNext());
    }

    {
        TStringInput in(ev->Get()->Data);
        auto data = NSolomon::NSlog::NSnapshotData::CreateSnapshotLogDataIterator(&in);

        EXPECT_EQ(data->TotalMetricCount(), 3u);
        EXPECT_EQ(data->TotalPointCount(), 9u);

        {
            ASSERT_TRUE(data->HasNext());
            auto record = data->Next();
            EXPECT_EQ(record.Type, NSolomon::NTsModel::EPointType::DGauge);
            EXPECT_EQ(record.Encoded, NTs::TBitBuffer::FromString("00010001"));
        }
        {
            ASSERT_TRUE(data->HasNext());
            auto record = data->Next();
            EXPECT_EQ(record.Type, NSolomon::NTsModel::EPointType::Hist);
            EXPECT_EQ(record.Encoded, NTs::TBitBuffer::FromString("00110011"));
        }
        {
            ASSERT_TRUE(data->HasNext());
            auto record = data->Next();
            EXPECT_EQ(record.Type, NSolomon::NTsModel::EPointType::DSummary);
            EXPECT_EQ(record.Encoded, NTs::TBitBuffer::FromString("01010101"));
        }
    }
}

TEST_F(Snapshot, SplitedSnapshot) {
    auto shard = Runtime->AllocateEdgeActor();

    auto snapshotActor = Runtime->Register(CreateSnapshotActor(
        shard,
        10,
        5,
        NSolomon::NMemStore::TLogId{1, 2},
        3,
        Metrics).release());

    EXPECT_EQ(Metrics->SnapshotInflight->Get(), 1);
    EXPECT_EQ(Metrics->SnapshotComplete->Get(), 0u);

    Runtime->Send(snapshotActor, THolder(new TSubShardEvents::TSealAndSnapshotFrameDone{
        {
            {
                MonLabels({{"sensor", "s1"}}),
                TTimeSeriesFrame{
                    NTs::TBitBuffer::FromString("00010001"), 5, NSolomon::NTsModel::EPointType::DGauge,
                    ~NSolomon::NTsModel::TPointColumns{}, {TInstant::Zero(), TInstant::Max()}
                }
            }
        }
    }));
    Runtime->Send(snapshotActor, THolder(new TSubShardEvents::TSealAndSnapshotFrameDone{
        {
            {
                MonLabels({{"sensor", "s2"}}),
                TTimeSeriesFrame{
                    NTs::TBitBuffer::FromString("00010001"), 5, NSolomon::NTsModel::EPointType::DGauge,
                    ~NSolomon::NTsModel::TPointColumns{}, {TInstant::Zero(), TInstant::Max()}
                }
            }
        }
    }));
    Runtime->Send(snapshotActor, THolder(new TSubShardEvents::TSealAndSnapshotFrameDone{
        {
            {
                MonLabels({{"sensor", "s3"}}),
                TTimeSeriesFrame{
                    NTs::TBitBuffer::FromString("00010001"), 5, NSolomon::NTsModel::EPointType::DGauge,
                    ~NSolomon::NTsModel::TPointColumns{}, {TInstant::Zero(), TInstant::Max()}
                }
            }
        }
    }));
    auto ev = Runtime->GrabEdgeEvent<TShardEvents::TSnapshotDone>(shard, TDuration::Seconds(2));

    ASSERT_TRUE(ev->HasEvent());

    TStringInput in(ev->Get()->Meta);
    auto meta = NSolomon::NSlog::NUnresolvedMeta::CreateUnresolvedMetaIterator(&in);

    EXPECT_EQ(meta->TotalMetricCount(), 3u);
    EXPECT_EQ(meta->TotalPointCount(), 15u);
}
