#include "common.h"

#include <solomon/services/memstore/lib/index/subshard.h>
#include <solomon/services/memstore/lib/index/index_limiter.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon;
using namespace NMemStore;
using namespace NIndex;
using namespace NLabels;
using namespace NActors;
using namespace NSolomon::NMemStore::NIndex;

class SubShard: public TIndexTestBase {
protected:
    std::unique_ptr<IActor> BuildSubShard() {
        Fts_ = Runtime->Register(FtsIndex(1u).release());
        return CreateSubShard(5, 0, Fts_, Limiter_, Metrics);
    }

    void SetUp() override {
        TIndexTestBase::SetUp();
        Limiter_ = Runtime->Register(CreateIndexLimiter(TIndexLimiterConfig::Default(), {}).release());
        SubShardId_ = Runtime->Register(BuildSubShard().release());
        Runtime->WaitForBootstrap();
    }

    TActorId SubShardId_;
    TActorId Fts_;
    TActorId Limiter_;
};

auto MakeDefaultSlog() {
    return NSlog::MakeSlog(10)
        .Gauge({{"sensor", "a"}})
            .Add("2000-01-01T00:00:00Z", 5)
            .Add("2000-01-01T00:00:10Z", 10)
            .Add("2000-01-01T00:00:20Z", 15)
        .Done()
        .Gauge({{"sensor", "b"}})
            .Add("2000-01-01T00:00:00Z", 5)
            .Add("2000-01-01T00:00:10Z", 10)
            .Add("2000-01-01T00:00:20Z", 15)
        .Done()
        .Done();
}

auto MakeDefaultVectorParser() {
    auto [meta, data] = MakeDefaultSlog();

    NSolomon::NMemStore::NSLog::TParser parser{std::move(meta), std::move(data)};

    TVector<NSLog::TTsParser> vectorParser;
    while (parser.HasNext()) {
        vectorParser.push_back(parser.Next());
    }
    return vectorParser;
}

TEST_F(SubShard, Create) {
    EXPECT_NE(BuildSubShard(), nullptr);
}

TEST_F(SubShard, AddFrame) {
    Runtime->Send(SubShardId_, MakeHolder<TSubShardEvents::TAddFrame>(0));
}

TEST_F(SubShard, AddFrames) {
    for (auto frameId = 0; frameId < 100; ++frameId) {
        Runtime->Send(SubShardId_, MakeHolder<TSubShardEvents::TAddFrame>(frameId));
    }
}

TEST_F(SubShard, AddFrameIdxMismatch) {
}

TEST_F(SubShard, SnapshotFrame) {
    auto apiActor = Runtime->AllocateEdgeActor();
    Runtime->Send(SubShardId_, MakeHolder<TSubShardEvents::TAddFrame>(0));

    auto vectorParser = MakeDefaultVectorParser();

    Runtime->Send(SubShardId_, MakeHolder<TSubShardEvents::TWriteToFrame>(0,std::move(vectorParser)));

    Runtime->Send(SubShardId_, MakeHolder<TSubShardEvents::TSealAndSnapshotFrame>(apiActor, 0));
    auto evResponse = Runtime->GrabEdgeEvent<TSubShardEvents::TSealAndSnapshotFrameDone>(apiActor);

    EXPECT_NE(evResponse, nullptr);

    EXPECT_EQ(evResponse->Get()->Snapshots.size(), 2u);
}

TEST_F(SubShard, SnapshotFrameEmpty) {
    auto apiActor = Runtime->AllocateEdgeActor();
    Runtime->Send(SubShardId_, MakeHolder<TSubShardEvents::TAddFrame>(0));
    Runtime->Send(SubShardId_, MakeHolder<TSubShardEvents::TSealAndSnapshotFrame>(apiActor, 0));
    auto evResponse = Runtime->GrabEdgeEvent<TSubShardEvents::TSealAndSnapshotFrameDone>(apiActor);

    EXPECT_NE(evResponse, nullptr);

    EXPECT_TRUE(evResponse->Get()->Snapshots.empty());
}

TEST_F(SubShard, SnapshotFrameIdxMismatch) {
}

TEST_F(SubShard, DropFrame) {
    auto apiActor = Runtime->AllocateEdgeActor();

    Runtime->Send(SubShardId_, MakeHolder<TSubShardEvents::TAddFrame>(0));
    Runtime->Send(Fts_, MakeHolder<TFtsIndexEvents::TAddFrame>(0));

    auto vectorParser = MakeDefaultVectorParser();

    Runtime->Send(SubShardId_, MakeHolder<TSubShardEvents::TWriteToFrame>(0,std::move(vectorParser)));
    EXPECT_EQ(Metrics->StorageMetrics->Get(), 2);
    Runtime->Send(SubShardId_, MakeHolder<TSubShardEvents::TSealAndSnapshotFrame>(SubShardId_, 0));

    Runtime->Send(SubShardId_, apiActor, MakeHolder<TSubShardEvents::TDropFrame>(0));
    EXPECT_EQ(Metrics->StorageMetrics->Get(), 0);

    Runtime->Send(SubShardId_, MakeHolder<TSubShardEvents::TReadMetricsByMetricId>(
        apiActor,
        TVector<ui32>{},
        TInstant::Zero(),
        TInstant::Zero()));

    auto evResponse = Runtime->GrabEdgeEvent<TSubShardEvents::TReadMetricsResponse>(apiActor);
    EXPECT_EQ(evResponse->Get()->LabelsPool->Size(), 0u);
}

TEST_F(SubShard, DropFrameAfterReading) {
    auto apiActor = Runtime->AllocateEdgeActor();

    Runtime->Send(SubShardId_, MakeHolder<TSubShardEvents::TAddFrame>(0));
    Runtime->Send(Fts_, MakeHolder<TFtsIndexEvents::TAddFrame>(0));

    auto [meta, data] = NSlog::MakeSlog(10)
        .Gauge({{"sensor", "a"}})
            .Add("2000-01-01T00:00:00Z", 5)
            .Add("2000-01-01T00:00:10Z", 10)
            .Add("2000-01-01T00:00:20Z", 15)
            .Done()
        .Gauge({{"sensor", "b"}})
            .Add("2000-01-01T00:00:00Z", 5)
            .Add("2000-01-01T00:00:10Z", 10)
            .Add("2000-01-01T00:00:20Z", 15)
            .Done()
        .Done();

    NSolomon::NMemStore::NSLog::TParser parser{std::move(meta), std::move(data)};

    TVector<NSLog::TTsParser> vectorParser;
    while (parser.HasNext()) {
        vectorParser.push_back(parser.Next());
    }
    Runtime->Send(SubShardId_, MakeHolder<TSubShardEvents::TWriteToFrame>(0,std::move(vectorParser)));
    EXPECT_EQ(Metrics->StorageMetrics->Get(), 2);

    TVector<NSolomon::NLabels::TLabels> labels{
        TLabels::OwnedStrings({{"sensor", "a"}})};
    Runtime->Send(SubShardId_, MakeHolder<TSubShardEvents::TReadMetricsByLabels>(
        apiActor,
        TInstant::Zero(), TInstant::Zero(), std::move(labels)));

    auto evResponse = Runtime->GrabEdgeEvent<TSubShardEvents::TReadMetricsResponse>(apiActor);
    EXPECT_EQ(evResponse->Get()->LabelsPool->Size(), 3u);

    Runtime->Send(SubShardId_, MakeHolder<TSubShardEvents::TSealAndSnapshotFrame>(SubShardId_, 0));
    Runtime->Send(SubShardId_, apiActor, MakeHolder<TSubShardEvents::TDropFrame>(0));

    EXPECT_EQ(Metrics->StorageMetrics->Get(), 0);

    Runtime->Send(SubShardId_, MakeHolder<TSubShardEvents::TReadMetricsByMetricId>(
        apiActor,
        TVector<ui32>{},
        TInstant::Zero(),
        TInstant::Zero()));

    evResponse = Runtime->GrabEdgeEvent<TSubShardEvents::TReadMetricsResponse>(apiActor);
    EXPECT_EQ(evResponse->Get()->LabelsPool->Size(), 0u);
}

TEST_F(SubShard, DropFrameWithDoubleWrite) {
    auto apiActor = Runtime->AllocateEdgeActor();

    Runtime->Send(SubShardId_, MakeHolder<TSubShardEvents::TAddFrame>(0));

    auto [meta, data] = NSlog::MakeSlog(10)
        .Gauge({{"sensor", "a"}})
            .Add("2000-01-01T00:00:00Z", 5)
            .Add("2000-01-01T00:00:10Z", 10)
            .Add("2000-01-01T00:00:20Z", 15)
            .Done()
        .Gauge({{"sensor", "b"}})
            .Add("2000-01-01T00:00:00Z", 5)
            .Add("2000-01-01T00:00:10Z", 10)
            .Add("2000-01-01T00:00:20Z", 15)
            .Done()
         .Gauge({{"sensor", "a"}})
            .Add("2000-01-01T00:01:00Z", 5)
            .Add("2000-01-01T00:01:10Z", 10)
            .Add("2000-01-01T00:01:20Z", 15)
            .Done()
        .Gauge({{"sensor", "b"}})
            .Add("2000-01-01T00:01:00Z", 5)
            .Add("2000-01-01T00:01:10Z", 10)
            .Add("2000-01-01T00:01:20Z", 15)
            .Done()
        .Done();

    NSolomon::NMemStore::NSLog::TParser parser{std::move(meta), std::move(data)};

    TVector<NSLog::TTsParser> vectorParser;
    while (parser.HasNext()) {
        vectorParser.push_back(parser.Next());
    }
    Runtime->Send(SubShardId_, MakeHolder<TSubShardEvents::TWriteToFrame>(0,std::move(vectorParser)));
    EXPECT_EQ(Metrics->StorageMetrics->Get(), 2);

    Runtime->Send(SubShardId_, MakeHolder<TSubShardEvents::TSealAndSnapshotFrame>(SubShardId_, 0));
    Runtime->Send(SubShardId_, apiActor, MakeHolder<TSubShardEvents::TDropFrame>(0));

    EXPECT_EQ(Metrics->StorageMetrics->Get(), 0);

    Runtime->Send(SubShardId_, MakeHolder<TSubShardEvents::TReadMetricsByMetricId>(
        apiActor,
        TVector<ui32>{},
        TInstant::Zero(),
        TInstant::Zero()));

    auto evResponse = Runtime->GrabEdgeEvent<TSubShardEvents::TReadMetricsResponse>(apiActor);
    EXPECT_EQ(evResponse->Get()->LabelsPool->Size(), 0u);
}

TEST_F(SubShard, DropFrameEmptyMetrics) {
}

TEST_F(SubShard, DropFrameIdxMismatch) {
    auto apiActor = Runtime->AllocateEdgeActor();

    Runtime->Send(SubShardId_, MakeHolder<TSubShardEvents::TAddFrame>(0));

    auto [meta, data] = NSlog::MakeSlog(10)
        .Gauge({{"sensor", "a"}})
            .Add("2000-01-01T00:00:00Z", 5)
            .Add("2000-01-01T00:00:10Z", 10)
            .Add("2000-01-01T00:00:20Z", 15)
        .Done()
        .Gauge({{"sensor", "b"}})
            .Add("2000-01-01T00:00:00Z", 5)
            .Add("2000-01-01T00:00:10Z", 10)
            .Add("2000-01-01T00:00:20Z", 15)
        .Done()
        .Gauge({{"sensor", "a"}})
            .Add("2000-01-01T00:01:00Z", 5)
            .Add("2000-01-01T00:01:10Z", 10)
            .Add("2000-01-01T00:01:20Z", 15)
        .Done()
        .Gauge({{"sensor", "b"}})
            .Add("2000-01-01T00:01:00Z", 5)
            .Add("2000-01-01T00:01:10Z", 10)
            .Add("2000-01-01T00:01:20Z", 15)
        .Done()
        .Done();

    NSolomon::NMemStore::NSLog::TParser parser{std::move(meta), std::move(data)};

    TVector<NSLog::TTsParser> vectorParser;
    while (parser.HasNext()) {
        vectorParser.push_back(parser.Next());
    }
    Runtime->Send(SubShardId_, MakeHolder<TSubShardEvents::TWriteToFrame>(0,std::move(vectorParser)));
    EXPECT_EQ(Metrics->StorageMetrics->Get(), 2);

    Runtime->Send(SubShardId_, MakeHolder<TSubShardEvents::TSealAndSnapshotFrame>(SubShardId_, 0));
    Runtime->Send(SubShardId_, apiActor, MakeHolder<TSubShardEvents::TDropFrame>(0));

    EXPECT_EQ(Metrics->StorageMetrics->Get(), 0);

    Runtime->Send(SubShardId_, MakeHolder<TSubShardEvents::TReadMetricsByMetricId>(
        apiActor,
        TVector<ui32>{0},
        TInstant::Zero(),
        TInstant::Zero()));

    auto evResponse = Runtime->GrabEdgeEvent<TSubShardEvents::TReadMetricsResponse>(apiActor);
    EXPECT_EQ(evResponse->Get()->LabelsPool->Size(), 0u);
}

TEST_F(SubShard, WriteToFrame) {
}

TEST_F(SubShard, WriteToFrameNewTs) {
}

TEST_F(SubShard, WriteToFrameMiddle) {
}

TEST_F(SubShard, WriteToFrameMiddleNewTs) {
}

TEST_F(SubShard, WriteToFrameIdxMismatch) {
}

TEST_F(SubShard, WriteToFrameIdxMismatchMiddle) {
}

TEST_F(SubShard, WriteToFrameSealed) {
}

TEST_F(SubShard, WriteToFrameSealedMiddle) {
}

TEST_F(SubShard, WriteToFrameTypeError) {
}

TEST_F(SubShard, WriteToFrameSyntaxError) {
}

TEST_F(SubShard, Poison) {
}
