#pragma once
#include "labels_pool.h"

#include <solomon/libs/cpp/intern/str_pool.h>
#include <solomon/libs/cpp/labels/labels.h>

#include <util/generic/array_ref.h>
#include <util/generic/hash.h>
#include <util/generic/hash_set.h>
#include <util/generic/maybe.h>
#include <util/generic/vector.h>
#include <util/system/types.h>
#include <util/memory/pool.h>
#include <util/digest/city.h>

namespace NSolomon::NMemStore::NLabels {

/**
 * Storage for metric labels that uses string interner as a backend.
 *
 * We store a value-terminated array of string identifiers. Even elements correspond to label keys,
 * odd -- to label values. The array is sorted lexicographically by label keys, so the order of elements
 * is persistent across different chunks and interners.
 *
 * We can compare labels that come from the same interner (i.e. the same chunk) by comparing arrays of identifiers
 * piecewise. This, however, does not work when labels come from different interners. This is because different
 * interners may assign different identifiers to the same string. To work with labels from different interners
 * we use `TLabelsInternerAgnostic` that actually compares strings rather than their identifiers.
 */
class TLabels {
public:
    /**
     * Iterate over labels.
     */
    class TIterator {
    public:
        TIterator(const NIntern::TStringId* pos)
            : Pos_(pos)
        {
        }

        /**
         * Get label key.
         *
         * Calling this function after the iterator has been exhausted is UB.
         */
        NIntern::TStringId Key() const {
            Y_ASSERT(HasValue());

            return *Pos_;
        }

        /**
         * Get label value.
         *
         * Calling this function after the iterator has been exhausted is UB.
         */
        NIntern::TStringId Value() const {
            Y_ASSERT(HasValue());

            return *(Pos_ + 1);
        }

        /**
         * Move to the next element.
         *
         * If the iterator is exhausted, this function returns `false`. Calling this function again
         * after the iterator has been exhausted is UB.
         */
        bool Advance() {
            Y_ASSERT(HasValue());

            Pos_ += 2;

            return HasValue();
        }

        /**
         * Return true if this iterator is not exhausted.
         */
        bool HasValue() const {
            return *Pos_ != NIntern::InvalidStringId;
        }

    private:
        const NIntern::TStringId* Pos_;
    };

protected:
    TLabels(const NIntern::TStringId* labels);

public:
    /**
     * Iterate over label's key-value pairs.
     *
     * It is guaranteed that the returned iterator yields labels sorted lexicographically by their keys.
     */
    TIterator Iter() const {
        return TIterator{Labels_};
    }

    /**
     * Calculate number of labels. This takes linear time.
     */
    size_t CalcSize() const {
        size_t size = 0;
        auto head = Labels_;
        {
            while (*head != NIntern::InvalidStringId) {
                size++;
                head += 2;
            }
        }
        return size;
    }

    /**
     * Check if there is at least a single label in this class.
     */
    bool Empty() const {
        return *Labels_ == NIntern::InvalidStringId;
    }

    template <typename TStringPool>
    NSolomon::NLabels::TLabels ToInternerAgnostic(const TStringPool& pool) const {
        TVector<NSolomon::NLabels::TLabels::TPair> contents;
        for (auto it = Iter(); it.HasValue(); it.Advance()) {
            contents.push_back({pool.Find(it.Key()).data(), pool.Find(it.Value()).data()});
        }
        return NSolomon::NLabels::TLabels::OwnedStorageSorted(contents.begin(), contents.end());
    }

public:
    inline friend bool operator==(const TLabels& lhs, const TLabels& rhs);
    inline friend bool operator!=(const TLabels& lhs, const TLabels& rhs);
    inline size_t Hash() const;

public:
    /**
     * Copy labels from one storage to another.
     * Returned instance will stay valid for as long as the given pool class is alive.
     */
    TLabels CopyTo(TLabelsPool& pool) const;

    /*
     * Copy labels from TLabelsView to TLabel stored in labelsPool string are stored in stringPool
     */
    template<typename TStringPool>
    static TLabels CopyLabelsTo(const NSolomon::NLabels::TLabelsView& labels, TStringPool& stringPool, TLabelsPool& labelsPool) {
        auto data = labelsPool.Allocate();

        auto current = data;
        for (auto it: labels) {
            *current++ = stringPool.Intern(it.first);
            *current++ = stringPool.Intern(it.second);
        }

        *current = NIntern::InvalidStringId;

        return {data};
    }

    /**
     * Delete labels from storage
     */
    void Free(TLabelsPool& pool);

protected:
    const NIntern::TStringId* Labels_;
};

bool operator==(const TLabels& lhs, const TLabels& rhs)  {
    auto lhsLabels = lhs.Labels_;
    auto rhsLabels = rhs.Labels_;
    while (*lhsLabels == *rhsLabels) {
        if (*lhsLabels == NIntern::InvalidStringId) {
            return true;
        }
        lhsLabels++;
        rhsLabels++;
    }
    return false;
}

bool operator!=(const TLabels& lhs, const TLabels& rhs) {
    return !(lhs == rhs);
}

size_t TLabels::Hash() const {
    auto begin = Labels_;
    auto end = Labels_;
    while (*end != NIntern::InvalidStringId) {
        end++;
    }
    auto charBegin = reinterpret_cast<const char*>(begin);
    auto charEnd = reinterpret_cast<const char*>(end);
    return CityHash64(charBegin, std::distance(charBegin, charEnd));
}

/**
 * Labels storage that owns its array of label identifiers.
 *
 * This class is used to create temporary `TLabels` instance that's not intended for storage, but rather
 * for quick lookup. It interns labels and their values as usual, but does not save array of label identifiers
 * to a pool. Instead, it uses a temporary buffer that's freed when this class dies.
 *
 * This class is not thread safe, and it cannot be safely transferred across thread boundaries (so it shouldn't
 * be saved in actors). Ideally, this class should always be allocated on stack, and it should not be moved to heap
 * or into heap-allocated objects.
 */
class TLabelsOwned: public TLabels {
private:
    TLabelsOwned(TTempArray<NIntern::TStringId> storage);

public:
    /**
     * Intern all labels and put their indices to an owned storage.
     * Labels array must be sorted.
     */
    template <typename TStringPool>
    static TLabelsOwned Intern(
        NSolomon::NLabels::TLabelsView labels,
        TStringPool& labelPool)
    {
        auto dataStorage = TTempArray<NIntern::TStringId>{labels.Size() * 2u + 1u};
        auto data = dataStorage.Data();
        auto current = data;

        for (auto label: labels) {
            *current++ = labelPool.Intern(label.first);
            *current++ = labelPool.Intern(label.second);
        }

        *current = NIntern::InvalidStringId;

        return TLabelsOwned{std::move(dataStorage)};
    }

    /**
     * Find all labels in the interner. If some label is not already interned, return empty result.
     * Labels array must be sorted.
     */
    template <typename TStringPool>
    static TMaybe<TLabelsOwned> Lookup(
        NSolomon::NLabels::TLabelsView labels,
        const TStringPool& labelPool)
    {
        auto dataStorage = TTempArray<NIntern::TStringId>{labels.Size() * 2u + 1u};
        auto data = dataStorage.Data();
        auto current = data;

        for (auto& label: labels) {
            *current = labelPool.Find(label.first);
            if (*current++ == NIntern::InvalidStringId) {
                return {};
            }

            *current = labelPool.Find(label.second);
            if (*current++ == NIntern::InvalidStringId) {
                return {};
            }
        }

        *current = NIntern::InvalidStringId;

        return {TLabelsOwned{std::move(dataStorage)}};
    }

private:
    TTempArray<NIntern::TStringId> Storage_;
};

} // namespace NSolomon::NMemStore::NIndex

template <>
struct THash<NSolomon::NMemStore::NLabels::TLabels> {
    inline size_t operator()(const NSolomon::NMemStore::NLabels::TLabels& labels) const noexcept {
        return labels.Hash();
    }
};

template <>
struct THash<NSolomon::NMemStore::NLabels::TLabelsOwned> {
    inline size_t operator()(const NSolomon::NMemStore::NLabels::TLabelsOwned& labels) const noexcept {
        return labels.Hash();
    }
};
