#include <library/cpp/testing/gtest/gtest.h>

#include "labels_common.h"

#include <solomon/services/memstore/lib/labels/labels.h>

using namespace NSolomon::NMemStore::NLabels;

TEST(Labels, Intern) {
    NSolomon::NIntern::TStringPool pool;

    {
        auto labels = TLabelsOwned::Intern(
            NSolomon::NLabels::TLabels::OwnedStorage({}),
            pool);

        EXPECT_EQ(labels.CalcSize(), 0u);
        EXPECT_EQ(pool.Size(), 0u);
    }

    {
        auto labels = TLabelsOwned::Intern(
            NSolomon::NLabels::TLabels::OwnedStorage({{"code", "200"}, {"sensor", "rps"}}),
            pool);

        EXPECT_EQ(labels.CalcSize(), 2u);
        EXPECT_EQ(pool.Size(), 4u);
    }

    {
        auto labels = TLabelsOwned::Intern(
            NSolomon::NLabels::TLabels::OwnedStorage({{"code", "500"}, {"sensor", "rps"}}),
            pool);

        EXPECT_EQ(labels.CalcSize(), 2u);
        EXPECT_EQ(pool.Size(), 5u);
    }
}

TEST(Labels, SuccessfulLookup) {
    NSolomon::NIntern::TStringPool pool;

    {
        auto labels = TLabelsOwned::Lookup(
            NSolomon::NLabels::TLabels::OwnedStorage({}),
            pool);

        EXPECT_EQ(pool.Size(), 0u);
        EXPECT_TRUE(labels.Defined());
        EXPECT_EQ(labels->CalcSize(), 0u);
    }

    pool.Intern("sensor");
    pool.Intern("rps");
    pool.Intern("code");
    pool.Intern("200");

    {
        auto labels = TLabelsOwned::Lookup(
            NSolomon::NLabels::TLabels::OwnedStorage({{"code", "200"}, {"sensor", "rps"}}),
            pool);

        EXPECT_TRUE(labels.Defined());
        EXPECT_EQ(labels->CalcSize(), 2u);
        EXPECT_EQ(pool.Size(), 4u);
    }
}

TEST(Labels, UnsuccessfulLookup) {
    NSolomon::NIntern::TStringPool pool;

    {
        auto labels = TLabelsOwned::Lookup(
            NSolomon::NLabels::TLabels::OwnedStorage({{"sensor", "rps"}}),
            pool);

        EXPECT_FALSE(labels.Defined());
        EXPECT_EQ(pool.Size(), 0u);
    }

    {
        auto labels = TLabelsOwned::Lookup(
            NSolomon::NLabels::TLabels::OwnedStorage({{"code", "200"}, {"sensor", "rps"}}),
            pool);

        EXPECT_FALSE(labels.Defined());
        EXPECT_EQ(pool.Size(), 0u);
    }

    pool.Intern("rps");
    pool.Intern("code");

    {
        auto labels = TLabelsOwned::Lookup(
            NSolomon::NLabels::TLabels::OwnedStorage({{"code", "200"}}),
            pool);

        EXPECT_FALSE(labels.Defined());
        EXPECT_EQ(pool.Size(), 2u);
    }

    {
        auto labels = TLabelsOwned::Lookup(
            NSolomon::NLabels::TLabels::OwnedStorage({{"sensor", "rps"}}),
            pool);

        EXPECT_FALSE(labels.Defined());
        EXPECT_EQ(pool.Size(), 2u);
    }
}

class LabelOpCases: public ::testing::TestWithParam<std::tuple<TCtor, NSolomon::NLabels::TLabels>> {
public:
    NSolomon::NIntern::TStringPool Pool;
};

INSTANTIATE_TEST_SUITE_P(
    Labels, LabelOpCases,
    ::testing::Combine(::testing::Values(&Intern, &Lookup), LabelCases));

TEST_P(LabelOpCases, Size) {
    auto [ctor, expected] = GetParam();
    auto actual = (*ctor)(expected, Pool);
    EXPECT_EQ(actual.Empty(), expected.Empty());
    EXPECT_EQ(actual.CalcSize(), expected.Size());
}

TEST_P(LabelOpCases, Iteration) {
    auto [ctor, expected] = GetParam();
    auto actual = (*ctor)(expected, Pool);

    auto it = actual.Iter();

    for (auto labelIt = expected.begin(); labelIt != expected.end(); ) {
        ASSERT_TRUE(it.HasValue());
        EXPECT_EQ(Pool.Find(it.Key()), labelIt->first);
        EXPECT_EQ(Pool.Find(it.Value()), labelIt->second);
        ASSERT_EQ(it.Advance(), ++labelIt != expected.end());
    }

    ASSERT_FALSE(it.HasValue());
}

TEST_P(LabelOpCases, CopyTo) {
    TLabelsPool storage{128};

    auto moved = std::invoke([&]() {
        auto [ctor, expected] = GetParam();
        auto actual = (*ctor)(expected, Pool);
        return actual.CopyTo(storage);
    });

    auto expected = std::get<1>(GetParam());

    EXPECT_EQ(moved.Empty(), expected.Empty());
    EXPECT_EQ(moved.CalcSize(), expected.Size());

    auto it = moved.Iter();

    for (auto labelIt = expected.begin(); labelIt != expected.end(); ) {
        ASSERT_TRUE(it.HasValue());
        EXPECT_EQ(Pool.Find(it.Key()), labelIt->first);
        EXPECT_EQ(Pool.Find(it.Value()), labelIt->second);
        ASSERT_EQ(it.Advance(), ++labelIt != expected.end());
    }

    ASSERT_FALSE(it.HasValue());
}

class LabelEqCases: public ::testing::TestWithParam<
        std::tuple<
            NSolomon::NLabels::TLabels,
            NSolomon::NLabels::TLabels>>
{
public:
    NSolomon::NIntern::TStringPool Pool;
};

INSTANTIATE_TEST_SUITE_P(
    Labels, LabelEqCases,
    ::testing::Combine(LabelCases, LabelCases));

TEST_P(LabelEqCases, EqAndHash) {
    auto [expectedLhs, expectedRhs] = GetParam();

    auto lhs = TLabelsOwned::Intern(expectedLhs, Pool);
    auto rhs = TLabelsOwned::Intern(expectedRhs, Pool);

    if (expectedLhs == expectedRhs) {
        EXPECT_EQ(lhs.Hash(), rhs.Hash());
        EXPECT_EQ(lhs, rhs);
    } else {
        EXPECT_NE(lhs.Hash(), rhs.Hash());
        EXPECT_NE(lhs, rhs);
    }
}
