#include "parser.h"

#include "common.h"

#include <solomon/services/memstore/lib/yasm/yasm.h>

#include <library/cpp/monlib/encode/format.h>
#include <library/cpp/monlib/encode/spack/compression.h>
#include <library/cpp/monlib/encode/spack/spack_v1.h>

#include <util/generic/algorithm.h>
#include <util/system/unaligned_mem.h>

namespace NSolomon::NMemStore::NSLog {

namespace {

constexpr ui16 VERSION_2_0 = 0x0200;
constexpr ui16 DATA_SKIP_BYTES = 1;
constexpr ui16 DATA_VALID_MAGIC = 0x444C; // "LD" in LE-order
constexpr ui16 CURRENT_DATA_VERSION = VERSION_2_0; // msb - major, lsb - minor

} // namespace

template <typename T>
T Read(TInputData& input) {
    Y_ENSURE_EX(input.Cur + sizeof(T) <= input.Data.size(), TParsingError{} << "unexpected end of input");
    T res = ReadUnaligned<T>(input.Data.data() + input.Cur);
    input.Cur += sizeof(T);
    return res;
}

ui32 ReadVar(TInputData& input) {
    ui32 result = 0;
    constexpr size_t MaxVarIntLength = 8u * sizeof(ui32) + 6u / 7u;
    size_t count = 0;
    const ui8* byte = reinterpret_cast<const ui8*>(input.Data.data()) + input.Cur;
    size_t end = Min(input.Data.size() - input.Cur, MaxVarIntLength);
    for (; count < end; ++count, ++byte) {
        result |= (static_cast<ui32>(*byte & 0x7F)) << (7 * count);
        if ((*byte & 0x80) == 0) {
            break;
        }
    }
    if (count == end) {
        if (end == input.Data.size() - input.Cur) {
            ythrow yexception() << "the data is too long to read ui32";
        } else {
            ythrow yexception() << "the data unexpectedly ended";
        }
    } else {
        ++count;
    }
    input.Cur += count;
    return result;
}

void Skip(TInputData& data, size_t size) {
    Y_ENSURE_EX(data.Cur + size <= data.Data.size(), TParsingError() << "the data unexpectedly ended");
    data.Cur += size;
}

TTsParser::TTsParser(
        TIntrusivePtr<NPrivate::TDataStorage> storage,
        NTsModel::EPointType metricType,
        NSolomon::NLabels::TLabels labels,
        TInputData input,
        TInstant commonTs,
        TDuration step,
        NMonitoring::ETimePrecision timePrecision)
    : Storage_{std::move(storage)}
    , MetricType_{metricType}
    , Labels_{std::move(labels)}
    , Input_{input}
    , CommonTs_{commonTs}
    , Step_{step}
    , TimePrecision_{timePrecision}
{
    ui8 typesByte = Read<ui8>(Input_);
    NMonitoring::EMetricType type;
    if (!TryDecodeMetricType(typesByte >> 2, &type)) {
        ythrow TParsingError{}
            << "unknown metric type: " << (typesByte >> 2);
    }
    if (NMonitoring::EMetricType::UNKNOWN == type) {
        ythrow TParsingError{}
            << "metric type is set to 'UNKNOWN'";
    }
    // TODO: if (type != MetricType_) ...

    Flags_ = Read<ELogFlagsComb>(Input_);

    NMonitoring::EValueType valueType;
    if (!TryDecodeValueType(typesByte & 0x03u, &valueType)) {
        ythrow TParsingError{}
            << "unknown metric value type: " << (typesByte & 0x03u);
    }

    switch (valueType) {
        case NMonitoring::EValueType::ONE_WITHOUT_TS: {
            UseCommonTs_ = true;
            NumPoints_ = 1;
            break;
        }
        case NMonitoring::EValueType::ONE_WITH_TS: {
            UseCommonTs_ = false;
            NumPoints_ = 1;
            break;
        }
        case NMonitoring::EValueType::MANY_WITH_TS: {
            UseCommonTs_ = false;
            NumPoints_ = ReadVar(Input_);
            break;
        }
        case NMonitoring::EValueType::NONE: {
            UseCommonTs_ = false;
            NumPoints_ = 0;
            break;
        }
        default:
            ythrow TParsingError{}
                << "invalid metric value type that was not reported by 'TryDecodeValueType': "
                << static_cast<ui16>(valueType);
    }
}

bool TTsParser::HasNext() const {
    return CurPoint_ < NumPoints_;
}

void TTsParser::NextPoint(NTsModel::TGaugePoint& point) {
    Y_VERIFY(MetricType_ == NTsModel::EPointType::DGauge);
    Y_VERIFY(HasNext());
    CurPoint_++;
    point.Time = ReadTs();
    ReadDoublePoint(point);
    point.Merge = ReadMerge();
    point.Count = ReadCount();
    point.Denom = ReadDenom();
    point.Step = Step_;
}

void TTsParser::NextPoint(NTsModel::TCounterPoint& point) {
    Y_VERIFY(MetricType_ == NTsModel::EPointType::Counter);
    Y_VERIFY(HasNext());
    CurPoint_++;
    point.Time = ReadTs();
    ReadUi64Point(point);
    point.Merge = ReadMerge();
    point.Count = ReadCount();
    Y_UNUSED(ReadDenom());
    point.Step = Step_;
}

void TTsParser::NextPoint(NTsModel::TRatePoint& point) {
    Y_VERIFY(MetricType_ == NTsModel::EPointType::Rate);
    Y_VERIFY(HasNext());
    CurPoint_++;
    point.Time = ReadTs();
    ReadUi64Point(point);
    point.Merge = ReadMerge();
    point.Count = ReadCount();
    Y_UNUSED(ReadDenom());
    point.Step = Step_;
}

void TTsParser::NextPoint(NTsModel::TIGaugePoint& point) {
    Y_VERIFY(MetricType_ == NTsModel::EPointType::IGauge);
    Y_VERIFY(HasNext());
    CurPoint_++;
    point.Time = ReadTs();
    ReadI64Point(point);
    point.Merge = ReadMerge();
    point.Count = ReadCount();
    Y_UNUSED(ReadDenom());
    point.Step = Step_;
}

void TTsParser::NextPoint(NTsModel::THistPoint& point) {
    Y_VERIFY(MetricType_ == NTsModel::EPointType::Hist || MetricType_ == NTsModel::EPointType::LogHist);
    Y_VERIFY(HasNext());
    CurPoint_++;
    point.Time = ReadTs();
    if (MetricType_ == NTsModel::EPointType::LogHist) {
        ReadLogHistPointAsHist(point);
    } else {
        ReadHistPoint(point);
    }
    point.Merge = ReadMerge();
    point.Count = ReadCount();
    point.Denom = ReadDenom();
    point.Step = Step_;
}

void TTsParser::NextPoint(NTsModel::THistRatePoint& point) {
    Y_VERIFY(MetricType_ == NTsModel::EPointType::HistRate);
    Y_VERIFY(HasNext());
    CurPoint_++;
    point.Time = ReadTs();
    ReadHistPoint(point);
    point.Merge = ReadMerge();
    point.Count = ReadCount();
    point.Denom = ReadDenom();
    point.Step = Step_;
}

void TTsParser::NextPoint(NTsModel::TDSummaryPoint& point) {
    Y_VERIFY(MetricType_ == NTsModel::EPointType::DSummary);
    Y_VERIFY(HasNext());
    CurPoint_++;
    point.Time = ReadTs();
    ReadDSummaryPoint(point);
    point.Merge = ReadMerge();
    point.Count = ReadCount();
    if (ReadDenom() != 0) {
        ythrow TParsingError{}
            << "DSummary point can't have a denom";
    }
    point.Step = Step_;
}

void TTsParser::NextPoint(NTsModel::TLogHistPoint& point) {
    Y_VERIFY(MetricType_ == NTsModel::EPointType::LogHist);
    Y_VERIFY(HasNext());
    CurPoint_++;
    point.Time = ReadTs();
    ReadLogHistPoint(point);
    point.Merge = ReadMerge();
    point.Count = ReadCount();
    if (ReadDenom() != 0) {
        ythrow TParsingError{}
            << "LogHist point can't have a denom";
    }
    point.Step = Step_;
}

TInstant TTsParser::ReadTs() {
    if (UseCommonTs_) {
        return CommonTs_;
    }

    ui64 tsMillis = 0;

    switch (TimePrecision_) {
        case NMonitoring::ETimePrecision::SECONDS: {
            tsMillis = Read<ui32>(Input_) * 1000ull;
            break;
        }
        case NMonitoring::ETimePrecision::MILLIS: {
            tsMillis = Read<ui64>(Input_);
            break;
        }
    }

    if (tsMillis == 0) {
        return CommonTs_;
    } else {
        return TInstant::MilliSeconds(tsMillis);
    }
}

bool TTsParser::ReadMerge() {
    return static_cast<ELogFlagsComb>(ELogFlags::Merge) & Flags_;
}

ui64 TTsParser::ReadCount() {
    ui64 count = 0;
    if (static_cast<ELogFlagsComb>(ELogFlags::Count) & Flags_) {
        count = Read<ui64>(Input_);
    }
    return count;
}

ui64 TTsParser::ReadDenom() {
    ui64 denom = 0;
    if (static_cast<ELogFlagsComb>(ELogFlags::Denom) & Flags_) {
        denom = Read<ui64>(Input_);
    }
    return denom;
}

void TTsParser::ReadDoublePoint(NTs::TDoublePoint& point) {
    point.Num = Read<double>(Input_);
}

void TTsParser::ReadI64Point(NTs::TDoublePoint& point) {
    point.Num = Read<i64>(Input_);
}

void TTsParser::ReadI64Point(NTs::TLongPoint& point) {
    point.Value = Read<i64>(Input_);
}

void TTsParser::ReadUi64Point(NTs::TDoublePoint& point) {
    point.Num = Read<ui64>(Input_);
}

void TTsParser::ReadUi64Point(NTs::TLongPoint& point) {
    point.Value = Read<ui64>(Input_);
}

void TTsParser::ReadHistPoint(NTs::THistogramPoint& point) {
    ui32 numBuckets = ReadVar(Input_);
    if (numBuckets > NMonitoring::HISTOGRAM_MAX_BUCKETS_COUNT) {
        ythrow TParsingError{}
            << "too many buckets in a histogram: " << numBuckets;
    }

    point.Buckets.resize(numBuckets);

    for (ui32 i = 0; i < numBuckets; i++) {
        point.Buckets[i].UpperBound = Read<double>(Input_);
    }

    for (ui32 i = 0; i < numBuckets; i++) {
        point.Buckets[i].Value = Read<ui64>(Input_);
    }
}

void TTsParser::ReadLogHistPointAsHist(NTs::THistogramPoint& point) {
    double base = Read<double>(Input_);
    size_t zeroCount = Read<ui64>(Input_);
    int startPower = ReadVar(Input_);
    auto size = ReadVar(Input_);

    NYasm::TUgramBuilder builder;
    builder.AddBucket(0.0, 0.0, zeroCount);

    double bound = std::pow(base, startPower);
    if ((ui32)size > NMonitoring::LOG_HIST_MAX_BUCKETS) {
        ythrow TParsingError{}
                << "too many buckets in a log histogram: " << size;
    }

    for (size_t i = 0; i < size; ++i) {
        builder.AddBucket(bound, Read<double>(Input_));
        bound *= base;
    }

    point.Buckets = builder.ReleaseBuckets();
}

void TTsParser::ReadLogHistPoint(NTs::TLogHistogramPoint& point) {
    point.Base = Read<double>(Input_);
    point.ZeroCount = Read<ui64>(Input_);
    point.StartPower = ReadVar(Input_);
    point.MaxBucketCount = NTs::TLogHistogramPoint::DefaultMaxBucketCount;
    auto size = ReadVar(Input_);
    if ((ui32)size > NMonitoring::LOG_HIST_MAX_BUCKETS) {
        ythrow TParsingError{}
            << "too many buckets in a log histogram: " << size;
    }

    point.Values.clear();
    point.Values.reserve(size);
    for (ui32 i = 0; i < size; ++i) {
        point.Values.emplace_back() = Read<double>(Input_);
    }
}

void TTsParser::ReadDSummaryPoint(NTs::TSummaryDoublePoint& point) {
    point.CountValue = Read<ui64>(Input_);
    point.Sum = Read<double>(Input_);
    point.Min = Read<double>(Input_);
    point.Max = Read<double>(Input_);
    point.Last = Read<double>(Input_);
}

TParser::TParser(TString meta, TString data) {
    try {
        auto metaIter = NSolomon::NSlog::NUnresolvedMeta::CreateUnresolvedMetaIterator(std::move(meta));
        Storage_ = MakeIntrusive<NPrivate::TDataStorage>(std::move(data), std::move(metaIter));
        Data_ = TInputData{Storage_->Data};

        ReadDataHeader();
    } catch (NSolomon::NSlog::TDecodeError& e) {
        ythrow TParsingError{} << e.what();
    }
}

bool TParser::HasNext() {
    return Storage_->MetaIter->HasNext();
}

TTsParser TParser::Next() {
    try {
        auto metaRecord = Storage_->MetaIter->Next();

        auto data = Data_.Data.data() + Data_.Cur;
        Skip(Data_, metaRecord.DataSize);

        return TTsParser{
            Storage_,
            metaRecord.Type,
            std::move(metaRecord.Labels),
            TInputData{data, metaRecord.DataSize},
            CommonTs_,
            Step_,
            TimePrecision_};
    } catch (NSolomon::NSlog::TDecodeError& e) {
        ythrow TParsingError{} << e.what();
    }
}

void TParser::ReadDataHeader() {
    ui16 magic = Read<ui16>(Data_);
    if (magic != DATA_VALID_MAGIC) {
        ythrow TParsingError{}
            << "invalid data magic: expected " << DATA_VALID_MAGIC << ", "
            << "got " << magic;
    }

    ui16 version = Read<ui16>(Data_);
    if (version > CURRENT_DATA_VERSION) {
        ythrow TParsingError{}
            << "unsupported version of data: " << version;
    }

    ui32 numId = Read<ui32>(Data_);
    ui32 metaNumId = Storage_->MetaIter->NumId();
    if (numId != metaNumId) {
        ythrow TParsingError{}
            << "num id in meta and data don't match: " << metaNumId << " vs " << numId;
    }

    CommonTs_ = TInstant::MilliSeconds(Read<ui64>(Data_));
    Step_ = TDuration::MilliSeconds(Read<ui32>(Data_));

    ui32 numMetrics = Read<ui32>(Data_);
    ui32 metaNumMetrics = Storage_->MetaIter->TotalMetricCount();
    if (numMetrics != metaNumMetrics) {
        ythrow TParsingError{}
            << "number of metrics in meta and data don't match: " << metaNumMetrics << " vs " << numMetrics;
    }

    ui32 numPoints = Read<ui32>(Data_);
    ui32 metaNumPoints = Storage_->MetaIter->TotalPointCount();
    if (numPoints != metaNumPoints) {
        ythrow TParsingError{}
            << "number of points in meta and data don't match: " << metaNumPoints << " vs " << numPoints;
    }

    ui8 timePrecision = Read<ui8>(Data_);
    if (!NMonitoring::TryDecodeTimePrecision(timePrecision, &TimePrecision_)) {
        ythrow TParsingError{}
            << "unknown data time precision: " << timePrecision;
    }

    ui8 compressionAlg = Read<ui8>(Data_);
    NMonitoring::ECompression compressionAlgParsed;
    if (!NMonitoring::TryDecodeCompression(compressionAlg, &compressionAlgParsed)) {
        ythrow TParsingError{}
            << "unknown metadata compression algorithm: " << compressionAlg;
    }
    if (compressionAlgParsed == NMonitoring::ECompression::UNKNOWN) {
        ythrow TParsingError{}
            << "metadata compression algorithm is set to 'UNKNOWN'";
    }

    ui8 encodingScheme = Read<ui8>(Data_);
    if (version >= VERSION_2_0 && encodingScheme != 0) {
        ythrow TParsingError{}
            << "encoding scheme " << encodingScheme << " is not supported";
    }

    Skip(Data_, DATA_SKIP_BYTES);
    TMemoryInput input{Data_.Data.substr(Data_.Cur)};
    auto compressedIn = NMonitoring::CompressedInput(&input, compressionAlgParsed);
    if (compressedIn) {
        Storage_->Data = compressedIn->ReadAll();
        Data_ = TInputData{Storage_->Data};
    }
}

} // namespace NSolomon::NMemStore::NSLog
