#pragma once

#include "error.h"

#include <solomon/libs/cpp/slog/unresolved_meta/iterator.h>
#include <solomon/libs/cpp/labels/labels.h>
#include <solomon/libs/cpp/log_writer/log_writer.h>
#include <solomon/libs/cpp/ts_model/iterator.h>
#include <solomon/libs/cpp/ts_model/points.h>

#include <library/cpp/monlib/encode/buffered/string_pool.h>
#include <library/cpp/monlib/encode/spack/spack_v1.h>

#include <util/stream/mem.h>

namespace NSolomon::NMemStore::NSLog {

namespace NPrivate {

// TString should not be treated as COW because it's being replaced with std::string.
// So we keep decompressed data in this struct to perform reference counting.
struct TDataStorage: public TThrRefBase {
    TString Data;
    NSlog::NUnresolvedMeta::IUnresolvedMetaIteratorPtr MetaIter;

    TDataStorage(TString data, NSlog::NUnresolvedMeta::IUnresolvedMetaIteratorPtr metaIter) noexcept
        : Data{std::move(data)}
        , MetaIter{std::move(metaIter)}
    {
    }

    size_t SizeBytes() const {
        size_t size = sizeof(TDataStorage) + Data.capacity();
        if (MetaIter) {
            size += MetaIter->SizeBytes();
        }
        return size;
    }
};

} // namespace NPrivate

class TParser;

struct TInputData {
    explicit TInputData(TStringBuf strBuf)
        : Data(strBuf)
    {
    }

    TInputData(const char* str, size_t len)
        : Data(str, len)
    {
    }

    TStringBuf Data;
    size_t Cur = 0u;
};

/**
 * Reads a single time series.
 */
class TTsParser: private TMoveOnly {
    friend class TParser;

private:
    TTsParser(
        TIntrusivePtr<NPrivate::TDataStorage> storage,
        NTsModel::EPointType metricType,
        NSolomon::NLabels::TLabels labels,
        TInputData input,
        TInstant commonTs,
        TDuration step,
        NMonitoring::ETimePrecision timePrecision);

public:
    /**
     * Iterator which allows read one (generic) type of point
     */
    template <typename TPoint>
    class TIterator: public NTsModel::IIterator<TPoint>
    {
    public:
        explicit TIterator(NSLog::TTsParser& data)
            : Data_{data}
        {
        }

        bool NextPoint(TPoint* point) override {
            if (!Data_.HasNext()) {
                return false;
            }
            Data_.NextPoint(*point);
            return true;
        }

    private:
        NSLog::TTsParser& Data_;
    };

    template <typename TPoint>
    TIterator<TPoint> Iterator() {
        return TIterator<TPoint>(*this);
    }

    /**
     * Get type for this time series. This type determines which signature of `Next`
     * you should use.
     */
    NTsModel::EPointType GetMetricType() const {
        return MetricType_;
    }

    ui32 NumPoints() const {
        return NumPoints_;
    }

    size_t SizeBytes() const {
        return sizeof(TTsParser) + Labels_.Size() * sizeof(NSolomon::NLabels::TLabels::TPair);
    }

    /**
     * Get labels for this time series. The returned array is guaranteed to be sorted.
     */
    NSolomon::NLabels::TLabelsView GetLabels() const {
        return Labels_;
    }

    /**
     * Check if there's a next point available.
     */
    bool HasNext() const;

    /**
     * Read next point.
     *
     * Point type must match the type returned by `GetMetricType`, otherwise the behaviour
     * is undefined. This method panics if there are no points left in the iterator
     * (i.e. `HasNext` returns false).
     *
     * Throws `TParsingError` in case of invalid input.
     */
    void NextPoint(NTsModel::TGaugePoint& out);
    void NextPoint(NTsModel::TCounterPoint& out);
    void NextPoint(NTsModel::TRatePoint& out);
    void NextPoint(NTsModel::TIGaugePoint& out);
    void NextPoint(NTsModel::THistPoint& out);
    void NextPoint(NTsModel::THistRatePoint& out);
    void NextPoint(NTsModel::TDSummaryPoint& out);
    void NextPoint(NTsModel::TLogHistPoint& out);

private:
    TInstant ReadTs();
    bool ReadMerge();
    ui64 ReadCount();
    ui64 ReadDenom();
    void ReadDoublePoint(NTs::TDoublePoint& point);
    void ReadI64Point(NTs::TDoublePoint& point);
    void ReadI64Point(NTs::TLongPoint& point);
    void ReadUi64Point(NTs::TDoublePoint& point);
    void ReadUi64Point(NTs::TLongPoint& point);
    void ReadHistPoint(NTs::THistogramPoint& point);
    void ReadLogHistPoint(NTs::TLogHistogramPoint& point);
    void ReadLogHistPointAsHist(NTs::THistogramPoint& point);
    void ReadDSummaryPoint(NTs::TSummaryDoublePoint& point);

private:
    TIntrusivePtr<NPrivate::TDataStorage> Storage_;
    NTsModel::EPointType MetricType_;
    NSolomon::NLabels::TLabels Labels_;
    ELogFlagsComb Flags_;
    TInputData Input_;
    bool UseCommonTs_;
    TInstant CommonTs_;
    TDuration Step_;
    NMonitoring::ETimePrecision TimePrecision_;
    ui32 NumPoints_;

    size_t CurPoint_{0};
};

/**
 * Reds multiple time series from SLog format. Tries to avoid additional
 * allocations when possible. If the input is compressed, this parser will
 * firs fully decompress it before parsing.
 */
class TParser: private TMoveOnly {
public:
    /**
     * Create new parser from meta and data.
     *
     * Throws `TParsingError` in case of invalid input.
     */
    TParser(TString meta, TString data);

    /**
     * Check if there is next time series available.
     */
    bool HasNext();

    /**
     * Read next time series (read metadata, does not parse points).
     *
     * This method panics if there are no points left in the iterator
     * (i.e. `HasNext` returns false).
     *
     * Throws `TParsingError` in case of invalid input.
     *
     * The returned time series parser does not reference this parser, it stays valid
     * even after `TParser` has died.
     */
    TTsParser Next();

    size_t GetStorageSizeBytes() const {
        return Storage_ ? Storage_->SizeBytes() : 0;
    }

private:
    void ReadDataHeader();

private:
    TIntrusivePtr<NPrivate::TDataStorage> Storage_;
    TInputData Data_{nullptr, 0};

    TInstant CommonTs_;
    TDuration Step_;
    NMonitoring::ETimePrecision TimePrecision_;
};

} // namespace NSolomon::NMemStore::NSLog
