#include <library/cpp/testing/gtest/gtest.h>

#include <solomon/services/memstore/lib/slog/parser.h>

#include <solomon/libs/cpp/slog/testlib/testlib.h>
#include <solomon/libs/cpp/ts_model/testlib/testlib.h>

TEST(SlogFormatMatch, Basics) {
    auto [meta, data] = NSolomon::NSlog::MakeSlog(42, {{"service", "memstore"}})
        .IGauge({{"sensor", "memory"}})
            .Add("2020-01-01T00:00:00Z", 10000)
            .Add("2020-01-01T00:00:15Z", 12000)
            .Add("2020-01-01T00:00:30Z", 13000)
            .Add("2020-01-01T00:00:45Z", 12000)
            .Done()
        .Gauge({{"sensor", "rps"}}, true)
            .Add("2020-01-01T00:00:00Z", 50, 15, 1)
            .Add("2020-01-01T00:00:15Z", 75, 15, 1)
            .Add("2020-01-01T00:00:30Z", 100, 15, 1)
            .Add("2020-01-01T00:00:45Z", 135, 15, 1)
            .Done()
        .Done();

    NSolomon::NMemStore::NSLog::TParser parser{std::move(meta), std::move(data)};

    ASSERT_TRUE(parser.HasNext());
    {
        auto ts = parser.Next();

        EXPECT_EQ(ts.GetLabels(), NSolomon::NLabels::TLabels::OwnedStorage(
            {{"sensor", "memory"}, {"service", "memstore"}}));
        ASSERT_EQ(ts.GetMetricType(), NSolomon::NTsModel::EPointType::IGauge);

        NSolomon::NTsModel::TIGaugePoint point;

        ASSERT_TRUE(ts.HasNext());
        {
            ts.NextPoint(point);
            EXPECT_EQ(point, NSolomon::NTsModel::IGauge("2020-01-01T00:00:00Z", 10000));
        }

        ASSERT_TRUE(ts.HasNext());
        {
            ts.NextPoint(point);
            EXPECT_EQ(point, NSolomon::NTsModel::IGauge("2020-01-01T00:00:15Z", 12000));
        }

        ASSERT_TRUE(ts.HasNext());
        {
            ts.NextPoint(point);
            EXPECT_EQ(point, NSolomon::NTsModel::IGauge("2020-01-01T00:00:30Z", 13000));
        }

        ASSERT_TRUE(ts.HasNext());
        {
            ts.NextPoint(point);
            EXPECT_EQ(point, NSolomon::NTsModel::IGauge("2020-01-01T00:00:45Z", 12000));
        }

        ASSERT_FALSE(ts.HasNext());
    }

    ASSERT_TRUE(parser.HasNext());
    {
        auto ts = parser.Next();

        EXPECT_EQ(ts.GetLabels(), NSolomon::NLabels::TLabels::OwnedStorage(
            {{"sensor", "rps"}, {"service", "memstore"}}));
        ASSERT_EQ(ts.GetMetricType(), NSolomon::NTsModel::EPointType::DGauge);

        NSolomon::NTsModel::TGaugePoint point;

        ASSERT_TRUE(ts.HasNext());
        {
            ts.NextPoint(point);
            EXPECT_EQ(point, NSolomon::NTsModel::Gauge("2020-01-01T00:00:00Z", 50, 15, true));
        }

        ASSERT_TRUE(ts.HasNext());
        {
            ts.NextPoint(point);
            EXPECT_EQ(point, NSolomon::NTsModel::Gauge("2020-01-01T00:00:15Z", 75, 15, true));
        }

        ASSERT_TRUE(ts.HasNext());
        {
            ts.NextPoint(point);
            EXPECT_EQ(point, NSolomon::NTsModel::Gauge("2020-01-01T00:00:30Z", 100, 15, true));
        }

        ASSERT_TRUE(ts.HasNext());
        {
            ts.NextPoint(point);
            EXPECT_EQ(point, NSolomon::NTsModel::Gauge("2020-01-01T00:00:45Z", 135, 15, true));
        }

        ASSERT_FALSE(ts.HasNext());
    }

    ASSERT_FALSE(parser.HasNext());
}

TEST(SlogFormatMatch, CommpresedLog) {
    auto[meta, data] = NSolomon::NSlog::MakeSlog(42,
                                          {{"service", "memstore"}},
                                          TInstant::ParseIso8601("2000-01-01T00:00:00Z"),
                                          TDuration::Seconds(15),
                                          NMonitoring::ECompression::ZLIB)
        .IGauge({{"sensor", "memory"}})
            .Add("2020-01-01T00:00:00Z", 10000)
            .Add("2020-01-01T00:00:15Z", 12000)
            .Add("2020-01-01T00:00:30Z", 13000)
            .Add("2020-01-01T00:00:45Z", 12000)
        .Done()
        .Gauge({{"sensor", "rps"}}, true)
            .Add("2020-01-01T00:00:00Z", 50, 15, 1)
            .Add("2020-01-01T00:00:15Z", 75, 15, 1)
            .Add("2020-01-01T00:00:30Z", 100, 15, 1)
            .Add("2020-01-01T00:00:45Z", 135, 15, 1)
        .Done()
        .Done();

    NSolomon::NMemStore::NSLog::TParser parser{std::move(meta), std::move(data)};
    ASSERT_TRUE(parser.HasNext());
    {
        auto ts = parser.Next();

        EXPECT_EQ(ts.GetLabels(), NSolomon::NLabels::TLabels::OwnedStorage(
            {{"sensor", "memory"}, {"service", "memstore"}}));
        ASSERT_EQ(ts.GetMetricType(), NSolomon::NTsModel::EPointType::IGauge);

        NSolomon::NTsModel::TIGaugePoint point;

        ASSERT_TRUE(ts.HasNext());
        {
            ts.NextPoint(point);
            EXPECT_EQ(point, NSolomon::NTsModel::IGauge("2020-01-01T00:00:00Z", 10000));
        }

        ASSERT_TRUE(ts.HasNext());
        {
            ts.NextPoint(point);
            EXPECT_EQ(point, NSolomon::NTsModel::IGauge("2020-01-01T00:00:15Z", 12000));
        }

        ASSERT_TRUE(ts.HasNext());
        {
            ts.NextPoint(point);
            EXPECT_EQ(point, NSolomon::NTsModel::IGauge("2020-01-01T00:00:30Z", 13000));
        }

        ASSERT_TRUE(ts.HasNext());
        {
            ts.NextPoint(point);
            EXPECT_EQ(point, NSolomon::NTsModel::IGauge("2020-01-01T00:00:45Z", 12000));
        }

        ASSERT_FALSE(ts.HasNext());
    }

    ASSERT_TRUE(parser.HasNext());
    {
        auto ts = parser.Next();

        EXPECT_EQ(ts.GetLabels(), NSolomon::NLabels::TLabels::OwnedStorage(
            {{"sensor", "rps"}, {"service", "memstore"}}));
        ASSERT_EQ(ts.GetMetricType(), NSolomon::NTsModel::EPointType::DGauge);

        NSolomon::NTsModel::TGaugePoint point;

        ASSERT_TRUE(ts.HasNext());
        {
            ts.NextPoint(point);
            EXPECT_EQ(point, NSolomon::NTsModel::Gauge("2020-01-01T00:00:00Z", 50, 15, true));
        }

        ASSERT_TRUE(ts.HasNext());
        {
            ts.NextPoint(point);
            EXPECT_EQ(point, NSolomon::NTsModel::Gauge("2020-01-01T00:00:15Z", 75, 15, true));
        }

        ASSERT_TRUE(ts.HasNext());
        {
            ts.NextPoint(point);
            EXPECT_EQ(point, NSolomon::NTsModel::Gauge("2020-01-01T00:00:30Z", 100, 15, true));
        }

        ASSERT_TRUE(ts.HasNext());
        {
            ts.NextPoint(point);
            EXPECT_EQ(point, NSolomon::NTsModel::Gauge("2020-01-01T00:00:45Z", 135, 15, true));
        }

        ASSERT_FALSE(ts.HasNext());
    }

    ASSERT_FALSE(parser.HasNext());
}
