#include "cluster.h"
#include "tablet.h"
#include "watcher.h"

#include <solomon/libs/cpp/actors/poison/poisoner.h>
#include <solomon/libs/cpp/logging/logging.h>
#include <solomon/libs/cpp/selfmon/selfmon.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/actorid.h>
#include <library/cpp/containers/absl_flat_hash/flat_hash_map.h>

using namespace NActors;
using namespace NMonitoring;

namespace NSolomon::NMemStore {
namespace {

using TTabletsMap = absl::flat_hash_map<ui64, TActorId>;

const TString PagePath = "/storage";

class TStorageSelfMon: public NActors::TActor<TStorageSelfMon> {
public:
    TStorageSelfMon(TNodeId nodeId, const TTabletsMap& tablets, TActorId watcher)
        : NActors::TActor<TStorageSelfMon>(&TThis::StateFunc)
        , Tablets_{tablets}
        , Watcher_{watcher}
    {
        Page_.set_title(TStringBuilder{} << "Storage of " << nodeId << " node");
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(NSelfMon::TEvPageDataReq, OnSelfMon)
            hFunc(TStorageTabletEvents::TStatsResp, OnTabletResp)
            hFunc(NSelfMon::TEvPageDataResp, OnWatcherResp)
        }
    }

    void OnSelfMon(const NSelfMon::TEvPageDataReq::TPtr& ev) {
        ReplyTo_ = ev->Sender;

        auto activeTab = ev->Get()->Param("tab");
        auto* g = Page_.mutable_grid();

        {
            auto* r = g->add_rows();
            auto* component = r->add_columns()->mutable_component();
            RenderTabs(activeTab, component->mutable_tabs());
        }
        {
            auto* r = g->add_rows();
            if (activeTab.empty() || activeTab == "tablets") {
                TryFromString(ev->Get()->Param("limit"), TabletsLimit_);
                QueryTablets();
            } else if (activeTab == "nodes") {
                QueryWatcher(ev->Release());
            } else {
                r->add_columns()->mutable_component()->mutable_value()->set_string("Selected invalid tab");
                ReplyAndDie();
            }
        }
    }

    void RenderTabs(TStringBuf activeTab, yandex::monitoring::selfmon::Tabs* tabs) {
        if (auto* tab = tabs->add_tabs()) {
            tab->set_active(activeTab.empty() || activeTab == "tablets");
            if (auto* ref = tab->mutable_reference()) {
                ref->set_title("Tablets");
                ref->set_page(PagePath);
                ref->set_args("tab=tablets");
            }
        }
        if (auto* tab = tabs->add_tabs()) {
            tab->set_active(activeTab == "nodes");
            if (auto* ref = tab->mutable_reference()) {
                ref->set_title("Nodes");
                ref->set_page(PagePath);
                ref->set_args("tab=nodes");
            }
        }
    }

    void QueryTablets() {
        for (const auto& [tabletId, actorId]: Tablets_) {
            if (actorId) {
                Send(actorId, new TStorageTabletEvents::TStatsReq);
                if (++TabletsCountdown_ == TabletsLimit_) {
                    break;
                }
            }
        }

        if (TabletsCountdown_ == 0) {
            RenderTablets();
            ReplyAndDie();
        }
    }

    void OnTabletResp(const TStorageTabletEvents::TStatsResp::TPtr& ev) {
        TabletsStats_.emplace_back(ev->Release().Release());
        if (--TabletsCountdown_ == 0) {
            RenderTablets();
            ReplyAndDie();
        }
    }

    void RenderTablets() {
        auto* g = Page_.mutable_grid();

        {
            auto* r = g->mutable_rows(1);
            auto* table = r->add_columns()->mutable_component()->mutable_table();
            table->set_numbered(true);

            auto* tabletIdColumn = table->add_columns();
            tabletIdColumn->set_title("TabletId");
            auto* tabletIdValues = tabletIdColumn->mutable_reference();

            auto* addressColumn = table->add_columns();
            addressColumn->set_title("Address");
            auto* addressValues = addressColumn->mutable_string();

            auto* logFilesSizeColumn = table->add_columns();
            logFilesSizeColumn->set_title("Log size");
            auto* logFilesSizeValues = logFilesSizeColumn->mutable_data_size();

            auto* snapshotFilesSizeColumn = table->add_columns();
            snapshotFilesSizeColumn->set_title("Snapshot size");
            auto* snapshotFilesSizeValues = snapshotFilesSizeColumn->mutable_data_size();

            auto* tempFilesSizeColumn = table->add_columns();
            tempFilesSizeColumn->set_title("Temp size");
            auto* tempFilesSizeValues = tempFilesSizeColumn->mutable_data_size();

            for (auto& s: TabletsStats_) {
                auto tabletIdStr = ToString(s->TabletId);
                auto* ref = tabletIdValues->add_values();
                ref->set_title(tabletIdStr);
                ref->set_page(PagePath);
                ref->set_args("tabletId=" + tabletIdStr);

                addressValues->add_values(std::move(s->Address));
                logFilesSizeValues->add_values(static_cast<double>(s->LogFilesSize));
                snapshotFilesSizeValues->add_values(static_cast<double>(s->SnapshotFilesSize));
                tempFilesSizeValues->add_values(static_cast<double>(s->TempFilesSize));
            }
        }
        {
            auto* r2 = g->add_rows();
            auto* ref = r2->add_columns()->mutable_component()->mutable_value()->mutable_reference();
            if (TabletsLimit_ != -1) {
                ref->set_title("Show all " + ToString(Tablets_.size()));
                ref->set_page(PagePath);
                ref->set_args("tab=tablets&limit=-1");
            } else {
                ref->set_title("Show first 20");
                ref->set_page(PagePath);
                ref->set_args("tab=tablets&limit=20");
            }
        }
    }

    void QueryWatcher(const TAutoPtr<NSelfMon::TEvPageDataReq>& req) {
        Send(Watcher_, req.Release());
    }

    void OnWatcherResp(const NSelfMon::TEvPageDataResp::TPtr& ev) {
        auto* col = Page_.mutable_grid()->mutable_rows(1)->add_columns();
        auto& page = ev->Get()->Page;
        if (page.has_component()) {
            col->mutable_component()->Swap(page.mutable_component());
        } else if (page.has_grid()) {
            col->mutable_grid()->Swap(page.mutable_grid());
        }
        ReplyAndDie();
    }

    void ReplyAndDie() {
        Send(ReplyTo_, new NSelfMon::TEvPageDataResp{std::move(Page_)});
        PassAway();
    }

private:
    const TTabletsMap& Tablets_;
    TActorId Watcher_;
    TActorId ReplyTo_;
    yandex::monitoring::selfmon::Page Page_;
    std::vector<std::unique_ptr<TStorageTabletEvents::TStatsResp>> TabletsStats_;
    i32 TabletsCountdown_{0};
    i32 TabletsLimit_{20};
};

struct TLocalEvents: private TPrivateEvents {
    enum {
        CalcFilesSize = SpaceBegin,
        End
    };
    static_assert(End < SpaceEnd, "too many event types");

    /**
     * Calculate total size of all files stored by this memstore node in the tablets.
     * Write result as a metric. Private event, occurs periodically.
     */
    struct TCalcFilesSize: public NActors::TEventLocal<TCalcFilesSize, CalcFilesSize> {
    };
};

constexpr TDuration CalcFilesSizePeriod_ = TDuration::Seconds(15);

class TStorageCluster: public TActorBootstrapped<TStorageCluster> {
public:
    explicit TStorageCluster(TStorageClusterOptions opts)
        : NodeId_{opts.NodeId}
        , Rpc_{std::move(opts.Rpc)}
        , Watcher_{StorageClusterWatcher(Rpc_, opts.VolumePath, opts.WatchDelay)}
    {
        TempFilesSizeMetric_ = opts.MetricRegistry.IntGauge({{"sensor", "storage.filesSize.temp"}});
        LogFilesSizeMetric_ = opts.MetricRegistry.IntGauge({{"sensor", "storage.filesSize.log"}});
        SnapshotFilesSizeMetric_ = opts.MetricRegistry.IntGauge({{"sensor", "storage.filesSize.snapshot"}});
    }

    void Bootstrap() {
        Watcher_ = Register(std::get<std::unique_ptr<IActor>>(Watcher_).release());
        Send(std::get<TActorId>(Watcher_), new TStorageWatcherEvents::TSubscribe);
        Become(&TThis::Normal);
        Schedule(CalcFilesSizePeriod_, new TLocalEvents::TCalcFilesSize);
    }

    STFUNC(Normal) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TStorageClusterEvents::TGetTablets, OnGetTablets);
            hFunc(TStorageClusterEvents::TResolveTablets, OnResolveTablets);
            sFunc(TLocalEvents::TCalcFilesSize, OnCalcFilesSize);
            hFunc(TStorageTabletEvents::TFilesSizeResp, OnFilesSizeResponse);
            hFunc(TStorageWatcherEvents::TStateChanged, OnStateChanged);
            HFunc(NSelfMon::TEvPageDataReq, OnSelfMon)
            hFunc(TEvents::TEvPoison, OnPoison);
        }
    }

    void OnGetTablets(TStorageClusterEvents::TGetTablets::TPtr& ev) {
        std::vector<TTabletActor> tabletActors;
        for (const auto& [tabletId, actorId]: Tablets_) {
            tabletActors.emplace_back(TTabletActor{tabletId, actorId});
        }
        Send(ev->Sender, new TStorageClusterEvents::TTablets{std::move(tabletActors)});
    }

    void OnResolveTablets(TStorageClusterEvents::TResolveTablets::TPtr& ev) {
        std::vector<TTabletActor> tabletActors;
        for (TTabletId id: ev->Get()->TabletIds) {
            if (auto it = Tablets_.find(id); it != Tablets_.end()) {
                tabletActors.emplace_back(TTabletActor{id, it->second});
            }
        }
        Send(ev->Sender, new TStorageClusterEvents::TTablets{std::move(tabletActors)});
    }

    void OnCalcFilesSize() {
        AllTabletsFilesSize_.Clear();
        FilesSizeRespCountdown_ = static_cast<decltype(FilesSizeRespCountdown_)>(Tablets_.size());
        for (const auto& [_, actorId]: Tablets_) {
            Send(actorId, new TStorageTabletEvents::TFilesSizeReq);
        }
        if (FilesSizeRespCountdown_ == 0) {
            // by some reason we are starting very slowly
            Schedule(CalcFilesSizePeriod_, new TLocalEvents::TCalcFilesSize);
        }
    }

    void OnFilesSizeResponse(TStorageTabletEvents::TFilesSizeResp::TPtr& ev) {
        const auto& filesSize = ev->Get()->TabletFilesSize;
        AllTabletsFilesSize_.LogFilesSize += filesSize.LogFilesSize;
        AllTabletsFilesSize_.SnapshotFilesSize += filesSize.SnapshotFilesSize;
        AllTabletsFilesSize_.TempFilesSize += filesSize.TempFilesSize;
        if (--FilesSizeRespCountdown_ == 0) {
            ReportFilesSizeMetrics();
            Schedule(CalcFilesSizePeriod_, new TLocalEvents::TCalcFilesSize);
        }
    }

    void ReportFilesSizeMetrics() {
        TempFilesSizeMetric_->Set(AllTabletsFilesSize_.TempFilesSize);
        LogFilesSizeMetric_->Set(AllTabletsFilesSize_.LogFilesSize);
        SnapshotFilesSizeMetric_->Set(AllTabletsFilesSize_.SnapshotFilesSize);
    }

    void OnStateChanged(TStorageWatcherEvents::TStateChanged::TPtr& ev) {
        MON_TRACE(StorageCluster, "receive status change {lost: "
                << ev->Get()->Lost.size() << ", moved: " << ev->Get()->Moved.size() << '}');

        // lost tablets must be processed first
        for (TTabletId id: ev->Get()->Lost) {
            auto& actorId = Tablets_[id];
            if (!actorId) {
                actorId = Register(StorageTablet(NodeId_, id, {}, Rpc_).release(), TMailboxType::HTSwap);
            } else {
                Send(actorId, new TStorageTabletEvents::TLost);
            }
        }

        for (auto& loc: ev->Get()->Moved) {
            auto& actorId = Tablets_[loc.Id];
            if (!actorId) {
                actorId = Register(StorageTablet(NodeId_, loc.Id, std::move(loc.Location), Rpc_).release());
            } else {
                Send(actorId, new TStorageTabletEvents::TMoved{std::move(loc.Location)});
            }
        }
    }

    void OnSelfMon(const NSelfMon::TEvPageDataReq::TPtr& ev, const NActors::TActorContext& ctx) {
        TTabletId tabletId;
        if (TryFromString(ev->Get()->Param("tabletId"), tabletId)) {
            if (auto it = Tablets_.find(tabletId); it != Tablets_.end()) {
                ctx.Send(ev->Forward(it->second));
                return;
            }
        }

        auto page = RegisterWithSameMailbox(
            new TStorageSelfMon{NodeId_, Tablets_, std::get<TActorId>(Watcher_)});
        ctx.Send(ev->Forward(page));
    }

    void OnPoison(const TEvents::TEvPoison::TPtr& ev) {
        std::set<TActorId> toPoison{std::get<TActorId>(Watcher_)};
        for (auto& [_, actorId]: Tablets_) {
            toPoison.insert(actorId);
        }
        PoisonAll(ev, std::move(toPoison));
        PassAway();
    }

private:
    const TNodeId NodeId_;
    std::shared_ptr<NKikimr::IKikimrClusterRpc> Rpc_;
    std::variant<TActorId, std::unique_ptr<IActor>> Watcher_; // id of registered actor or actor itself
    TTabletsMap Tablets_;
    TTabletFilesSize AllTabletsFilesSize_;
    i32 FilesSizeRespCountdown_{0};
    NMonitoring::IIntGauge* TempFilesSizeMetric_{nullptr};
    NMonitoring::IIntGauge* LogFilesSizeMetric_{nullptr};
    NMonitoring::IIntGauge* SnapshotFilesSizeMetric_{nullptr};
};

} // namespace

std::unique_ptr<IActor> StorageCluster(TStorageClusterOptions opts) {
    return std::make_unique<TStorageCluster>(std::move(opts));
}

} // namespace NSolomon::NMemStore
